use windows::Win32::{
    Foundation::*,
    System::{
        Com::{CoInitializeEx, COINIT_APARTMENTTHREADED},
        LibraryLoader::GetModuleHandleW,
    },
    UI::WindowsAndMessaging::*,
};

use headless_webview::{
    window::{WindowAttributes, WindowId},
    HeadlessWindow, Result,
};

use super::webview2::WindowsWebview;

pub struct Win32Window {
    hwnd: HWND,
}

impl Win32Window {
    pub(crate) fn hwnd(&self) -> HWND {
        self.hwnd
    }
}

const WINDOW_STYLE: u32 = WS_OVERLAPPEDWINDOW;
const WINDOW_EX_STYLE: u32 = 0;

impl HeadlessWindow for Win32Window {
    type NativeWindow = ();

    type Webview = WindowsWebview;

    fn id(&self) -> WindowId {
        todo!()
    }

    fn new(_native_window: Self::NativeWindow, attributes: WindowAttributes) -> Result<Self>
    where
        Self: Sized,
    {
        let offscreen = false;

        let handle = unsafe {
            if offscreen {
                CoInitializeEx(std::ptr::null(), COINIT_APARTMENTTHREADED).unwrap();
            }

            let instance = GetModuleHandleW(PWSTR::default());
            debug_assert!(instance.0 != 0);

            let window_class = "webview_window";
            let wc = WNDCLASSA {
                hCursor: HCURSOR::default(),
                hInstance: instance,
                lpszClassName: PSTR(b"webview_window\0".as_ptr() as _),
                style: CS_HREDRAW | CS_VREDRAW,
                lpfnWndProc: Some(wndproc),
                ..Default::default()
            };

            let atom = RegisterClassA(&wc);
            debug_assert!(atom != 0);

            let inner_size = attributes.get_inner_size();

            // WIN10 api?
            let dpi = 96.;
            let scale_factor = dpi / 96.; // 150. / 96.; // FIXME: GetDpiForWindow - https://docs.microsoft.com/en-gb/windows/win32/hidpi/high-dpi-desktop-application-development-on-windows

            let mut rect = RECT {
                left: 0,
                top: 0,
                right: scale(inner_size.0, scale_factor),
                bottom: scale(inner_size.1, scale_factor),
            };

            assert!(AdjustWindowRectEx(&mut rect, WINDOW_STYLE, true, WINDOW_EX_STYLE).as_bool());

            let (width, height) = ((rect.right - rect.left), (rect.bottom - rect.top));

            println!("CREATE WINDOW: width={} height={}", width, height);

            let handle = CreateWindowExA(
                WINDOW_EX_STYLE,
                window_class,
                "bevy_webview",
                WINDOW_STYLE,
                0,
                0,
                width,
                height,
                None,
                None,
                instance,
                std::ptr::null_mut(),
            );

            if offscreen {
                ShowWindow(handle, SW_HIDE);
            } else {
                ShowWindow(handle, SW_SHOW);
            }

            handle
        };

        Ok(Self { hwnd: handle })
    }

    fn inner_size(&self) -> (u32, u32) {
        let mut rect = RECT::default();

        //DwmGetWindowAttribute(hWnd, DWMWA_EXTENDED_FRAME_BOUNDS, &wrect, sizeof(wrect));

        if !unsafe { GetClientRect(self.hwnd, &mut rect) }.as_bool() {
            panic!("Unexpected GetClientRect failure") // FIXME return error?
        }

        (
            (rect.right - rect.left) as u32,
            (rect.bottom - rect.top) as u32,
        )
    }

    fn _resize(&self, new_size: (u32, u32)) -> Result<(u32, u32)> {
        let scale_factor = 96. / 96.; // FIXME: GetDpiForWindow - https://docs.microsoft.com/en-gb/windows/win32/hidpi/high-dpi-desktop-application-development-on-windows

        let mut rect = RECT {
            left: 0,
            top: 0,
            right: scale(new_size.0 as usize, scale_factor),
            bottom: scale(new_size.1 as usize, scale_factor),
        };

        assert!(
            unsafe { AdjustWindowRectEx(&mut rect, WINDOW_STYLE, true, WINDOW_EX_STYLE) }.as_bool()
        );

        let (width, height) = ((rect.right - rect.left), (rect.bottom - rect.top));
        println!("W={} H={}", width, height);

        unsafe {
            SetWindowPos(
                self.hwnd,
                HWND::default(),
                0,
                0,
                width,
                height,
                SWP_NOMOVE | SWP_NOZORDER,
            )
        }
        .unwrap();

        // https://stackoverflow.com/a/27945625
        //todo!()
        Ok((width as u32, height as u32))
    }
}

extern "system" fn wndproc(window: HWND, message: u32, wparam: WPARAM, lparam: LPARAM) -> LRESULT {
    unsafe {
        match message as u32 {
            WM_PAINT => DefWindowProcA(window, message, wparam, lparam),
            WM_DESTROY => {
                PostQuitMessage(0);
                LRESULT(0)
            }
            WM_LBUTTONDOWN => {
                println!("LBUTTON DOWN!");
                LRESULT(0)
            }
            _ => DefWindowProcA(window, message, wparam, lparam),
        }
    }
}

fn scale(source: usize, scale_factor: f32) -> i32 {
    (source as f32 * scale_factor).round() as i32
}
