use std::rc::Rc;

use headless_webview::HeadlessWindow;
use windows::{
    core::Interface,
    Foundation::Numerics::Vector2,
    Win32::{Foundation::RECT, System::WinRT::Composition::ICompositorDesktopInterop},
    UI::Composition::{Compositor, ContainerVisual, Desktop::DesktopWindowTarget},
};

use webview2_com::Microsoft::Web::WebView2::Win32::*;

use super::win32_window::Win32Window;

pub fn setup_webview_visual(
    window: &Rc<Win32Window>,
    composition_controller: &ICoreWebView2CompositionController,
    compositor: &Compositor,
    offscreen_only: bool,
) -> VisualTarget {
    let webview_controller: ICoreWebView2Controller3 = composition_controller.cast().unwrap();
    let size = window.inner_size();

    unsafe {
        webview_controller
            .SetBoundsMode(COREWEBVIEW2_BOUNDS_MODE_USE_RAW_PIXELS)
            .unwrap();

        webview_controller
            .SetShouldDetectMonitorScaleChanges(false)
            .unwrap();

        webview_controller.SetRasterizationScale(1.0).unwrap();
    }

    let root: ContainerVisual = compositor.CreateContainerVisual().unwrap();
    root.SetSize(Vector2::new(size.0 as f32, size.1 as f32))
        .unwrap(); // TODO resized?

    root.SetIsVisible(true).unwrap();

    let webview_visual = compositor.CreateSpriteVisual().unwrap();
    webview_visual
        .SetRelativeSizeAdjustment(Vector2::new(1., 1.))
        .unwrap();

    root.Children()
        .unwrap()
        .InsertAtTop(webview_visual.clone()) // FIXME ok?
        .unwrap();

    unsafe { composition_controller.SetRootVisualTarget(webview_visual) }.unwrap();
    unsafe { webview_controller.SetIsVisible(true) }.unwrap();

    if !offscreen_only {
        let interop: ICompositorDesktopInterop = compositor.cast().unwrap();
        let window_target: DesktopWindowTarget =
            unsafe { interop.CreateDesktopWindowTarget(window.hwnd(), true) }.unwrap();

        window_target.SetRoot(root).unwrap();
        VisualTarget::DesktopWindowTarget(window_target)
    } else {
        VisualTarget::ContainerVisual(root)
    }
}

#[derive(Clone)]
pub enum VisualTarget {
    DesktopWindowTarget(DesktopWindowTarget),
    ContainerVisual(ContainerVisual),
}
