/**
 * Based on
 * https://github.com/tauri-apps/wry/blob/d202573c2c68a2ff0411c1aa797ecc10f727e93b/src/webview/webview2/mod.rs
 *
 * MIT License
 *
 * Copyright (c) 2020-2021 Ngo Iok Ui & Tauri Apps Contributors
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 **/
use std::{collections::HashSet, rc::Rc, sync::mpsc};

use windows::{
    core::Interface,
    Win32::Foundation::*,
    Win32::System::Com::{IStream, StructuredStorage::CreateStreamOnHGlobal},
    Win32::System::WinRT::EventRegistrationToken,
    Win32::{
        Foundation::{E_POINTER, HWND, PWSTR, RECT},
        UI::WindowsAndMessaging::*,
    },
};

use headless_webview::{
    http::RequestBuilder as HttpRequestBuilder, webview::rpc_proxy,
    webview::web_context::WebContext, webview::WebViewAttributes, HeadlessWindow,
};

use webview2_com::{Microsoft::Web::WebView2::Win32::*, *};

use super::webview2_web_context::WindowsWebContext;
use super::Win32Window;

pub fn create_environment(
    web_context: Option<&mut WebContext<WindowsWebContext>>,
) -> webview2_com::Result<ICoreWebView2Environment3> {
    let (tx, rx) = mpsc::channel();

    let data_directory = web_context
        .as_deref()
        .and_then(|context| context.data_directory())
        .and_then(|path| path.to_str())
        .map(String::from);

    assert!(data_directory.is_none()); // FIXME remove /handle this

    CreateCoreWebView2EnvironmentCompletedHandler::wait_for_async_operation(
        Box::new(move |environmentcreatedhandler| unsafe {
            let options: ICoreWebView2EnvironmentOptions =
                CoreWebView2EnvironmentOptions::default().into();

            CreateCoreWebView2EnvironmentWithOptions(
                PWSTR::default(),
                None,
                options,
                environmentcreatedhandler,
            )
            .map_err(webview2_com::Error::WindowsError)
        }),
        Box::new(move |error_code, environment| {
            error_code.unwrap();
            tx.send(environment.ok_or_else(|| windows::core::Error::fast_error(E_POINTER)))
                .expect("send over mpsc channel");
            Ok(())
        }),
    )
    .unwrap();

    rx.recv()
        .map_err(|_| webview2_com::Error::SendError)?
        .map_err(webview2_com::Error::WindowsError)
        // TODO: document the effect of upcasting? - version limitations?
        .and_then(|env| Ok(env.cast::<ICoreWebView2Environment3>()?))
}

pub fn create_controller(
    hwnd: HWND,
    env: &ICoreWebView2Environment3,
) -> webview2_com::Result<ICoreWebView2CompositionController> {
    let (tx, rx) = mpsc::channel();
    let env = env.clone();

    CreateCoreWebView2CompositionControllerCompletedHandler::wait_for_async_operation(
        Box::new(move |handler| unsafe {
            env.CreateCoreWebView2CompositionController(hwnd, handler)
                .map_err(webview2_com::Error::WindowsError)
        }),
        Box::new(move |error_code, controller| {
            error_code.unwrap();
            tx.send(controller.ok_or_else(|| windows::core::Error::fast_error(E_POINTER)))
                .expect("send over mpsc channel");
            Ok(())
        }),
    )
    .unwrap();

    rx.recv()
        .map_err(|_| webview2_com::Error::SendError)?
        .map_err(webview2_com::Error::WindowsError)
}

pub fn init_webview(
    window: Rc<Win32Window>,
    hwnd: HWND,
    mut attributes: WebViewAttributes<Win32Window>,
    env: &ICoreWebView2Environment3,
    controller: &ICoreWebView2Controller,
) -> webview2_com::Result<ICoreWebView2> {
    let webview = unsafe { controller.CoreWebView2() }
        .map_err(webview2_com::Error::WindowsError)
        .unwrap();

    // Transparent
    if attributes.transparent && !is_windows_7() {
        let controller2: ICoreWebView2Controller2 = controller
            .cast()
            .map_err(webview2_com::Error::WindowsError)
            .unwrap();
        unsafe {
            controller2
                .SetDefaultBackgroundColor(COREWEBVIEW2_COLOR {
                    R: 0,
                    G: 0,
                    B: 0,
                    A: 0,
                })
                .map_err(webview2_com::Error::WindowsError)
                .unwrap();
        }
    }

    // The EventRegistrationToken is an out-param from all of the event registration calls. We're
    // taking it in the local variable and then just ignoring it because all of the event handlers
    // are registered for the life of the webview, but if we wanted to be able to remove them later
    // we would hold onto them in self.
    let mut token = EventRegistrationToken::default();

    // Safety: System calls are unsafe
    unsafe {
        let handler: ICoreWebView2WindowCloseRequestedEventHandler =
            WindowCloseRequestedEventHandler::create(Box::new(move |_, _| {
                if DestroyWindow(hwnd).as_bool() {
                    Ok(())
                } else {
                    Err(E_FAIL.into())
                }
            }));
        webview
            .WindowCloseRequested(handler, &mut token)
            .map_err(webview2_com::Error::WindowsError)
            .unwrap();

        let settings = webview
            .Settings()
            .map_err(webview2_com::Error::WindowsError)
            .unwrap();
        settings
            .SetIsStatusBarEnabled(false)
            .map_err(webview2_com::Error::WindowsError)
            .unwrap();
        settings
            .SetAreDefaultContextMenusEnabled(true)
            .map_err(webview2_com::Error::WindowsError)
            .unwrap();
        settings
            .SetIsZoomControlEnabled(false)
            .map_err(webview2_com::Error::WindowsError)
            .unwrap();
        settings
            .SetAreDevToolsEnabled(false)
            .map_err(webview2_com::Error::WindowsError)
            .unwrap();
        debug_assert_eq!(settings.SetAreDevToolsEnabled(true), Ok(()));

        /*
        let mut rect = RECT::default();
        GetClientRect(hwnd, &mut rect);
        controller
            .SetBounds(rect)
            .map_err(webview2_com::Error::WindowsError)
            .unwrap();
        */
        // TODO: is it ok to set non-window bounds? Or should this be from the visual?
        let size = window.inner_size();
        println!("SIZE: {:?}", size);
        controller
            .SetBounds(RECT {
                left: 0,
                top: 0,
                right: size.0 as i32,
                bottom: size.1 as i32,
            })
            .unwrap();
    }

    // Initialize scripts
    add_script_to_execute_on_document_created(
          &webview,
          String::from(
            r#"
              window.external={invoke:s=>window.chrome.webview.postMessage(s)};
              window.addEventListener('mousedown', (e) => {
                if (e.buttons === 1) window.chrome.webview.postMessage('__WEBVIEW_LEFT_MOUSE_DOWN__')
              });
              window.addEventListener('mousemove', (e) => window.chrome.webview.postMessage('__WEBVIEW_MOUSE_MOVE__'));
            "#,
          ),
        ).unwrap();
    for js in attributes.initialization_scripts {
        add_script_to_execute_on_document_created(&webview, js).unwrap();
    }

    // Message handler
    let rpc_handler = attributes.rpc_handler.take();
    unsafe {
        webview.WebMessageReceived(
            WebMessageReceivedEventHandler::create(Box::new(move |webview, args| {
                if let (Some(webview), Some(args)) = (webview, args) {
                    let mut js = PWSTR::default();
                    args.TryGetWebMessageAsString(&mut js).unwrap();
                    let js = take_pwstr(js);
                    if js == "__WEBVIEW_LEFT_MOUSE_DOWN__" || js == "__WEBVIEW_MOUSE_MOVE__" {
                        println!("MOUSE : {:?}", js);
                    }
                    // if js == "__WEBVIEW_LEFT_MOUSE_DOWN__" || js == "__WEBVIEW_MOUSE_MOVE__" {
                    //     if !window.is_decorated() && window.is_resizable() {
                    //         use crate::application::{
                    //             platform::windows::hit_test, window::CursorIcon,
                    //         };

                    //         let mut point = POINT::default();
                    //         GetCursorPos(&mut point);
                    //         let result = hit_test(HWND(window.hwnd() as _), point.x, point.y);
                    //         let cursor = match result.0 as u32 {
                    //             win32wm::HTLEFT => CursorIcon::WResize,
                    //             win32wm::HTTOP => CursorIcon::NResize,
                    //             win32wm::HTRIGHT => CursorIcon::EResize,
                    //             win32wm::HTBOTTOM => CursorIcon::SResize,
                    //             win32wm::HTTOPLEFT => CursorIcon::NwResize,
                    //             win32wm::HTTOPRIGHT => CursorIcon::NeResize,
                    //             win32wm::HTBOTTOMLEFT => CursorIcon::SwResize,
                    //             win32wm::HTBOTTOMRIGHT => CursorIcon::SeResize,
                    //             _ => CursorIcon::Arrow,
                    //         };
                    //         // don't use `CursorIcon::Arrow` variant or cursor manipulation using css will cause cursor flickering
                    //         if cursor != CursorIcon::Arrow {
                    //             window.set_cursor_icon(cursor);
                    //         }

                    //         if js == "__WEBVIEW_LEFT_MOUSE_DOWN__" {
                    //             // we ignore `HTCLIENT` variant so the webview receives the click correctly if it is not on the edges
                    //             // and prevent conflict with `tao::window::drag_window`.
                    //             if result.0 as u32 != win32wm::HTCLIENT {
                    //                 window.begin_resize_drag(
                    //                     result.0,
                    //                     WM_NCLBUTTONDOWN,
                    //                     point.x,
                    //                     point.y,
                    //                 );
                    //             }
                    //         }
                    //     }
                    //     // these are internal messages, rpc_handlers don't need it so exit early
                    //     return Ok(());
                    // }

                    if let Some(rpc_handler) = &rpc_handler {
                        match rpc_proxy(&window, js, rpc_handler) {
                            Ok(result) => {
                                if let Some(script) = result {
                                    execute_script(&webview, script).unwrap();
                                }
                            }
                            Err(e) => {
                                eprintln!("{}", e);
                            }
                        }
                    }
                }

                Ok(())
            })),
            &mut token,
        )
    }
    .map_err(webview2_com::Error::WindowsError)
    .unwrap();

    let mut custom_protocol_names = HashSet::new();
    if !attributes.custom_protocols.is_empty() {
        for (name, _) in &attributes.custom_protocols {
            // WebView2 doesn't support non-standard protocols yet, so we have to use this workaround
            // See https://github.com/MicrosoftEdge/WebView2Feedback/issues/73
            custom_protocol_names.insert(name.clone());
            unsafe {
                webview.AddWebResourceRequestedFilter(
                    format!("https://{}.*", name),
                    COREWEBVIEW2_WEB_RESOURCE_CONTEXT_ALL,
                )
            }
            .map_err(webview2_com::Error::WindowsError)
            .unwrap();
        }

        let custom_protocols = attributes.custom_protocols;
        let env = env.clone();
        unsafe {
            webview
                .WebResourceRequested(
                    WebResourceRequestedEventHandler::create(Box::new(move |_, args| {
                        if let Some(args) = args {
                            let webview_request = args.Request().unwrap();
                            let mut request = HttpRequestBuilder::new();

                            // request method (GET, POST, PUT etc..)
                            let mut request_method = PWSTR::default();
                            webview_request.Method(&mut request_method).unwrap();
                            let request_method = take_pwstr(request_method);

                            // get all headers from the request
                            let headers = webview_request.Headers()?.GetIterator().unwrap();
                            let mut has_current = BOOL::default();
                            headers.HasCurrentHeader(&mut has_current).unwrap();
                            if has_current.as_bool() {
                                loop {
                                    let mut key = PWSTR::default();
                                    let mut value = PWSTR::default();
                                    headers.GetCurrentHeader(&mut key, &mut value).unwrap();
                                    let (key, value) = (take_pwstr(key), take_pwstr(value));
                                    request = request.header(&key, &value);

                                    headers.MoveNext(&mut has_current).unwrap();
                                    if !has_current.as_bool() {
                                        break;
                                    }
                                }
                            }

                            // get the body content if available
                            let mut body_sent = Vec::new();
                            if let Ok(content) = webview_request.Content() {
                                let mut buffer: [u8; 1024] = [0; 1024];
                                loop {
                                    let mut cb_read = 0;
                                    let content: IStream = content.cast().unwrap();
                                    content
                                        .Read(
                                            buffer.as_mut_ptr() as *mut _,
                                            buffer.len() as u32,
                                            &mut cb_read,
                                        )
                                        .unwrap();

                                    if cb_read == 0 {
                                        break;
                                    }

                                    body_sent.extend_from_slice(&buffer[..(cb_read as usize)]);
                                }
                            }

                            // uri
                            let mut uri = PWSTR::default();
                            webview_request.Uri(&mut uri).unwrap();
                            let uri = take_pwstr(uri);

                            if let Some(custom_protocol) = custom_protocols
                                .iter()
                                .find(|(name, _)| uri.starts_with(&format!("https://{}.", name)))
                            {
                                // Undo the protocol workaround when giving path to resolver
                                let path = uri.replace(
                                    &format!("https://{}.", custom_protocol.0),
                                    &format!("{}://", custom_protocol.0),
                                );
                                let final_request = request
                                    .uri(&path)
                                    .method(request_method.as_str())
                                    .body(body_sent)
                                    .unwrap();

                                return match (custom_protocol.1)(&final_request) {
                                    Ok(sent_response) => {
                                        let content = sent_response.body();
                                        let status_code = sent_response.status().as_u16() as i32;

                                        let mut headers_map = String::new();

                                        // set mime type if provided
                                        if let Some(mime) = sent_response.mimetype() {
                                            headers_map
                                                .push_str(&format!("Content-Type: {}\n", mime))
                                        }

                                        // build headers
                                        for (name, value) in sent_response.headers().iter() {
                                            let header_key = name.to_string();
                                            if let Ok(value) = value.to_str() {
                                                headers_map.push_str(&format!(
                                                    "{}: {}\n",
                                                    header_key, value
                                                ))
                                            }
                                        }

                                        let mut body_sent = None;
                                        if !content.is_empty() {
                                            let stream = CreateStreamOnHGlobal(0, true).unwrap();
                                            stream.SetSize(content.len() as u64).unwrap();
                                            if stream.Write(
                                                content.as_ptr() as *const _,
                                                content.len() as u32,
                                            )?
                                                as usize
                                                == content.len()
                                            {
                                                body_sent = Some(stream);
                                            }
                                        }

                                        // FIXME: Set http response version

                                        let body_sent =
                                            body_sent.map(|content| content.cast().unwrap());
                                        let response = env
                                            .CreateWebResourceResponse(
                                                body_sent,
                                                status_code,
                                                "OK",
                                                headers_map,
                                            )
                                            .unwrap();

                                        args.SetResponse(response).unwrap();
                                        Ok(())
                                    }
                                    Err(_) => Err(E_FAIL.into()),
                                };
                            }
                        }

                        Ok(())
                    })),
                    &mut token,
                )
                .map_err(webview2_com::Error::WindowsError)
                .unwrap();
        }
    }

    // Enable clipboard
    // if attributes.clipboard {
    //     unsafe {
    //         webview
    //             .PermissionRequested(
    //                 PermissionRequestedEventHandler::create(Box::new(|_, args| {
    //                     if let Some(args) = args {
    //                         let mut kind = COREWEBVIEW2_PERMISSION_KIND_UNKNOWN_PERMISSION;
    //                         args.PermissionKind(&mut kind).unwrap();
    //                         if kind == COREWEBVIEW2_PERMISSION_KIND_CLIPBOARD_READ {
    //                             args.SetState(COREWEBVIEW2_PERMISSION_STATE_ALLOW).unwrap();
    //                         }
    //                     }
    //                     Ok(())
    //                 })),
    //                 &mut token,
    //             )
    //             .map_err(webview2_com::Error::WindowsError)
    //             .unwrap();
    //     }
    // }

    // Set user agent
    if let Some(user_agent) = attributes.user_agent {
        unsafe {
            let settings: ICoreWebView2Settings2 = webview
                .Settings()?
                .cast()
                .map_err(webview2_com::Error::WindowsError)
                .unwrap();
            settings
                .SetUserAgent(String::from(user_agent.as_str()))
                .unwrap();
        }
    }

    // Navigation
    if let Some(url) = attributes.url {
        if url.cannot_be_a_base() {
            let s = url.as_str();
            if let Some(pos) = s.find(',') {
                let (_, path) = s.split_at(pos + 1);
                unsafe {
                    webview
                        .NavigateToString(path.to_string())
                        .map_err(webview2_com::Error::WindowsError)
                        .unwrap();
                }
            }
        } else {
            let mut url_string = String::from(url.as_str());
            let name = url.scheme();
            if custom_protocol_names.contains(name) {
                // WebView2 doesn't support non-standard protocols yet, so we have to use this workaround
                // See https://github.com/MicrosoftEdge/WebView2Feedback/issues/73
                url_string = url
                    .as_str()
                    .replace(&format!("{}://", name), &format!("https://{}.", name))
            }
            unsafe {
                webview
                    .Navigate(url_string)
                    .map_err(webview2_com::Error::WindowsError)
                    .unwrap();
            }
        }
    } else if let Some(html) = attributes.html {
        unsafe {
            webview
                .NavigateToString(html)
                .map_err(webview2_com::Error::WindowsError)
                .unwrap();
        }
    }

    unsafe {
        controller
            .SetIsVisible(true)
            .map_err(webview2_com::Error::WindowsError)
            .unwrap();
        controller
            .MoveFocus(COREWEBVIEW2_MOVE_FOCUS_REASON_PROGRAMMATIC)
            .map_err(webview2_com::Error::WindowsError)
            .unwrap();
    }

    Ok(webview)
}

fn add_script_to_execute_on_document_created(
    webview: &ICoreWebView2,
    js: String,
) -> webview2_com::Result<()> {
    let handler_webview = webview.clone();
    AddScriptToExecuteOnDocumentCreatedCompletedHandler::wait_for_async_operation(
        Box::new(move |handler| unsafe {
            handler_webview
                .AddScriptToExecuteOnDocumentCreated(js, handler)
                .map_err(webview2_com::Error::WindowsError)
        }),
        Box::new(|_, _| Ok(())),
    )
}

pub fn execute_script(webview: &ICoreWebView2, js: String) -> windows::core::Result<()> {
    unsafe {
        webview.ExecuteScript(
            js,
            ExecuteScriptCompletedHandler::create(Box::new(|_, _| (Ok(())))),
        )
    }
}

fn is_windows_7() -> bool {
    sys_info::os_release()
        .map(|release| release.starts_with("6.1"))
        .unwrap_or_default()
}
