/**
 * Based partly on
 *   https://github.com/robmikh/screenshot-rs/blob/26a1bb9cabab7922a7725e7959c9f984c2d40473/src/main.rs
 *
 * MIT License
 *
 * Copyright (c) 2020 Robert Mikhayelyan
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */
use std::{
    thread,
    time::{Duration, Instant},
};

use crossbeam_channel::{Receiver, Sender};

use crossbeam_deque::{Steal, Worker};
use windows::{
    core::{IInspectable, Interface},
    Foundation::TypedEventHandler,
    Graphics::{
        Capture::{Direct3D11CaptureFramePool, GraphicsCaptureItem},
        DirectX::DirectXPixelFormat,
        SizeInt32,
    },
    Win32::Graphics::Direct3D11::{
        ID3D11Resource, ID3D11Texture2D, D3D11_CPU_ACCESS_READ, D3D11_MAP_READ,
        D3D11_TEXTURE2D_DESC, D3D11_USAGE_STAGING,
    },
    UI::Composition::Visual,
};

use headless_webview::types::{self, Texture};

use super::{webview2_d3d, webview2_visual::VisualTarget};

pub fn initialize_capture_thread(
    container_visual: VisualTarget,
    texture_sender: Worker<(Instant, Texture)>,
    capture_event_receiver: Receiver<CaptureEvent>,
) {
    thread::spawn(move || {
        let capture_item = GraphicsCaptureItem::CreateFromVisual(match &container_visual {
            VisualTarget::DesktopWindowTarget(target) => target.Root().unwrap(),
            VisualTarget::ContainerVisual(target) => target.cast::<Visual>().unwrap(),
        })
        .unwrap();

        let item_size = capture_item.Size().unwrap();

        let d3d_device = webview2_d3d::create_d3d_device().unwrap();
        let d3d_context = unsafe {
            let mut d3d_context = None;
            d3d_device.GetImmediateContext(&mut d3d_context);
            d3d_context.unwrap()
        };

        let device = webview2_d3d::create_direct3d_device(&d3d_device).unwrap();
        let frame_pool = Direct3D11CaptureFramePool::CreateFreeThreaded(
            &device,
            DirectXPixelFormat::B8G8R8A8UIntNormalized,
            2,
            &item_size,
        )
        .unwrap();

        let session = frame_pool.CreateCaptureSession(capture_item).unwrap();

        let worker = crossbeam_deque::Worker::<(Instant, ID3D11Texture2D)>::new_lifo();
        let stealer = worker.stealer();

        frame_pool
            .FrameArrived(
                TypedEventHandler::<Direct3D11CaptureFramePool, IInspectable>::new({
                    // TODO: are these clones really needed?
                    let d3d_device = d3d_device.clone();
                    let d3d_context = d3d_context.clone();

                    move |frame_pool, _| unsafe {
                        // TODO: is_running check - if hidden, ignore
                        let frame_pool = frame_pool.as_ref().unwrap();
                        let frame = frame_pool.TryGetNextFrame().unwrap();

                        let source_texture: ID3D11Texture2D =
                            webview2_d3d::get_d3d_interface_from_object(&frame.Surface()?).unwrap();

                        let mut desc = D3D11_TEXTURE2D_DESC::default();

                        source_texture.GetDesc(&mut desc);
                        desc.BindFlags = 0;
                        desc.MiscFlags = 0;
                        desc.Usage = D3D11_USAGE_STAGING;
                        desc.CPUAccessFlags = D3D11_CPU_ACCESS_READ;

                        let copy_texture = { d3d_device.CreateTexture2D(&desc, std::ptr::null())? };

                        d3d_context
                            .CopyResource(Some(copy_texture.cast()?), Some(source_texture.cast()?));
                        // TODO: CopySubresourceRegion ?

                        let i = Instant::now();
                        println!("PUSH {:?} w={} h={}", i, desc.Width, desc.Height);
                        worker.push((i, copy_texture));

                        Ok(())
                    }
                }),
            )
            .unwrap();

        session.StartCapture().unwrap();

        loop {
            // consume
            while stealer.len() > 1 {
                let _ = stealer.steal();
            }

            if let Steal::Success((instant, texture)) = stealer.steal() {
                let mut desc = D3D11_TEXTURE2D_DESC::default();

                let mut bits = unsafe {
                    texture.GetDesc(&mut desc as *mut _);

                    println!("HAVE TEX! {:?} w={} h={}", instant, desc.Width, desc.Height);
                    let resource: ID3D11Resource = texture.cast().unwrap();
                    let mapped = d3d_context
                        .Map(Some(resource.clone()), 0, D3D11_MAP_READ, 0)
                        .unwrap();

                    // Get a slice of bytes
                    let slice: &[u8] = {
                        std::slice::from_raw_parts(
                            mapped.pData as *const _,
                            (desc.Height * mapped.RowPitch) as usize,
                        )
                    };

                    let bytes_per_pixel = 4;
                    let mut bits = vec![0u8; (desc.Width * desc.Height * bytes_per_pixel) as usize];
                    for row in 0..desc.Height {
                        let data_begin = (row * (desc.Width * bytes_per_pixel)) as usize;
                        let data_end = ((row + 1) * (desc.Width * bytes_per_pixel)) as usize;
                        let slice_begin = (row * mapped.RowPitch) as usize;
                        let slice_end = slice_begin + (desc.Width * bytes_per_pixel) as usize;
                        bits[data_begin..data_end].copy_from_slice(&slice[slice_begin..slice_end]);
                    }

                    d3d_context.Unmap(Some(resource), 0);

                    bits
                };

                // BGRA => RGBA
                bits.chunks_mut(4).for_each(|chunk| chunk.swap(0, 2));

                let _ = texture_sender.push((
                    instant,
                    Texture {
                        width: desc.Width as usize,
                        height: desc.Height as usize,
                        format: types::TextureFormat::Rgba8,
                        data: bits,
                    },
                ));
            }

            for event in capture_event_receiver.try_iter() {
                match event {
                    CaptureEvent::AppExit => break,
                    CaptureEvent::Resize(size) => frame_pool
                        .Recreate(
                            &device,
                            DirectXPixelFormat::B8G8R8A8UIntNormalized,
                            2,
                            &SizeInt32 {
                                Width: size.0,
                                Height: size.1,
                            },
                        )
                        .unwrap(),
                }
            }

            thread::sleep(Duration::from_millis(5));
        }
    });
}

pub enum CaptureEvent {
    AppExit,
    Resize((i32, i32)),
}
