use std::{rc::Rc, time::Instant};

use crossbeam_channel::{bounded, unbounded, Receiver, Sender, TryRecvError};

use crossbeam_deque::{Steal, Stealer};
use windows::{
    core::Interface,
    Foundation::Numerics::Vector2,
    Win32::{
        Foundation::*,
        System::WinRT::{
            CreateDispatcherQueueController, DispatcherQueueOptions, DQTAT_COM_STA,
            DQTYPE_THREAD_CURRENT,
        },
        UI::WindowsAndMessaging::*,
    },
    UI::Composition::Compositor,
};

use headless_webview::{
    types::{self, Texture},
    webview::web_context::WebContext,
    webview::WebViewAttributes,
    EngineWebview, HeadlessWindow, Result,
};

use webview2_com::{Microsoft::Web::WebView2::Win32::*, *};

use super::{
    webview2_capture::{initialize_capture_thread, CaptureEvent},
    webview2_visual::{setup_webview_visual, VisualTarget},
    Win32Window,
};
use super::{webview2_impl, webview2_web_context::WindowsWebContext};

pub struct WindowsWebview {
    controller: ICoreWebView2CompositionController,
    webview: ICoreWebView2,
    window: Rc<Win32Window>,
    tick_count: usize,
    compositor: Compositor,
    texture_receiver: Stealer<(Instant, Texture)>,
    capture_event_sender: Sender<CaptureEvent>,
    container_visual: VisualTarget,
}

impl EngineWebview for WindowsWebview {
    type Window = Win32Window;
    type WebContext = WindowsWebContext;

    fn new(
        window: Rc<Self::Window>,
        attributes: WebViewAttributes<Self::Window>,
        web_context: Option<&mut WebContext<Self::WebContext>>,
    ) -> Result<Self>
    where
        Self: Sized,
    {
        // TODO: Create only once
        let dispatcher_options = DispatcherQueueOptions {
            dwSize: std::mem::size_of::<DispatcherQueueOptions>() as u32,
            threadType: DQTYPE_THREAD_CURRENT,
            apartmentType: DQTAT_COM_STA,
        };

        let _dispatcher_queue_controller =
            unsafe { CreateDispatcherQueueController(dispatcher_options) }.unwrap();

        // start window setup
        let env = webview2_impl::create_environment(web_context).unwrap();
        let controller = webview2_impl::create_controller(window.hwnd(), &env).unwrap();

        let compositor = Compositor::new().unwrap();
        let container_visual = setup_webview_visual(&window, &controller, &compositor, false);

        let webview = webview2_impl::init_webview(
            window.clone(),
            window.hwnd(),
            attributes,
            &env,
            &controller.cast::<ICoreWebView2Controller>().unwrap(),
        )
        .unwrap();

        let worker = crossbeam_deque::Worker::<(Instant, Texture)>::new_lifo();
        let stealer = worker.stealer();

        let (capture_event_sender, capture_event_receiver) = unbounded::<CaptureEvent>();

        initialize_capture_thread(container_visual.clone(), worker, capture_event_receiver);

        Ok(Self {
            controller,
            webview,
            window,
            tick_count: 0,
            compositor,
            texture_receiver: stealer,
            capture_event_sender,
            container_visual,
        })
    }

    fn window(&self) -> &Self::Window {
        &self.window
    }

    fn evaluate_script(&self, js: &str) -> Result<()> {
        Ok(webview2_impl::execute_script(&self.webview, js.to_string())
            //.map_err(|err| Error::WebView2Error(webview2_com::Error::WindowsError(err)))
            .unwrap()) // FIXME
    }

    fn resize(&self, new_size: (u32, u32)) -> Result<()> {
        self.window()._resize(new_size)?;

        unsafe {
            let mut rect = RECT::default();
            GetClientRect(self.window().hwnd(), &mut rect);

            println!("============ RECT: {:?}", rect);
        }

        // TODO: is this really needed?
        let new_size_vec2 = Vector2::new(new_size.0 as f32, new_size.1 as f32);
        match &self.container_visual {
            VisualTarget::DesktopWindowTarget(dwt) => {
                dwt.Root().unwrap().SetSize(new_size_vec2).unwrap()
            }
            VisualTarget::ContainerVisual(cv) => cv.SetSize(new_size_vec2).unwrap(),
        }

        unsafe {
            let mut rect = RECT::default();
            GetClientRect(self.window().hwnd(), &mut rect);

            println!("============ RECT: {:?}", rect);
            self.controller
                .cast::<ICoreWebView2Controller>()
                .unwrap()
                .SetBounds(rect)
                .unwrap();
        }

        self.capture_event_sender
            .send(CaptureEvent::Resize((new_size.0 as i32, new_size.1 as i32)))
            .unwrap();

        Ok(())
    }

    fn version(&self) -> Result<String> {
        let mut versioninfo = PWSTR::default();
        unsafe { GetAvailableCoreWebView2BrowserVersionString(PWSTR::default(), &mut versioninfo) }
            .map_err(webview2_com::Error::WindowsError)
            .unwrap();
        Ok(format!("webview2-v{}", take_pwstr(versioninfo)))
    }

    fn send_keyboard_input(&self, keyboard_input: types::KeyboardInput) {
        println!("KEYBOARD: {:?}", keyboard_input);
        // https://stackoverflow.com/questions/1220820/how-do-i-send-key-strokes-to-a-window-without-having-to-activate-it-using-window

        // LPCWSTR Target_window_Name = TEXT("Untitled - Notepad"); //<- Has to match window name
        // HWND hWindowHandle = FindWindow(NULL, Target_window_Name);
        // HWND EditClass = FindWindowEx(hWindowHandle, NULL, L"Edit", NULL);

        //let handle = self.webview.Handle();
        return;

        unsafe extern "system" fn jeh(
            hwnd: HWND,
            param: windows::Win32::Foundation::LPARAM,
        ) -> BOOL {
            println!("WINDOW");
            unsafe {
                let ret = SendMessageA(hwnd, WM_KEYDOWN, WPARAM(0x5A), LPARAM(0x002C0001));
                println!("RES: {:?}", ret);

                let ret = SendMessageA(hwnd, WM_CHAR, WPARAM(0x7A), LPARAM(0x002C0001));
                println!("RES: {:?}", ret);

                let ret = SendMessageA(hwnd, WM_KEYUP, WPARAM(0x5A), LPARAM(0x002C0001));
                println!("RES: {:?}", ret);
            };

            BOOL(1)
        }

        let w = unsafe { EnumChildWindows(self.window().hwnd(), Some(jeh), None) };
        let handle = unsafe { GetWindow(self.window().hwnd(), GW_CHILD) };

        println!("SENDING INPUT!");
        unsafe {
            let ret = SendMessageA(handle, WM_KEYDOWN, WPARAM(0x5A), LPARAM(0x002C0001));
            println!("RES: {:?}", ret);

            let ret = SendMessageA(handle, WM_CHAR, WPARAM(0x7A), LPARAM(0x002C0001));
            println!("RES: {:?}", ret);

            let ret = SendMessageA(handle, WM_KEYUP, WPARAM(0x5A), LPARAM(0x002C0001));
            println!("RES: {:?}", ret);
        };
        // SendMessage(EditClass, WM_KEYDOWN, 0x5A, 0x002C0001);
        // SendMessage(EditClass, WM_CHAR, 0x7A, 0x002C0001); //"z"
        // SendMessage(EditClass, WM_KEYUP, 0x5A, 0xC02C0001);
    }

    fn send_mouse_event(&self, mouse_event: types::MouseEvent) {
        println!("MOUSE: {:?}", mouse_event);
        let position = (mouse_event.position.x as isize) << 16 | mouse_event.position.x as isize;

        // https://stackoverflow.com/questions/10355286/programmatically-mouse-click-in-another-window
        // https://docs.microsoft.com/en-us/windows/win32/api/winuser/nf-winuser-sendinput?redirectedfrom=MSDN
        // https://stackoverflow.com/a/12363411
        unsafe extern "system" fn jeh(hwnd: HWND, position: LPARAM) -> BOOL {
            println!("WINDOW: {:?}", hwnd);
            let ret = SendMessageA(hwnd, WM_LBUTTONDOWN, WPARAM(0x00000001), position);
            println!("RES: {:?}", ret);

            let ret = SendMessageA(hwnd, WM_LBUTTONUP, WPARAM(0x00000001), position);
            println!("RES: {:?}", ret);

            BOOL(0)
        }

        println!("============ MOUSE");
        let ret = unsafe { EnumChildWindows(self.window().hwnd(), Some(jeh), LPARAM(position)) };
    }

    fn get_texture(&mut self) -> Result<Option<types::Texture>> {
        while self.texture_receiver.len() > 1 {
            let _ = self.texture_receiver.steal();
        }

        if let Steal::Success((instant, texture)) = self.texture_receiver.steal() {
            println!("STEALED TEXTURE: {:?}", instant);
            Ok(Some(texture))
        } else {
            Ok(None)
        }
    }

    fn tick_once(&mut self) {
        self.tick_count += 1;

        let mut message = MSG::default(); // TODO: move to class for perf?

        unsafe {
            while PeekMessageA(&mut message, HWND(0), 0, 0, PM_REMOVE).into() {
                TranslateMessage(&mut message);
                DispatchMessageA(&mut message);
            }
        }
    }

    fn close(&mut self) {
        self.capture_event_sender
            .send(CaptureEvent::AppExit)
            .unwrap();
    }
}
