use core::{
    cell::Cell,
    fmt::{self, Formatter},
    hash::Hash,
    mem,
};

use super::{node::Node, Ham};
use crate::{FromIndex, IntoIndex};

/// An view into an entry in a [`Ham`]
///
/// [`Ham`]: super::Ham
pub enum Entry<'a, K, V> {
    Occupied(OccupiedEntry<'a, K, V>),
    Vacant(VacantEntry<'a, K, V>),
}

use Entry::{Occupied, Vacant};

impl<K: fmt::Debug, V: fmt::Debug> fmt::Debug for Entry<'_, K, V> {
    fn fmt(&self, f: &mut Formatter<'_>) -> fmt::Result {
        match *self {
            Vacant(ref v) => f.debug_tuple("Entry").field(v).finish(),
            Occupied(ref o) => f.debug_tuple("Entry").field(o).finish(),
        }
    }
}

unsafe fn extend_lifetime<'a, 'b, T>(v: &'a mut T) -> &'b mut T {
    unsafe { &mut *(v as *mut _) }
}

impl<'a, K: IntoIndex + FromIndex + Hash + Eq, V> Entry<'a, K, V> {
    pub(super) fn make_new(ham: &'a mut Ham<K, V>, key: K) -> Self {
        if let Some(v) = ham.get_array_mut(&key) {
            // We will use the array
            // SAFETY: Ham is not used after this, since Self borrows it.
            let v = unsafe { extend_lifetime(v) };
            if v.is_some() {
                Occupied(OccupiedEntry {
                    entry: OccupiedEntryTy::Array(key, v),
                })
            } else {
                Vacant(VacantEntry {
                    key,
                    entry: VacantEntryTy::Array(v),
                })
            }
        } else if let Some((node, is_prev)) = ham.get_hash_mut_for_take(&key) {
            // We will use the hash
            // SAFETY: Ham is not used after this, since Self borrows it.
            let node = unsafe { extend_lifetime(node) };
            if is_prev {
                Occupied(OccupiedEntry {
                    entry: OccupiedEntryTy::HashPrev(key, node),
                })
            } else {
                Occupied(OccupiedEntry {
                    entry: OccupiedEntryTy::Hash(key, node),
                })
            }
        } else {
            Vacant(VacantEntry {
                key,
                entry: VacantEntryTy::NotThere(ham),
            })
        }
    }

    /// Ensures a value is in the entry by inserting the default if empty,
    /// and returns a mutable reference to the value.
    ///
    /// # Examples
    ///
    /// ```
    /// use hash_arr_map::Ham;
    ///
    /// let mut map = Ham::new();
    ///
    /// map.entry(1).or_insert(5);
    /// assert_eq!(map[&1], 5);
    ///
    /// *map.entry(1).or_insert(10) += 1;
    /// assert_eq!(map[&1], 6);
    /// ```
    pub fn or_insert(self, default: V) -> &'a mut V {
        match self {
            Occupied(o) => o.into_mut(),
            Vacant(v) => v.insert(default),
        }
    }

    /// Ensures a value is in the entry by inserting the result of the
    /// default function if empty, and returns a mutable reference to the
    /// value.
    ///
    /// # Examples
    ///
    /// ```
    /// use hash_arr_map::Ham;
    ///
    /// let mut map: Ham<u8, String> = Ham::new();
    ///
    /// map.entry(6).or_insert_with(|| "pick up sticks".to_string());
    ///
    /// assert_eq!(map[&6], "pick up sticks");
    /// ```
    pub fn or_insert_with<F: FnOnce() -> V>(self, default: F) -> &'a mut V {
        match self {
            Occupied(o) => o.into_mut(),
            Vacant(v) => v.insert(default()),
        }
    }

    /// Ensures a value is in the entry by inserting the result of the
    /// default function if empty, and returns a mutable reference to the
    /// value. The default function is given a reference to the key, so
    /// you can have key-derived values inserted without having to clone
    /// the key.
    ///
    /// # Examples
    ///
    /// ```
    /// use hash_arr_map::Ham;
    ///
    /// let mut repeats: Ham<usize, String> = Ham::new();
    ///
    /// repeats.entry(6).or_insert_with_key(|&n| "x".repeat(n));
    ///
    /// assert_eq!(repeats[&6], "xxxxxx");
    /// ```
    pub fn or_insert_with_key<F: FnOnce(&K) -> V>(self, default: F) -> &'a mut V {
        match self {
            Occupied(o) => o.into_mut(),
            Vacant(v) => {
                let value = default(v.key());
                v.insert(value)
            }
        }
    }

    // pub fn insert(self, value: V) -> OccupiedEntry<'a, K, V> {
    //     match self {
    //         Occupied(mut o) => {
    //             o.insert(value);
    //             o
    //         },
    //         Vacant(v) => v.insert_entry(value),
    //     }
    // }
}

impl<'a, K: IntoIndex + FromIndex + Hash + Eq, V: Default> Entry<'a, K, V> {
    pub fn or_default(self) -> &'a mut V {
        match self {
            Occupied(o) => o.into_mut(),
            Vacant(v) => v.insert(Default::default()),
        }
    }
}

impl<'a, K, V> Entry<'a, K, V> {
    /// Returns a reference to this entry's key.
    ///
    /// # Examples
    ///
    /// ```
    /// use hash_arr_map::Ham;
    ///
    /// let mut map: Ham<String, ()> = Ham::new();
    /// assert_eq!(map.entry("rust".into()).key(), "rust");
    /// ```
    pub const fn key(&self) -> &K {
        match *self {
            Occupied(ref o) => o.key(),
            Vacant(ref v) => v.key(),
        }
    }

    /// Provides in-place mutable access to an occupied entry before any
    /// potential inserts into the map.
    ///
    /// # Examples
    ///
    /// ```
    /// use hash_arr_map::Ham;
    ///
    /// let mut map: Ham<u32, String> = Ham::new();
    ///
    /// map.entry(1)
    ///     .and_modify(|e| e.push_str(" again"))
    ///     .or_insert_with(|| "Hello".to_string());
    /// assert_eq!(map[&1], "Hello");
    ///
    /// map.entry(1)
    ///     .and_modify(|e| e.push_str(" again"))
    ///     .or_insert_with(|| "Hello".to_string());
    /// assert_eq!(map[&1], "Hello again");
    /// ```
    pub fn and_modify<F: FnOnce(&mut V)>(self, f: F) -> Self {
        match self {
            Occupied(mut o) => {
                f(o.get_mut());
                Occupied(o)
            }
            Vacant(v) => Vacant(v),
        }
    }
}

/// An occupied slot in a [`Ham`]
///
/// [`Ham`]: super::Ham
pub struct OccupiedEntry<'a, K, V> {
    entry: OccupiedEntryTy<'a, K, V>,
}

impl<K: fmt::Debug, V: fmt::Debug> fmt::Debug for OccupiedEntry<'_, K, V> {
    fn fmt(&self, f: &mut Formatter<'_>) -> fmt::Result {
        f.debug_struct("OccupiedEntry")
            .field("key", self.key())
            .field("value", self.get())
            .finish()
    }
}

impl<'a, K, V> OccupiedEntry<'a, K, V> {
    pub const fn key(&self) -> &K {
        use OccupiedEntryTy::*;
        match self.entry {
            Array(ref k, _) | Hash(ref k, _) | HashPrev(ref k, _) => k,
        }
    }

    pub fn get(&self) -> &V {
        use OccupiedEntryTy::*;
        match self.entry {
            Array(_, ref arr) => unsafe { arr.as_ref().unwrap_unchecked() },
            Hash(_, ref node) => &unsafe { node.value().as_ref().unwrap_unchecked() }.1,
            HashPrev(_, ref node) => {
                &unsafe { node.next_ref().value().as_ref().unwrap_unchecked() }.1
            }
        }
    }

    pub fn get_mut(&mut self) -> &mut V {
        use OccupiedEntryTy::*;
        match self.entry {
            Array(_, ref mut arr) => unsafe { arr.as_mut().unwrap_unchecked() },
            Hash(_, ref mut node) => &mut unsafe { node.value_mut().as_mut().unwrap_unchecked() }.1,
            HashPrev(_, ref mut node) => {
                &mut unsafe { node.next_mut().value_mut().as_mut().unwrap_unchecked() }.1
            }
        }
    }

    pub fn insert(&mut self, value: V) -> V {
        mem::replace(self.get_mut(), value)
    }

    pub fn into_mut(self) -> &'a mut V {
        use OccupiedEntryTy::*;
        match self.entry {
            Array(_, arr) => unsafe { arr.as_mut().unwrap_unchecked() },
            Hash(_, node) => &mut unsafe { node.value_mut().as_mut().unwrap_unchecked() }.1,
            HashPrev(_, node) => {
                &mut unsafe { node.next_mut().value_mut().as_mut().unwrap_unchecked() }.1
            }
        }
    }

    pub fn remove(self) -> V {
        self.remove_entry().1
    }

    pub fn remove_entry(self) -> (K, V) {
        use OccupiedEntryTy::*;
        match self.entry {
            Array(k, arr) => {
                let v = unsafe { arr.take().unwrap_unchecked() };
                (k, v)
            }
            HashPrev(_, prev_v) => {
                let prev_v = Cell::from_mut(prev_v);
                let v = Node::next(prev_v);
                let next_v = Node::next(v);

                // prev_v -> v -> next_v...

                Node::set_next(prev_v, next_v);
                Node::set_next(v, v);

                // prev_v -> next_v...; v

                unsafe { (&mut *v.as_ptr()).value_mut().take().unwrap_unchecked() }
            }
            Hash(_, v) => {
                // This is like in Ham::get_hash_take
                let v = Cell::from_mut(v);

                if Node::next(v) as *const _ == v {
                    // No chain: we can just remove it and be done.
                    unsafe { (&mut *v.as_ptr()).value_mut().take().unwrap_unchecked() }
                } else {
                    let next_v = Node::next(v);
                    let next_next_v = Node::next(next_v);

                    // v -> next_v -> next_next_v

                    // We move next_v into v and return the value of v

                    let next_v_value = next_v.replace(Node::default());

                    let v_value = v.replace(next_v_value);

                    // We now fix the pointer in next_v_value

                    // If there was something after next_v, then set it
                    if next_next_v as *const _ != next_v {
                        Node::set_next(v, next_next_v);
                    }

                    // v_next (in *v) -> v_next_next -> ...

                    unsafe { v_value.into_inner().unwrap_unchecked() }
                }
            }
        }
    }
}

enum OccupiedEntryTy<'a, K, V> {
    Array(K, &'a mut Option<V>),
    Hash(K, &'a mut Node<K, V>),
    HashPrev(K, &'a mut Node<K, V>),
}

pub struct VacantEntry<'a, K, V> {
    key: K,
    entry: VacantEntryTy<'a, K, V>,
}

impl<K: fmt::Debug, V: fmt::Debug> fmt::Debug for VacantEntry<'_, K, V> {
    fn fmt(&self, f: &mut Formatter<'_>) -> fmt::Result {
        f.debug_tuple("VacantEntry").field(self.key()).finish()
    }
}

impl<'a, K, V> VacantEntry<'a, K, V> {
    pub const fn key(&self) -> &K {
        &self.key
    }

    pub const fn into_key(self) -> K {
        self.key
    }
}

impl<'a, K: IntoIndex + FromIndex + Hash + Eq, V> VacantEntry<'a, K, V> {
    pub fn insert(self, value: V) -> &'a mut V {
        use VacantEntryTy::*;
        match self.entry {
            Array(arr) => arr.insert(value),
            NotThere(ham) => ham.new_key(self.key, value),
        }
    }
}

enum VacantEntryTy<'a, K, V> {
    Array(&'a mut Option<V>),
    NotThere(&'a mut Ham<K, V>),
}
