use core::hash::{BuildHasher, Hasher};

#[derive(Copy, Clone, Debug, Eq, PartialEq)]
pub(super) struct LuaBuildHasher(u64);

impl LuaBuildHasher {
    #[cfg(feature = "std")]
    pub(super) fn new() -> Self {
        Self(fastrand::u64(..))
    }

    pub(super) const fn with_seed(seed: u64) -> Self {
        Self(seed)
    }
}

impl BuildHasher for LuaBuildHasher {
    type Hasher = LuaHasher;

    fn build_hasher(&self) -> LuaHasher {
        LuaHasher::new(self.0)
    }
}

pub(super) struct LuaHasher {
    state: u64,
}

impl LuaHasher {
    const fn new(seed: u64) -> Self {
        Self { state: seed }
    }
}

impl Hasher for LuaHasher {
    fn write(&mut self, bytes: &[u8]) {
        let mut state = self.state;
        for byte in bytes {
            state ^= (state << 5)
                .wrapping_add(state >> 2)
                .wrapping_add((*byte).into());
        }
        self.state = state;
    }

    fn write_u8(&mut self, n: u8) {
        self.write_u64(n.into());
    }

    fn write_u16(&mut self, n: u16) {
        self.write_u64(n.into());
    }

    fn write_u32(&mut self, n: u32) {
        self.write_u64(n.into());
    }

    fn write_u64(&mut self, n: u64) {
        let state = self.state;
        self.state ^= (state << 5).wrapping_add(state >> 2).wrapping_add(n);
    }

    fn write_u128(&mut self, n: u128) {
        self.write_u64((n >> 64) as u64);
        self.write_u64(n as u64);
    }

    fn write_usize(&mut self, n: usize) {
        #[cfg(target_pointer_width = "16")]
        self.write_u16(n as u16);
        #[cfg(target_pointer_width = "32")]
        self.write_u32(n as u32);
        #[cfg(target_pointer_width = "64")]
        self.write_u64(n as u64);
    }

    fn finish(&self) -> u64 {
        self.state
    }
}
