use core::{
    fmt,
    fmt::{Binary, Debug, Display, Formatter, LowerExp, LowerHex, Octal, UpperExp, UpperHex},
    num::NonZeroUsize,
    ops::Deref,
};

use crate::{FromIndex, Idx, IntoIndex, NewIndex};

/// This is partly like [`Cow`], but the 'owned' type is the same as the 'borrowed'
/// type.
///
/// [`Cow`]: alloc::borrow::Cow
#[cfg_attr(feature = "gc", derive(gc::Trace))]
#[derive(Copy, Clone)]
pub enum Key<'a, T> {
    Owned(T),
    Borrowed(&'a T),
}

impl<'a, T> Deref for Key<'a, T> {
    type Target = T;
    fn deref(&self) -> &T {
        match *self {
            Key::Owned(ref t) => t,
            Key::Borrowed(t) => t,
        }
    }
}

impl<'a, T: IntoIndex> IntoIndex for Key<'a, T> {
    fn into_index(&self) -> Option<Idx<Self>> {
        Some(unsafe { (**self).into_index()?.cast() })
    }
}

impl<'a, T: FromIndex> FromIndex for Key<'a, T> {
    fn from_index(v: Idx<Self>) -> Self {
        Self::Owned(T::from_index(unsafe { v.cast() }))
    }
}

unsafe impl<'a, T: NewIndex> NewIndex for Key<'a, T> {
    fn new_index_allowed(idx: NonZeroUsize) -> bool {
        T::new_index_allowed(idx)
    }
}

// All the formatting traits

impl<T: Display> Display for Key<'_, T> {
    fn fmt(&self, f: &mut Formatter) -> fmt::Result {
        Display::fmt(&**self, f)
    }
}

impl<T: Debug> Debug for Key<'_, T> {
    fn fmt(&self, f: &mut Formatter) -> fmt::Result {
        Debug::fmt(&**self, f)
    }
}

impl<T: Binary> Binary for Key<'_, T> {
    fn fmt(&self, f: &mut Formatter) -> fmt::Result {
        Binary::fmt(&**self, f)
    }
}

impl<T: LowerExp> LowerExp for Key<'_, T> {
    fn fmt(&self, f: &mut Formatter) -> fmt::Result {
        LowerExp::fmt(&**self, f)
    }
}

impl<T: LowerHex> LowerHex for Key<'_, T> {
    fn fmt(&self, f: &mut Formatter) -> fmt::Result {
        LowerHex::fmt(&**self, f)
    }
}

impl<T: Octal> Octal for Key<'_, T> {
    fn fmt(&self, f: &mut Formatter) -> fmt::Result {
        Octal::fmt(&**self, f)
    }
}

impl<T: UpperExp> UpperExp for Key<'_, T> {
    fn fmt(&self, f: &mut Formatter) -> fmt::Result {
        UpperExp::fmt(&**self, f)
    }
}

impl<T: UpperHex> UpperHex for Key<'_, T> {
    fn fmt(&self, f: &mut Formatter) -> fmt::Result {
        UpperHex::fmt(&**self, f)
    }
}
