use alloc::vec;
use core::{
    iter::{Enumerate, ExactSizeIterator, FusedIterator},
    slice,
};

use super::{Ham, Node, Nzu};
use crate::{FromIndex, Key}; // NonZeroUsize

impl<'a, K: FromIndex, V> IntoIterator for &'a Ham<K, V> {
    type Item = (Key<'a, K>, &'a V);

    type IntoIter = Iter<'a, K, V>;

    fn into_iter(self) -> Iter<'a, K, V> {
        self.iter()
    }
}

#[derive(Debug, Clone)]
pub struct Iter<'a, K, V> {
    pub(super) array: Enumerate<slice::Iter<'a, Option<V>>>,
    pub(super) hash: slice::Iter<'a, Node<K, V>>,
    pub(super) len: usize,
}

impl<'a, K: FromIndex + 'a, V: 'a> Iterator for Iter<'a, K, V> {
    type Item = (Key<'a, K>, &'a V);

    fn next(&mut self) -> Option<(Key<'a, K>, &'a V)> {
        for (i, v) in &mut self.array {
            if let Some(ref val) = *v {
                self.len -= 1;
                return Some((unsafe { Key::from_index(Nzu::new_unchecked(i + 1)) }, val));
            }
        }

        for v in &mut self.hash {
            if let Some((ref k, ref v)) = *v.value() {
                self.len -= 1;
                return Some((Key::Borrowed(k), v));
            }
        }

        debug_assert_eq!(self.len, 0);

        None
    }

    fn size_hint(&self) -> (usize, Option<usize>) {
        (self.len, Some(self.len))
    }
}

impl<K: FromIndex, V> ExactSizeIterator for Iter<'_, K, V> {
    fn len(&self) -> usize {
        self.len
    }
}

impl<K: FromIndex, V> FusedIterator for Iter<'_, K, V> {}

impl<'a, K: FromIndex, V> IntoIterator for &'a mut Ham<K, V> {
    type Item = (Key<'a, K>, &'a mut V);

    type IntoIter = IterMut<'a, K, V>;

    fn into_iter(self) -> IterMut<'a, K, V> {
        self.iter_mut()
    }
}

#[derive(Debug)]
pub struct IterMut<'a, K, V> {
    pub(super) array: Enumerate<slice::IterMut<'a, Option<V>>>,
    pub(super) hash: slice::IterMut<'a, Node<K, V>>,
    pub(super) len: usize,
}

impl<'a, K: FromIndex, V> Iterator for IterMut<'a, K, V> {
    type Item = (Key<'a, K>, &'a mut V);

    fn next(&mut self) -> Option<(Key<'a, K>, &'a mut V)> {
        for (i, v) in &mut self.array {
            if let Some(ref mut val) = *v {
                self.len -= 1;
                return Some((unsafe { Key::from_index(Nzu::new_unchecked(i + 1)) }, val));
            }
        }

        for v in &mut self.hash {
            if let Some((ref k, ref mut v)) = *v.value_mut() {
                self.len -= 1;
                return Some((Key::Borrowed(k), v));
            }
        }

        debug_assert_eq!(self.len, 0);

        None
    }

    fn size_hint(&self) -> (usize, Option<usize>) {
        (self.len, Some(self.len))
    }
}

impl<K: FromIndex, V> ExactSizeIterator for IterMut<'_, K, V> {
    fn len(&self) -> usize {
        self.len
    }
}

impl<K: FromIndex, V> FusedIterator for IterMut<'_, K, V> {}

impl<K: FromIndex, V> IntoIterator for Ham<K, V> {
    type Item = (K, V);
    type IntoIter = IntoIter<K, V>;
    fn into_iter(self) -> IntoIter<K, V> {
        let Ham {
            array, hash, len, ..
        } = self;

        IntoIter {
            array: array.into_iter().enumerate(),
            hash: hash.into_iter(),
            len,
        }
    }
}

pub struct IntoIter<K, V> {
    array: Enumerate<vec::IntoIter<Option<V>>>,
    hash: vec::IntoIter<Node<K, V>>,
    len: usize,
}

impl<K: FromIndex, V> Iterator for IntoIter<K, V> {
    type Item = (K, V);

    fn next(&mut self) -> Option<(K, V)> {
        for (i, v) in &mut self.array {
            if let Some(val) = v {
                self.len -= 1;
                return Some((unsafe { K::from_index(Nzu::new_unchecked(i + 1)) }, val));
            }
        }

        for v in &mut self.hash {
            if let Some((k, v)) = v.into_inner() {
                self.len -= 1;
                return Some((k, v));
            }
        }

        debug_assert_eq!(self.len, 0);

        None
    }

    fn size_hint(&self) -> (usize, Option<usize>) {
        (self.len, Some(self.len))
    }
}

impl<K: FromIndex, V> ExactSizeIterator for IntoIter<K, V> {
    fn len(&self) -> usize {
        self.len
    }
}

impl<K: FromIndex, V> FusedIterator for IntoIter<K, V> {}

#[derive(Debug, Clone)]
pub struct Keys<'a, K, V> {
    pub(super) inner: Iter<'a, K, V>,
}

impl<'a, K: FromIndex, V> Iterator for Keys<'a, K, V> {
    type Item = Key<'a, K>;

    fn next(&mut self) -> Option<Key<'a, K>> {
        self.inner.next().map(|(k, _)| k)
    }

    fn size_hint(&self) -> (usize, Option<usize>) {
        self.inner.size_hint()
    }
}

impl<K: FromIndex, V> ExactSizeIterator for Keys<'_, K, V> {
    fn len(&self) -> usize {
        self.inner.len()
    }
}

impl<K: FromIndex, V> FusedIterator for Keys<'_, K, V> {}

#[derive(Debug, Clone)]
pub struct Values<'a, K, V> {
    pub(super) array: slice::Iter<'a, Option<V>>,
    pub(super) hash: slice::Iter<'a, Node<K, V>>,
    pub(super) len: usize,
}

impl<'a, K, V: 'a> Iterator for Values<'a, K, V> {
    type Item = &'a V;

    fn next(&mut self) -> Option<&'a V> {
        for v in &mut self.array {
            if let Some(ref val) = *v {
                self.len -= 1;
                return Some(val);
            }
        }

        for v in &mut self.hash {
            if let Some((_, ref v)) = *v.value() {
                self.len -= 1;
                return Some(v);
            }
        }

        debug_assert_eq!(self.len, 0);

        None
    }

    fn size_hint(&self) -> (usize, Option<usize>) {
        (self.len, Some(self.len))
    }
}

impl<K, V> ExactSizeIterator for Values<'_, K, V> {
    fn len(&self) -> usize {
        self.len
    }
}

impl<K, V> FusedIterator for Values<'_, K, V> {}

/// An iterator over mutable values of a [`Ham`].
#[derive(Debug)]
pub struct ValuesMut<'a, K, V> {
    pub(super) array: slice::IterMut<'a, Option<V>>,
    pub(super) hash: slice::IterMut<'a, Node<K, V>>,
    pub(super) len: usize,
}

impl<'a, K, V: 'a> Iterator for ValuesMut<'a, K, V> {
    type Item = &'a mut V;

    fn next(&mut self) -> Option<&'a mut V> {
        for v in &mut self.array {
            if let Some(ref mut val) = *v {
                self.len -= 1;
                return Some(val);
            }
        }

        for v in &mut self.hash {
            if let Some((_, ref mut v)) = *v.value_mut() {
                self.len -= 1;
                return Some(v);
            }
        }

        debug_assert_eq!(self.len, 0);

        None
    }

    fn size_hint(&self) -> (usize, Option<usize>) {
        (self.len, Some(self.len))
    }
}

impl<K, V> ExactSizeIterator for ValuesMut<'_, K, V> {
    fn len(&self) -> usize {
        self.len
    }
}

impl<K, V> FusedIterator for ValuesMut<'_, K, V> {}
