use core::hash::{BuildHasher, Hasher};

use self::rand::rand_u64;

#[cfg(windows)]
mod rand {
    mod c {
        extern "C" {
            fn rand_s(v: &mut u32) -> i32;
        }

        pub fn call_rand(v: &mut u32) {
            match unsafe { rand_s(v) } {
                0 => {}
                ret => panic!("Error calling `rand_s()` (returned {:x})", ret),
            }
        }
    }

    pub fn rand_u32() -> u32 {
        let mut v = 0;
        c::call_rand(&mut v);
        v
    }

    pub fn rand_u64() -> u64 {
        let v1 = rand_u32() as u64;
        let v2 = rand_u32() as u64;
        (v1 << 32) + v2
    }
}

#[cfg(unix)]
mod rand {
    mod c {
        extern "C" {
            fn rand(v: &mut u32) -> i32;
        }

        pub fn call_rand(v: &mut u32) {
            match unsafe { rand(v) } {
                0 => {}
                ret => panic!("Error calling `rand()` (returned {:x})", ret),
            }
            debug_assert!(
                *v <= RAND_MAX,
                "rand() was {:x}, which was larger than RAND_MAX ({:x})",
                *v,
                RAND_MAX
            );
        }

        // RAND_MAX is (2 ** 31) - 1
        pub const RAND_MAX: u32 = (1 << RAND_BITS) - 1;

        pub const RAND_BITS: u32 = 31;
    }

    pub fn rand_u32() -> u32 {
        let mut v = 0;
        c::call_rand(&mut v);

        // Get the top bit
        let mut v2 = 0;
        c::call_rand(&mut v2);
        v |= v2 << c::RAND_BITS;

        v
    }

    pub fn rand_u64() -> u64 {
        let v1 = rand_u32() as u64;
        let v2 = rand_u32() as u64;
        (v1 << 32) + v2
    }
}

#[cfg(not(any(windows, unix)))]
compile_error!("Not Windows or Unix!");

#[derive(Copy, Clone, Debug, Eq, PartialEq)]
pub(super) struct LuaBuildHasher(u64);

impl LuaBuildHasher {
    #[cfg(feature = "std")]
    pub(super) fn new() -> Self {
        Self(rand_u64())
    }

    pub(super) const fn with_seed(seed: u64) -> Self {
        Self(seed)
    }
}

impl BuildHasher for LuaBuildHasher {
    type Hasher = LuaHasher;

    fn build_hasher(&self) -> LuaHasher {
        LuaHasher::new(self.0)
    }
}

pub(super) struct LuaHasher {
    state: u64,
}

impl LuaHasher {
    const fn new(seed: u64) -> Self {
        Self { state: seed }
    }
}

impl Hasher for LuaHasher {
    fn write(&mut self, bytes: &[u8]) {
        let mut state = self.state;
        for byte in bytes {
            state ^= (state << 5)
                .wrapping_add(state >> 2)
                .wrapping_add((*byte).into());
        }
        self.state = state;
    }

    fn write_u8(&mut self, n: u8) {
        self.write_u64(n.into());
    }

    fn write_u16(&mut self, n: u16) {
        self.write_u64(n.into());
    }

    fn write_u32(&mut self, n: u32) {
        self.write_u64(n.into());
    }

    fn write_u64(&mut self, n: u64) {
        let state = self.state;
        self.state ^= (state << 5).wrapping_add(state >> 2).wrapping_add(n);
    }

    fn write_u128(&mut self, n: u128) {
        self.write_u64((n >> 64) as u64);
        self.write_u64(n as u64);
    }

    fn write_usize(&mut self, n: usize) {
        #[cfg(target_pointer_width = "16")]
        self.write_u16(n as u16);
        #[cfg(target_pointer_width = "32")]
        self.write_u32(n as u32);
        #[cfg(target_pointer_width = "64")]
        self.write_u64(n as u64);
    }

    fn finish(&self) -> u64 {
        self.state
    }
}
