use alloc::{
    borrow::{Cow, ToOwned},
    boxed::Box,
    string::{String, ToString},
};
use core::{
    convert::{TryFrom, TryInto},
    num::{
        NonZeroI128,
        NonZeroI16,
        NonZeroI32,
        NonZeroI64,
        NonZeroI8,
        NonZeroIsize,
        NonZeroU128,
        NonZeroU16,
        NonZeroU32,
        NonZeroU64,
        NonZeroU8,
        NonZeroUsize,
    },
};

/// An attempted conversion into a [`NonZeroUsize`].
///
/// This is used when converting a value into an appropriate index. Therefore,
/// only types that are integer-like should return `Ok` here.
///
/// Example:
///
/// ```rust
/// # use core::convert::{TryFrom, TryInto}; // If not edition-2021
/// use core::num::NonZeroUsize;
///
/// use hash_arr_map::{FromIndex, IntoIndex};
///
/// struct CouldBeAnIndex(i32);
///
/// impl IntoIndex for CouldBeAnIndex {
///     fn into_index(&self) -> Option<NonZeroUsize> {
///         usize::try_from(self.0).ok()?.try_into().ok()
///     }
/// }
///
/// impl FromIndex for CouldBeAnIndex {
///     fn try_from_index(idx: NonZeroUsize) -> Option<Self> {
///         Some(Self(idx.get() as i32))
///     }
/// }
/// ```
pub trait IntoIndex {
    /// Try to convert `self` into a [`NonZeroUsize`].
    ///
    /// ```rust
    /// use core::num::NonZeroUsize;
    ///
    /// use hash_arr_map::IntoIndex;
    ///
    /// let a: i32 = 5;
    ///
    /// assert_eq!(a.into_index(), Some(NonZeroUsize::new(5).unwrap()));
    /// ```
    #[allow(clippy::wrong_self_convention)]
    fn into_index(&self) -> Option<NonZeroUsize>;
}

/// Conversion from an index.
///
/// This trait implements conversion from an index into self.
/// The indices converted are guaranteed to have been returned from
/// [`IntoIndex::into_index`] at some point in the past.
pub trait FromIndex: Sized + IntoIndex {
    /// Convert it.
    ///
    /// # Safety
    ///
    /// The index must have been returned from [`IntoIndex::into_index`] at some
    /// point in the past.
    unsafe fn from_index(idx: NonZeroUsize) -> Self {
        Self::try_from_index(idx).unwrap()
    }

    /// Fallibly convert it.
    fn try_from_index(idx: NonZeroUsize) -> Option<Self>;
}

macro_rules! impl_test_index {
    ($($t:ty,)+) => {
        $(
            impl IntoIndex for $t {
                fn into_index(&self) -> Option<NonZeroUsize> {
                    usize::try_from(*self).ok()?.try_into().ok()
                }
            }

            impl FromIndex for $t {
                unsafe fn from_index(idx: NonZeroUsize) -> Self {
                    idx.get() as $t
                }

                fn try_from_index(idx: NonZeroUsize) -> Option<Self> {
                    Some(idx.get() as $t)
                }
            }
        )+
    }
}

impl_test_index! {
    i8,    u8,
    i16,   u16,
    i32,   u32,
    i64,   u64,
    i128,  u128,
    isize, usize,
}

macro_rules! impl_test_index_nonzero {
    ($($t:ty,)+) => {
        $(
            impl IntoIndex for $t {
                fn into_index(&self) -> Option<NonZeroUsize> {
                    (*self).try_into().ok()
                }
            }

            impl FromIndex for $t {
                unsafe fn from_index(idx: NonZeroUsize) -> Self {
                    unsafe { Self::new_unchecked(idx.get() as _) }
                }

                fn try_from_index(idx: NonZeroUsize) -> Option<Self> {
                    Self::new(idx.get() as _)
                }
            }
        )+
    }
}

impl_test_index_nonzero! {
    NonZeroI8,    NonZeroU8,
    NonZeroI16,   NonZeroU16,
    NonZeroI32,   NonZeroU32,
    NonZeroI64,   NonZeroU64,
    NonZeroI128,  NonZeroU128,
    NonZeroIsize,
}

impl IntoIndex for NonZeroUsize {
    fn into_index(&self) -> Option<NonZeroUsize> {
        Some(*self)
    }
}

impl FromIndex for NonZeroUsize {
    unsafe fn from_index(idx: NonZeroUsize) -> Self {
        idx
    }

    fn try_from_index(idx: NonZeroUsize) -> Option<Self> {
        Some(idx)
    }
}

impl IntoIndex for char {
    fn into_index(&self) -> Option<NonZeroUsize> {
        u32::from(*self).into_index()
    }
}

impl FromIndex for char {
    unsafe fn from_index(idx: NonZeroUsize) -> Self {
        unsafe { char::from_u32_unchecked(idx.get() as u32) }
    }

    fn try_from_index(idx: NonZeroUsize) -> Option<Self> {
        char::from_u32(idx.get().try_into().ok()?)
    }
}

// Note: There is no matching FromIndex.
impl<T: IntoIndex> IntoIndex for &T {
    fn into_index(&self) -> Option<NonZeroUsize> {
        (**self).into_index()
    }
}

impl<T: IntoIndex> IntoIndex for Box<T> {
    fn into_index(&self) -> Option<NonZeroUsize> {
        (**self).into_index()
    }
}

impl<T: FromIndex> FromIndex for Box<T> {
    unsafe fn from_index(idx: NonZeroUsize) -> Self {
        Box::new(unsafe { T::from_index(idx) })
    }

    fn try_from_index(idx: NonZeroUsize) -> Option<Self> {
        Some(Box::new(T::try_from_index(idx)?))
    }
}

impl<T: IntoIndex> IntoIndex for Option<T> {
    fn into_index(&self) -> Option<NonZeroUsize> {
        self.as_ref().and_then(T::into_index)
    }
}

impl<T: FromIndex> FromIndex for Option<T> {
    unsafe fn from_index(idx: NonZeroUsize) -> Self {
        Some(unsafe { T::from_index(idx) })
    }

    fn try_from_index(idx: NonZeroUsize) -> Option<Self> {
        Some(Some(T::try_from_index(idx)?))
    }
}

impl<T: IntoIndex, E> IntoIndex for Result<T, E> {
    fn into_index(&self) -> Option<NonZeroUsize> {
        self.as_ref().ok().into_index()
    }
}

impl<T: FromIndex, E> FromIndex for Result<T, E> {
    unsafe fn from_index(idx: NonZeroUsize) -> Self {
        Ok(unsafe { T::from_index(idx) })
    }

    fn try_from_index(idx: NonZeroUsize) -> Option<Self> {
        Some(Ok(T::try_from_index(idx)?))
    }
}

impl<T: ?Sized + IntoIndex + ToOwned> IntoIndex for Cow<'_, T> {
    fn into_index(&self) -> Option<NonZeroUsize> {
        (**self).into_index()
    }
}

impl<T: ?Sized + IntoIndex + ToOwned> FromIndex for Cow<'static, T>
where
    <T as ToOwned>::Owned: FromIndex,
{
    unsafe fn from_index(idx: NonZeroUsize) -> Self {
        Cow::Owned(unsafe { FromIndex::from_index(idx) })
    }

    fn try_from_index(idx: NonZeroUsize) -> Option<Self> {
        Some(Cow::Owned(FromIndex::try_from_index(idx)?))
    }
}

impl IntoIndex for str {
    fn into_index(&self) -> Option<NonZeroUsize> {
        if self.starts_with('0') {
            None
        } else {
            self.bytes()
                .try_fold(0, |acc, c| -> Option<usize> {
                    acc.checked_mul(10)?
                        .checked_add(matches!(c, b'0'..=b'9').then(|| c - b'0')? as usize)
                })
                .and_then(|v| v.try_into().ok())
        }
    }
}

impl IntoIndex for String {
    fn into_index(&self) -> Option<NonZeroUsize> {
        (**self).into_index()
    }
}

impl FromIndex for String {
    unsafe fn from_index(idx: NonZeroUsize) -> Self {
        idx.to_string()
    }

    fn try_from_index(idx: NonZeroUsize) -> Option<Self> {
        Some(idx.to_string())
    }
}
