use core::{borrow::Borrow, cell::Cell, mem};

#[cfg_attr(feature = "gc", derive(gc::Trace))]
#[derive(Clone, Debug)]
pub(super) struct Node<K, V> {
    pub(super) inner: Option<(K, V)>,

    // Pointer to the next place in the table for chaining
    pub(super) next: isize,
}

impl<K, V> Default for Node<K, V> {
    fn default() -> Self {
        Node {
            inner: None,
            next: 0,
        }
    }
}

impl<K, V> Node<K, V> {
    pub(super) const fn value(&self) -> &Option<(K, V)> {
        &self.inner
    }

    pub(super) const fn value_mut(&mut self) -> &mut Option<(K, V)> {
        &mut self.inner
    }

    pub(super) fn take(&mut self) -> Option<(K, V)> {
        mem::replace(&mut self.inner, None)
    }

    pub(super) const fn into_inner(self) -> Option<(K, V)> {
        let Node { inner, .. } = self;
        inner
    }

    pub(super) fn matches<Q: ?Sized + Eq>(this: &Cell<Self>, key: &Q) -> bool
    where
        K: Borrow<Q>,
    {
        // SAFETY: this is inaccessible from here, even though it calls user code.
        unsafe { &*this.as_ptr() }.matches_ref(key)
    }

    pub(super) fn matches_ref<Q: ?Sized + Eq>(&self, key: &Q) -> bool
    where
        K: Borrow<Q>,
    {
        match self.inner {
            Some((ref k, _)) => key == k.borrow(),
            None => false,
        }
    }

    pub(super) const fn next_ref(&self) -> &Self {
        unsafe { &*(self as *const Self).offset(self.next) }
    }

    pub(super) fn next(this: &Cell<Self>) -> &Cell<Self> {
        let offset = cp!(Self => isize, this => next).get();

        unsafe { &*(this as *const Cell<Self>).offset(offset) }
    }

    pub(super) fn set_next(this: &Cell<Self>, next: &Cell<Self>) {
        let next_num = cp!(Self => isize, this => next);

        next_num.set(unsafe { (next as *const Cell<Self>).offset_from(this) });
    }
}
