use core::{
    fmt::{self, Formatter},
    hash::Hash,
    mem,
};

use super::{node::Node, HashArrMap};
use crate::{FromIndex, IntoIndex};

pub enum Entry<'a, K, V> {
    Occupied(OccupiedEntry<'a, K, V>),
    Vacant(VacantEntry<'a, K, V>),
}

use Entry::{Occupied, Vacant};

impl<K: fmt::Debug, V: fmt::Debug> fmt::Debug for Entry<'_, K, V> {
    fn fmt(&self, f: &mut Formatter<'_>) -> fmt::Result {
        match *self {
            Vacant(ref v) => f.debug_tuple("Entry").field(v).finish(),
            Occupied(ref o) => f.debug_tuple("Entry").field(o).finish(),
        }
    }
}

unsafe fn extend_lifetime<'a, 'b, T>(v: &'a mut T) -> &'b mut T {
    unsafe { &mut *(v as *mut _) }
}

impl<'a, K: IntoIndex + FromIndex + Hash + Eq, V> Entry<'a, K, V> {
    pub(super) fn make_new(ham: &'a mut HashArrMap<K, V>, key: K) -> Self {
        if let Some(v) = ham.get_array_mut(&key) {
            // We will use the array
            // SAFETY: Ham is not used after this, since Self borrows it.
            let v = unsafe { extend_lifetime(v) };
            if v.is_some() {
                Occupied(OccupiedEntry {
                    entry: OccupiedEntryTy::Array(key, v),
                })
            } else {
                Vacant(VacantEntry {
                    key,
                    entry: VacantEntryTy::Array(v),
                })
            }
        } else if let Some(node) = ham.get_hash_mut(&key) {
            // We will use the hash
            // SAFETY: Ham is not used after this, since Self borrows it.
            let node = unsafe { extend_lifetime(node) };
            return if node.value().is_some() {
                Occupied(OccupiedEntry {
                    entry: OccupiedEntryTy::Hash(Some(key), node),
                })
            } else {
                Vacant(VacantEntry {
                    key,
                    entry: VacantEntryTy::Hash(node),
                })
            };
        } else {
            Vacant(VacantEntry {
                key,
                entry: VacantEntryTy::NotThere(ham),
            })
        }
    }

    pub fn or_insert(self, default: V) -> &'a mut V {
        match self {
            Occupied(o) => o.into_mut(),
            Vacant(v) => v.insert(default),
        }
    }

    pub fn or_insert_with<F: FnOnce() -> V>(self, default: F) -> &'a mut V {
        match self {
            Occupied(o) => o.into_mut(),
            Vacant(v) => v.insert(default()),
        }
    }

    pub fn or_insert_with_key<F: FnOnce(&K) -> V>(self, default: F) -> &'a mut V {
        match self {
            Occupied(o) => o.into_mut(),
            Vacant(v) => {
                let value = default(v.key());
                v.insert(value)
            }
        }
    }

    // pub fn insert(self, value: V) -> OccupiedEntry<'a, K, V> {
    //     match self {
    //         Occupied(mut o) => {
    //             o.insert(value);
    //             o
    //         },
    //         Vacant(v) => v.insert_entry(value),
    //     }
    // }
}

impl<'a, K: IntoIndex + FromIndex + Hash + Eq, V: Default> Entry<'a, K, V> {
    pub fn or_default(self) -> &'a mut V {
        match self {
            Occupied(o) => o.into_mut(),
            Vacant(v) => v.insert(Default::default()),
        }
    }
}

impl<'a, K, V> Entry<'a, K, V> {
    pub fn key(&self) -> &K {
        match *self {
            Occupied(ref o) => o.key(),
            Vacant(ref v) => v.key(),
        }
    }

    pub fn and_modify<F: FnOnce(&mut V)>(self, f: F) -> Self {
        match self {
            Occupied(mut o) => {
                f(o.get_mut());
                Occupied(o)
            }
            Vacant(v) => Vacant(v),
        }
    }
}

pub struct OccupiedEntry<'a, K, V> {
    entry: OccupiedEntryTy<'a, K, V>,
}

impl<K: fmt::Debug, V: fmt::Debug> fmt::Debug for OccupiedEntry<'_, K, V> {
    fn fmt(&self, f: &mut Formatter<'_>) -> fmt::Result {
        f.debug_struct("OccupiedEntry")
            .field("key", self.key())
            .field("value", self.get())
            .finish()
    }
}

impl<'a, K, V> OccupiedEntry<'a, K, V> {
    pub fn key(&self) -> &K {
        use OccupiedEntryTy::*;
        match self.entry {
            Array(ref k, _) => k,
            Hash(_, ref node) => &unsafe { node.value().as_ref().unwrap_unchecked() }.0,
        }
    }

    pub fn get(&self) -> &V {
        use OccupiedEntryTy::*;
        match self.entry {
            Array(_, ref arr) => unsafe { arr.as_ref().unwrap_unchecked() },
            Hash(_, ref node) => &unsafe { node.value().as_ref().unwrap_unchecked() }.1,
        }
    }

    pub fn get_mut(&mut self) -> &mut V {
        use OccupiedEntryTy::*;
        match self.entry {
            Array(_, ref mut arr) => unsafe { arr.as_mut().unwrap_unchecked() },
            Hash(_, ref mut node) => &mut unsafe { node.value_mut().as_mut().unwrap_unchecked() }.1,
        }
    }

    pub fn insert(&mut self, value: V) -> V {
        mem::replace(self.get_mut(), value)
    }

    pub fn into_mut(self) -> &'a mut V {
        use OccupiedEntryTy::*;
        match self.entry {
            Array(_, arr) => unsafe { arr.as_mut().unwrap_unchecked() },
            Hash(_, node) => &mut unsafe { node.value_mut().as_mut().unwrap_unchecked() }.1,
        }
    }
}

enum OccupiedEntryTy<'a, K, V> {
    Array(K, &'a mut Option<V>),
    Hash(Option<K>, &'a mut Node<K, V>),
}

pub struct VacantEntry<'a, K, V> {
    key: K,
    entry: VacantEntryTy<'a, K, V>,
}

impl<K: fmt::Debug, V: fmt::Debug> fmt::Debug for VacantEntry<'_, K, V> {
    fn fmt(&self, f: &mut Formatter<'_>) -> fmt::Result {
        f.debug_tuple("VacantEntry").field(self.key()).finish()
    }
}

impl<'a, K, V> VacantEntry<'a, K, V> {
    pub const fn key(&self) -> &K {
        &self.key
    }

    pub const fn into_key(self) -> K {
        self.key
    }
}

impl<'a, K: IntoIndex + FromIndex + Hash + Eq, V> VacantEntry<'a, K, V> {
    pub fn insert(self, value: V) -> &'a mut V {
        use VacantEntryTy::*;
        match self.entry {
            Array(arr) => arr.insert(value),
            Hash(hash) => &mut hash.value_mut().insert((self.key, value)).1,
            NotThere(ham) => ham.new_key(self.key, value),
        }
    }
}

enum VacantEntryTy<'a, K, V> {
    Array(&'a mut Option<V>),
    Hash(&'a mut Node<K, V>),
    NotThere(&'a mut HashArrMap<K, V>),
}
