use std::{collections::{BTreeMap, VecDeque}, sync::{Arc, RwLock}};

use hakuban::{events::Action, message::Message, ObjectDescriptor, ObjectType, ObjectVersion, TagDescriptor};
use serde::Deserialize;
use tokio::sync::{mpsc, oneshot};



#[derive(Debug, Deserialize, Clone)]
#[serde(tag = "type")]
pub enum Block {
	Test(Test),
	Fragment(Test),
}


#[derive(Debug, Deserialize, Clone)]
pub struct Test {
	pub description: String,
	pub disabled: Option<bool>,
	pub steps: Vec<BTreeMap<String, Vec<TestAction>>>,
}

#[derive(Debug, Deserialize, Clone)]
pub struct TagObjectObserveEvent {
	pub tag_descriptor: TagDescriptor,
	pub action: Action,
	pub object_descriptor: ObjectDescriptor,
	pub version: Option<ObjectVersion>,
	pub data: Option<serde_json::Value>,
	pub data_type: Option<ObjectType>,
	pub synchronized: Option<bool>,
}

#[derive(Debug, Deserialize, Clone)]
pub struct TagObjectEvent {
	pub tag_descriptor: TagDescriptor,
	pub action: Action,
	pub object_descriptor: ObjectDescriptor,
}

//TODO: fix inconsistencies and bloat
#[derive(Debug, Deserialize, Clone)]
#[serde(rename_all = "snake_case")]
pub enum TestAction {
	Create {
		actor: String,
		processor: String,
	},
	Run {
		id: String,
		repeat: Option<usize>,
	},
	EnsureRssBelow(usize),
	Sleep(f32),

	//---- network actions
	Connect {
		connect_to: String,
	},
	Disconnect,
	NetPartition,
	NetJoin,
	GetDisconnected,
	Listen {
		diff_produce: Option<bool>,
		diff_request: Option<bool>,
		timeout: f32,
	},

	Send(Vec<Message>),
	SendNothing(f32),
	SendSomeMessages {
		count: usize,
	},

	Receive(Vec<Message>),
	ReceiveNothing(f32),
	ReceiveSomeMessages {
		count: usize,
	},

	//---- contract lifetime actions
	ObjectObserveNew {
		descriptor: ObjectDescriptor,
	},
	ObjectExposeNew {
		descriptor: ObjectDescriptor,
	},
	TagObserveNew {
		descriptor: TagDescriptor,
	},
	TagExposeNew {
		descriptor: TagDescriptor,
	},
	ObjectObserveDrop {
		descriptor: ObjectDescriptor,
	},
	ObjectExposeDrop {
		descriptor: ObjectDescriptor,
	},
	TagObserveDrop {
		descriptor: TagDescriptor,
	},
	TagExposeDrop {
		descriptor: TagDescriptor,
	},

	//---- events
	//TODO: vec of events, like it's done for tags
	ObjectObserveEvent {
		action: Action,
		descriptor: ObjectDescriptor,
		version: Option<ObjectVersion>,
		data: Option<serde_json::Value>,
		data_type: Option<ObjectType>,
		synchronized: Option<u64>,
		last_synchronized_ago: Option<u64>,
	},
	ObjectExposeEvent {
		action: Action,
		descriptor: ObjectDescriptor,
		assignment: Option<u64>,
	},
	TagObjectObserveEvent(Vec<TagObjectObserveEvent>),
	TagObjectExposeEvent(Vec<TagObjectEvent>),

	//---- object expose actions
	ObjectExposeNewVersion {
		descriptor: ObjectDescriptor,
		data_version: Option<ObjectVersion>,
		data: serde_json::Value,
		data_type: ObjectType,
		last_synchronized_ago: Option<u64>,
	},
	ObjectExposeData {
		descriptor: ObjectDescriptor,
		data: serde_json::Value,
	},
	ObjectExposeAssigned {
		descriptor: ObjectDescriptor,
		assignment_id: Option<u64>,
	},
	ObjectVersionExposed {
		descriptor: ObjectDescriptor,
		version: Option<ObjectVersion>,
		#[serde(with = "serde_bytes")]
		data: Vec<u8>,
		data_type: ObjectType,
	},

	//---- tag expose actions
	TagObjectExposeNewVersion {
		tag: TagDescriptor,
		descriptor: ObjectDescriptor,
		data_version: ObjectVersion,
		data: serde_json::Value,
		data_type: ObjectType,
	},
	TagExposeData {
		tag_descriptor: TagDescriptor,
		object_descriptor: ObjectDescriptor,
		data: serde_json::Value,
	},
	TagExposeVersion {
		tag_descriptor: TagDescriptor,
		object_descriptor: ObjectDescriptor,
		data_version: Option<ObjectVersion>,
	},

	//---- object observe actions
	ObjectObserveData {
		descriptor: ObjectDescriptor,
		data: serde_json::Value,
	},
	ObjectObserveVersion {
		descriptor: ObjectDescriptor,
		data_version: ObjectVersion,
	},

	//---- tag observe actions
	TagObserveData {
		tag_descriptor: TagDescriptor,
		object_descriptor: ObjectDescriptor,
		data: serde_json::Value,
	},
	TagObserveVersion {
		tag_descriptor: TagDescriptor,
		object_descriptor: ObjectDescriptor,
		data_version: ObjectVersion,
	},
}


#[derive(Debug)]
pub enum Order {
	Event(Arc<StackFrame<RunFrame>>, TestAction, oneshot::Sender<Result<(), String>>),
	Exit,
}

pub struct Scene {
	pub actors: RwLock<BTreeMap<String, mpsc::Sender<Order>>>,
	pub ports: RwLock<BTreeMap<String, u16>>,
	pub scripts: BTreeMap<String, Block>,
	pub interstep_delay: f32,
}


#[derive(Debug, Clone)]
pub struct StackFrame<T: Clone> {
	pub top: T,
	pub rest: Option<Arc<StackFrame<T>>>,
}

impl<T: Clone> IntoIterator for StackFrame<T> {
	type IntoIter = <std::vec::Vec<T> as IntoIterator>::IntoIter;
	type Item = T;

	fn into_iter(self) -> Self::IntoIter {
		let mut v = vec![];
		let step = self;
		while let Some(ref previous) = step.rest {
			v.push(previous.top.clone());
		}
		v.into_iter()
	}
}


impl<T: Clone> StackFrame<T> {
	fn iter(&self) -> std::collections::vec_deque::IntoIter<T> {
		let mut step = self;
		let mut v = VecDeque::new();
		loop {
			v.push_front(step.top.clone());
			if let Some(ref previous) = step.rest {
				step = previous;
			} else {
				break;
			};
		}
		v.into_iter()
	}
}



#[derive(Debug, Clone)]
pub struct RunFrame {
	pub fragment_id: String,
	pub step_number: usize,
	pub repetition: usize,
}


impl StackFrame<RunFrame> {
	pub fn path(&self) -> String {
		self.iter()
			.skip(1)
			.map(|frame| {
				format!("{}:{}{}", frame.fragment_id, frame.step_number, if frame.repetition == 0 { "".to_string() } else { format!("#{}", frame.repetition) })
			})
			.collect::<Vec<String>>()
			.join(" / ")
	}
}
