use std::{collections::{BTreeMap, HashMap}, path::PathBuf};

extern crate hakuban;
use std::sync::{Arc, RwLock};

use client::{MockClient, RealClient};
use director::{TestDirector, TypeMap};
use process::ServerProcess;
use tokio::sync::oneshot;

mod process;
mod scene;
use scene::*;
mod client;
mod director;


async fn run_scripts(path: PathBuf, types: TypeMap) {
	let mut scripts: BTreeMap<String, Block> = BTreeMap::new();
	let mut candidate_scripts: Vec<String> = vec![];
	for filename in glob::glob("tests/*.yaml").unwrap() {
		let mut more_scripts: BTreeMap<String, Block> = serde_yaml::from_str(std::fs::read_to_string(filename.as_ref().unwrap()).unwrap().as_str()).unwrap();
		if path == filename.unwrap() {
			candidate_scripts = more_scripts.keys().cloned().collect();
		}
		scripts.append(&mut more_scripts);
	}
	let patterns: Vec<String> = std::env::var("TEST_SEQUENCE")
		.unwrap_or_else(|_| "".to_string())
		.split(',')
		.map(|pattern| pattern.to_string())
		.filter(|pattern| !pattern.is_empty())
		.collect();
	let interstep_delay: f32 = std::env::var("DELAY").unwrap_or_else(|_| "0.0".to_string()).parse().unwrap();
	for block_id in candidate_scripts {
		if let Block::Test(test) = scripts.get(&block_id).unwrap() {
			if !test.disabled.unwrap_or(false) && (patterns.is_empty() || patterns.iter().any(|pattern| block_id.contains(pattern))) {
				println!("---------------------------------------------------------------------------------------------------------------------------------");
				println!("ID: {}", block_id);
				//println!("Disabled: {:?}", test.disabled.unwrap_or(false));
				println!("Description: {}", test.description);
				println!();
				let scene =
					Arc::new(Scene { actors: RwLock::new(BTreeMap::new()), ports: RwLock::new(BTreeMap::new()), scripts: scripts.clone(), interstep_delay });
				let (director_tx, director_join_handle) = TestDirector::spawn(scene.clone(), "director".to_string(), types.clone());
				scene.actors.write().unwrap().insert("director".to_string(), director_tx.clone());
				let (done_tx, done_rx) = oneshot::channel();
				let first_frame = Arc::new(StackFrame { rest: None, top: RunFrame { fragment_id: "/".to_string(), step_number: 1, repetition: 1 } });
				director_tx.send(Order::Event(first_frame, TestAction::Run { id: block_id.clone(), repeat: None }, done_tx)).await.unwrap();
				done_rx.await.unwrap().unwrap();
				director_tx.send(Order::Exit).await.unwrap();
				director_join_handle.await.unwrap();
				println!();
			};
		};
	}
}


#[tokio::test]
async fn router_common() {
	let _ = tracing_subscriber::fmt::try_init();

	let mut types: TypeMap = HashMap::new();
	types.insert("client", MockClient::spawn);
	types.insert("router", ServerProcess::spawn);
	run_scripts("tests/common.yaml".into(), types).await;
}

#[tokio::test]
async fn client_common() {
	let _ = tracing_subscriber::fmt::try_init();

	let mut types: TypeMap = HashMap::new();
	types.insert("client", RealClient::spawn);
	types.insert("router", ServerProcess::spawn);
	run_scripts("tests/common.yaml".into(), types).await;
}

#[tokio::test]
async fn client_only() {
	let _ = tracing_subscriber::fmt::try_init();

	let mut types: TypeMap = HashMap::new();
	types.insert("client", RealClient::spawn);
	types.insert("router", ServerProcess::spawn);
	run_scripts("tests/client-only.yaml".into(), types).await;
}

#[tokio::test]
async fn router_only() {
	let _ = tracing_subscriber::fmt::try_init();

	let mut types: TypeMap = HashMap::new();
	types.insert("client", MockClient::spawn);
	types.insert("router", ServerProcess::spawn);
	run_scripts("tests/router-only.yaml".into(), types).await;
}
