#![allow(clippy::not_unsafe_ptr_arg_deref)]
use std::{mem::forget, sync::Arc};

use log::error;
use wasm_bindgen::prelude::*;

use super::{raw_deserialize, WasmError};
use crate::{events::EventStream, object::{core::ObjectRawData, ObjectObserveState}, LocalNode, ObjectDescriptor, TagDescriptor, TagObserve};


#[wasm_bindgen]
pub struct WasmTagObserveResult {
	pub error: WasmError,
	pub tag_observe_pointer: *mut TagObserve<ObjectRawData, ()>,
}


#[wasm_bindgen]
pub fn hakuban_tag_observe_new(local_node: *mut LocalNode, descriptor: String) -> WasmTagObserveResult {
	let descriptor: TagDescriptor = match serde_json::from_str::<TagDescriptor>(&descriptor) {
		Ok(descriptor) => descriptor,
		Err(error) => {
			error!("Invalid object descriptor json: {:?}", error);
			return WasmTagObserveResult { error: WasmError::InvalidJSON, tag_observe_pointer: std::ptr::null_mut() };
		}
	};
	let local_node = unsafe { Box::from_raw(local_node) };
	let tag_observe = local_node.tag(descriptor).with_deserializer(raw_deserialize).observe();
	let tag_observe_pointer = Box::into_raw(Box::new(tag_observe));
	forget(local_node);
	WasmTagObserveResult { error: WasmError::None, tag_observe_pointer }
}


#[wasm_bindgen]
pub fn hakuban_tag_observe_drop(tag_observe_pointer: *mut TagObserve<ObjectRawData, ()>) {
	drop(unsafe { Box::from_raw(tag_observe_pointer) });
}



#[wasm_bindgen]
pub struct WasmTagObserveObjectStateBorrowResult {
	pub error: WasmError,
	pub state_pointer: *mut ObjectObserveState<Arc<Vec<u8>>>,
}


#[wasm_bindgen]
pub fn hakuban_tag_observe_object_state_borrow(
	tag_observe_pointer: *mut TagObserve<ObjectRawData, ()>, object_descriptor: String,
) -> WasmTagObserveObjectStateBorrowResult {
	let object_descriptor: ObjectDescriptor = match serde_json::from_str::<ObjectDescriptor>(&object_descriptor) {
		Ok(descriptor) => descriptor,
		Err(error) => {
			error!("Invalid object descriptor json: {:?}", error);
			return WasmTagObserveObjectStateBorrowResult { error: WasmError::InvalidJSON, state_pointer: std::ptr::null_mut() };
		}
	};
	let tag_observe = unsafe { Box::from_raw(tag_observe_pointer) };
	let ret = match tag_observe.object_state(&object_descriptor).unwrap() {
		Some(state) => WasmTagObserveObjectStateBorrowResult { error: WasmError::None, state_pointer: Box::into_raw(Box::new(state)) },
		None => WasmTagObserveObjectStateBorrowResult { error: WasmError::ObjectNotFound, state_pointer: std::ptr::null_mut() },
	};
	forget(tag_observe);
	ret
}


#[wasm_bindgen]
pub fn hakuban_tag_observe_object_descriptors(tag_observe_pointer: *mut TagObserve<ObjectRawData, ()>) -> String {
	let tag_observe = unsafe { Box::from_raw(tag_observe_pointer) };
	let descriptors: Vec<ObjectDescriptor> = tag_observe.object_descriptors().into_iter().collect();
	forget(tag_observe);
	serde_json::to_string(&descriptors).unwrap()
}


#[wasm_bindgen]
pub fn hakuban_tag_observe_events_get(tag_observe_ptr: *mut TagObserve<ObjectRawData, ()>) -> *mut EventStream<ObjectDescriptor> {
	let tag_observe = unsafe { Box::from_raw(tag_observe_ptr) };
	let ret = Box::new(tag_observe.changes().into());
	forget(tag_observe);
	Box::into_raw(ret)
}
