#![allow(clippy::not_unsafe_ptr_arg_deref)]
use std::mem::forget;

use log::error;
use wasm_bindgen::prelude::*;

use super::{raw_serialize, WasmError};
use crate::{events::EventStream, LocalNode, ObjectDescriptor, TagDescriptor, TagExpose};



#[wasm_bindgen]
pub struct WasmTagExposeResult {
	pub error: WasmError,
	pub tag_expose_pointer: *mut TagExpose<(Vec<String>, Vec<u8>), ()>,
}


#[wasm_bindgen]
pub fn hakuban_tag_expose_new(local_node: *mut LocalNode, descriptor: String) -> WasmTagExposeResult {
	let descriptor: TagDescriptor = match serde_json::from_str::<TagDescriptor>(&descriptor) {
		Ok(descriptor) => descriptor,
		Err(error) => {
			error!("Invalid object descriptor json: {:?}", error);
			return WasmTagExposeResult { error: WasmError::InvalidJSON, tag_expose_pointer: std::ptr::null_mut() };
		}
	};
	let local_node = unsafe { Box::from_raw(local_node) };
	let tag_expose = local_node.tag(descriptor).with_serializer(raw_serialize).expose();
	let tag_expose_pointer = Box::into_raw(Box::new(tag_expose));
	forget(local_node);
	WasmTagExposeResult { error: WasmError::None, tag_expose_pointer }
}

#[wasm_bindgen]
pub fn hakuban_tag_expose_drop(tag_expose_pointer: *mut TagExpose<(Vec<String>, Vec<u8>), ()>) {
	drop(unsafe { Box::from_raw(tag_expose_pointer) });
}


#[wasm_bindgen]
pub fn hakuban_tag_expose_object_descriptors(tag_expose_pointer: *mut TagExpose<(Vec<String>, Vec<u8>), ()>) -> String {
	let tag_expose = unsafe { Box::from_raw(tag_expose_pointer) };
	let descriptors: Vec<ObjectDescriptor> = tag_expose.object_descriptors().into_iter().collect();
	forget(tag_expose);
	serde_json::to_string(&descriptors).unwrap()
}


#[wasm_bindgen]
pub struct WasmTagExposeSetObjectStateResult {
	pub error: WasmError,
	pub changed: bool,
}

#[wasm_bindgen]
pub fn hakuban_tag_expose_set_object_state(
	tag_expose_pointer: *mut TagExpose<(Vec<String>, Vec<u8>), ()>, object_descriptor: String, data_version: String, data_type: String, data: Vec<u8>,
	assignment_id: u64,
) -> WasmTagExposeSetObjectStateResult {
	let object_descriptor: ObjectDescriptor = match serde_json::from_str::<ObjectDescriptor>(&object_descriptor) {
		Ok(descriptor) => descriptor,
		Err(error) => {
			error!("Invalid object descriptor json: {:?}", error);
			return WasmTagExposeSetObjectStateResult { error: WasmError::InvalidJSON, changed: false };
		}
	};
	let data_version = match serde_json::from_str(&data_version) {
		Ok(state) => state,
		Err(error) => {
			error!("Can't parse data_version string as JSON: {:?}", error);
			return WasmTagExposeSetObjectStateResult { error: WasmError::InvalidJSON, changed: false };
		}
	};
	let data_type = match serde_json::from_str(&data_type) {
		Ok(state) => state,
		Err(error) => {
			error!("Can't parse data_type string as JSON: {:?}", error);
			return WasmTagExposeSetObjectStateResult { error: WasmError::InvalidJSON, changed: false };
		}
	};
	let cooked = (data_type, data);
	let tag_expose = unsafe { Box::from_raw(tag_expose_pointer) };
	let changed = tag_expose.set_object_state(&object_descriptor, &data_version, &cooked, assignment_id).unwrap();
	forget(tag_expose);
	forget(cooked.1);
	forget(data_version);
	WasmTagExposeSetObjectStateResult { error: WasmError::None, changed }
}


#[wasm_bindgen]
pub struct WasmTagExposeAssignmentResult {
	pub error: WasmError,
	pub assignment: u64,
}


#[wasm_bindgen]
pub fn hakuban_tag_expose_assignment(
	tag_expose_pointer: *mut TagExpose<(Vec<String>, Vec<u8>), ()>, object_descriptor: String,
) -> WasmTagExposeAssignmentResult {
	let object_descriptor: ObjectDescriptor = match serde_json::from_str::<ObjectDescriptor>(&object_descriptor) {
		Ok(descriptor) => descriptor,
		Err(error) => {
			error!("Invalid object descriptor json: {:?}", error);
			return WasmTagExposeAssignmentResult { error: WasmError::InvalidJSON, assignment: 0 };
		}
	};
	let tag_expose = unsafe { Box::from_raw(tag_expose_pointer) };
	let assignment = if let Some(assignment) = tag_expose.object_assignment(&object_descriptor) { assignment } else { 0 };
	forget(tag_expose);
	WasmTagExposeAssignmentResult { error: WasmError::None, assignment }
}


#[wasm_bindgen]
pub fn hakuban_tag_expose_desynchronize(
	tag_expose_pointer: *mut TagExpose<(Vec<String>, Vec<u8>), ()>, object_descriptor: String, assignment_id: u64,
) -> WasmError {
	let object_descriptor: ObjectDescriptor = match serde_json::from_str::<ObjectDescriptor>(&object_descriptor) {
		Ok(descriptor) => descriptor,
		Err(error) => {
			error!("Invalid object descriptor json: {:?}", error);
			return WasmError::InvalidJSON;
		}
	};
	let tag_expose = unsafe { Box::from_raw(tag_expose_pointer) };
	tag_expose.object_desynchronize(&object_descriptor, assignment_id);
	forget(tag_expose);
	WasmError::None
}


#[wasm_bindgen]
pub fn hakuban_tag_expose_events_get(tag_expose_ptr: *mut TagExpose<(Vec<String>, Vec<u8>), ()>) -> *mut EventStream<ObjectDescriptor> {
	let tag_expose = unsafe { Box::from_raw(tag_expose_ptr) };
	let ret = Box::new(tag_expose.changes().into());
	forget(tag_expose);
	Box::into_raw(ret)
}
