#![allow(clippy::not_unsafe_ptr_arg_deref)]

use std::{mem::forget, ptr, sync::Arc};

use futures::StreamExt;
use log::error;
use wasm_bindgen::prelude::*;
use wasm_bindgen_futures::JsFuture;

use super::WasmError;
use crate::{events::EventStream, object::{core::{ObjectRawData, ObjectType}, observe::ObjectObserveState}, LocalNode, ObjectDescriptor, ObjectObserve};

#[allow(clippy::ptr_arg)] //it's less powerful but more readable
pub(super) fn raw_deserialize(_data_type: &ObjectType, data: &ObjectRawData) -> Result<ObjectRawData, ()> {
	Ok(data.clone())
}


#[wasm_bindgen]
pub struct WasmObjectObserveResult {
	pub error: WasmError,
	pub object_observe_pointer: *mut ObjectObserve<ObjectRawData, ()>,
}


#[wasm_bindgen]
pub fn hakuban_object_observe_new(local_node: *mut LocalNode, descriptor: String) -> WasmObjectObserveResult {
	let descriptor: ObjectDescriptor = match serde_json::from_str::<ObjectDescriptor>(&descriptor) {
		Ok(descriptor) => descriptor,
		Err(error) => {
			error!("Invalid object descriptor json: {:?}", error);
			return WasmObjectObserveResult { error: WasmError::InvalidJSON, object_observe_pointer: std::ptr::null_mut() };
		}
	};
	let local_node = unsafe { Box::from_raw(local_node) };
	let object_observe = local_node.object(descriptor).with_deserializer(raw_deserialize).observe();
	let object_observe_pointer = Box::into_raw(Box::new(object_observe));
	forget(local_node);
	WasmObjectObserveResult { error: WasmError::None, object_observe_pointer }
}


#[wasm_bindgen]
pub fn hakuban_object_observe_drop(object_ptr: *mut ObjectObserve<ObjectRawData, ()>) {
	drop(unsafe { Box::from_raw(object_ptr) });
}


#[wasm_bindgen]
pub fn hakuban_object_observe_state_borrow(object_observe_pointer: *mut ObjectObserve<ObjectRawData, ()>) -> *const ObjectObserveState<Arc<Vec<u8>>> {
	let object_observe = unsafe { Box::from_raw(object_observe_pointer) };
	let ret = if let Some(state) = object_observe.object_state().unwrap() { Box::into_raw(Box::new(state)) } else { ptr::null() };
	forget(object_observe);
	ret
}

#[wasm_bindgen]
pub fn hakuban_object_observe_state_return(state_ptr: *mut ObjectObserveState<Arc<Vec<u8>>>) {
	let state = unsafe { Box::from_raw(state_ptr) };
	drop(state);
}

#[wasm_bindgen]
pub fn hakuban_object_observe_state_synchronized(state_ptr: *mut ObjectObserveState<Arc<Vec<u8>>>) -> u64 {
	let state = unsafe { Box::from_raw(state_ptr) };
	let ret = state.synchronized;
	forget(state);
	ret
}

#[wasm_bindgen]
pub fn hakuban_object_observe_state_data_version(state_ptr: *mut ObjectObserveState<Arc<Vec<u8>>>) -> String {
	let state = unsafe { Box::from_raw(state_ptr) };
	let ret = serde_json::to_string(&state.data_version).unwrap();
	forget(state);
	ret
}

#[wasm_bindgen]
pub fn hakuban_object_observe_state_data_type(state_ptr: *mut ObjectObserveState<Arc<Vec<u8>>>) -> String {
	let state = unsafe { Box::from_raw(state_ptr) };
	let ret = serde_json::to_string(&state.data_type).unwrap();
	forget(state);
	ret
}

#[wasm_bindgen]
pub fn hakuban_object_observe_state_data(state_ptr: *mut ObjectObserveState<Arc<Vec<u8>>>) -> Vec<u8> {
	let state = unsafe { Box::from_raw(state_ptr) };
	if state.data.len() > u32::MAX as usize {
		error!("Can't handle over u32::MAX of data");
		panic!();
	}
	let ret = (*state.data).clone();
	forget(state);
	ret
}


#[wasm_bindgen]
pub fn hakuban_object_observe_events_get(object_observe_ptr: *mut ObjectObserve<ObjectRawData, ()>) -> *mut EventStream<ObjectDescriptor> {
	let object_observe = unsafe { Box::from_raw(object_observe_ptr) };
	let ret = Box::new(object_observe.changes().into());
	forget(object_observe);
	Box::into_raw(ret)
}

#[wasm_bindgen]
pub fn hakuban_object_descriptor_events_drop(events_pointer: *mut EventStream<ObjectDescriptor>) {
	drop(unsafe { Box::from_raw(events_pointer) });
}

// I hate this. It's way too complicated, and it pulls in futures support.
#[wasm_bindgen]
pub async fn hakuban_object_descriptor_events_next(events_pointer: *mut EventStream<ObjectDescriptor>, cancel: js_sys::Promise) -> JsValue {
	let cancel: JsFuture = cancel.into();
	let mut events = unsafe { Box::from_raw(events_pointer) };
	let ret = match futures::future::select(cancel, events.next()).await {
		futures::future::Either::Left((Ok(cancel), _)) => cancel,
		futures::future::Either::Left((Err(error), _)) => {
			error!("Error in cancel promise: {:?}", error);
			JsValue::NULL
		}
		futures::future::Either::Right((event, _)) => event.map(|event| serde_json::to_string(&event).unwrap()).into(),
	};
	forget(events);
	ret
}


/*
// Having callbacks instead of promises could lead to smaller wasm, but js_sys::Function in not Send, so we'd have to support non-Send event pipeline first


#[wasm_bindgen]
pub extern "C" fn hakuban_object_observe_events_get(
	object_observe_ptr: *mut ObjectObserve<ObjectRawData, ()>,
) -> *mut Box<dyn CallbackRegistry<Params = Event<ObjectDescriptor>>> {
	let object_observe = unsafe { Box::from_raw(object_observe_ptr) };
	let ret = Box::new(Box::new(object_observe.changes()) as Box<dyn CallbackRegistry<Params = Event<ObjectDescriptor>>>);
	forget(object_observe);
	Box::into_raw(ret)
}

pub(super) fn make_object_descriptor_callback(callback: js_sys::Function) -> Box<impl Fn(Event<ObjectDescriptor>)> {
	Box::new(move |event: Event<ObjectDescriptor>| {
		let string_descriptor: JsValue = (&event.key).json_to_string().into();
		let string_action: JsValue = match event.action {
	crate::events::Action::Insert => "insert",
	crate::events::Action::Change => "change",
	crate::events::Action::Remove => "remove",
}.into();
		callback.call2(&JsValue::null(), &string_action, &string_descriptor);
	})
}

#[no_mangle]
pub extern "C" fn hakuban_object_descriptor_events_callback_register(
	events_ptr: *mut Box<dyn CallbackRegistry<Params = Event<ObjectDescriptor>>>, callback: js_sys::Function
) -> *mut Box<dyn Callback<Event<ObjectDescriptor>>> {
	let events = unsafe { Box::from_raw(events_ptr) };
	let ret = Box::new(events.register(make_object_descriptor_callback(callback))); //not Send nor Sync :(
	forget(events);
	Box::into_raw(ret)
}
*/
