#![allow(clippy::not_unsafe_ptr_arg_deref)]
use std::{mem::forget, sync::Arc};

use log::error;
use wasm_bindgen::prelude::*;

use super::WasmError;
use crate::{events::EventStream, object::core::{ObjectRawData, ObjectType}, LocalNode, ObjectDescriptor, ObjectExpose};


pub(super) fn raw_serialize(cooked: &(ObjectType, Vec<u8>)) -> Result<(ObjectType, ObjectRawData), ()> {
	Ok((cooked.0.clone(), Arc::new(cooked.1.clone())))
}



#[wasm_bindgen]
pub struct WasmObjectExposeResult {
	pub error: WasmError,
	pub object_expose_pointer: *mut ObjectExpose<(Vec<String>, Vec<u8>), ()>,
}


#[wasm_bindgen]
pub fn hakuban_object_expose_new(local_node: *mut LocalNode, descriptor: String) -> WasmObjectExposeResult {
	let descriptor: ObjectDescriptor = match serde_json::from_str::<ObjectDescriptor>(&descriptor) {
		Ok(descriptor) => descriptor,
		Err(error) => {
			error!("Invalid object descriptor json: {:?}", error);
			return WasmObjectExposeResult { error: WasmError::InvalidJSON, object_expose_pointer: std::ptr::null_mut() };
		}
	};
	let local_node = unsafe { Box::from_raw(local_node) };
	let object_expose = local_node.object(descriptor).with_serializer(raw_serialize).expose();
	let object_expose_pointer = Box::into_raw(Box::new(object_expose));
	forget(local_node);
	WasmObjectExposeResult { error: WasmError::None, object_expose_pointer }
}


#[wasm_bindgen]
pub fn hakuban_object_expose_drop(object_expose_pointer: *mut ObjectExpose<(Vec<String>, Vec<u8>), ()>) {
	drop(unsafe { Box::from_raw(object_expose_pointer) });
}


#[wasm_bindgen]
pub struct WasmObjectExposeStateResult {
	pub error: WasmError,
	pub changed: bool,
}


#[wasm_bindgen]
pub fn hakuban_object_expose_set_state(
	object_expose_pointer: *mut ObjectExpose<(Vec<String>, Vec<u8>), ()>, data_version: String, data_type: String, data: Vec<u8>, assignment_id: u64,
) -> WasmObjectExposeStateResult {
	let data_version = match serde_json::from_str(&data_version) {
		Ok(state) => state,
		Err(error) => {
			error!("Can't parse data_version string as JSON: {:?}", error);
			return WasmObjectExposeStateResult { error: WasmError::InvalidJSON, changed: false };
		}
	};
	let data_type = match serde_json::from_str(&data_type) {
		Ok(state) => state,
		Err(error) => {
			error!("Can't parse data_type string as JSON: {:?}", error);
			return WasmObjectExposeStateResult { error: WasmError::InvalidJSON, changed: false };
		}
	};
	let cooked = (data_type, data);
	let object_expose = unsafe { Box::from_raw(object_expose_pointer) };
	let changed = object_expose.set_object_state(&data_version, &cooked, assignment_id).unwrap();
	forget(object_expose);
	forget(cooked.1);
	forget(data_version);
	WasmObjectExposeStateResult { error: WasmError::None, changed }
}


#[wasm_bindgen]
pub fn hakuban_object_expose_assignment(object_expose_pointer: *mut ObjectExpose<(Vec<String>, Vec<u8>), ()>) -> u64 {
	let object_expose = unsafe { Box::from_raw(object_expose_pointer) };
	let ret = if let Some(assignment) = object_expose.assignment() { assignment } else { 0 };
	forget(object_expose);
	ret
}


#[wasm_bindgen]
pub fn hakuban_object_expose_desynchronize(object_expose_pointer: *mut ObjectExpose<(Vec<String>, Vec<u8>), ()>, assignment_id: u64) {
	let object_expose = unsafe { Box::from_raw(object_expose_pointer) };
	object_expose.desynchronize(assignment_id);
	forget(object_expose);
}


#[wasm_bindgen]
pub fn hakuban_object_expose_events_get(object_expose_ptr: *mut ObjectExpose<ObjectRawData, ()>) -> *mut EventStream<ObjectDescriptor> {
	let object_expose = unsafe { Box::from_raw(object_expose_ptr) };
	let ret = Box::new(object_expose.changes().into());
	forget(object_expose);
	Box::into_raw(ret)
}
