/*! Provides Websocket transport running in Tokio runtime
*/

use std::{cmp::min, collections::HashMap, convert::TryInto, sync::{Arc, Mutex, RwLock}, time::Duration};

use async_tungstenite::{tokio::connect_async, tungstenite};
use futures::{stream::{SplitSink, SplitStream}, SinkExt, StreamExt};
use instant::Instant;
use log::{error, info, trace};
use tokio::{net::TcpListener, time::{sleep, timeout}};
use url::Url;

use crate::{events::EventStream, message::Message, node::remote::RemoteNode, Descriptor, LocalNode};


// tiny state machine engine
struct State<T: Clone> {
	state: RwLock<T>,
	notify: tokio::sync::Notify,
}

#[derive(Debug)]
enum StateError<T> {
	UnexpectedState(T),
}


impl<T: std::fmt::Debug> std::fmt::Display for StateError<T> {
	fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
		match self {
			StateError::UnexpectedState(_) => write!(f, "State error {:?}", self),
		}
	}
}

impl<T: std::fmt::Debug> std::error::Error for StateError<T> {}


impl<T: Clone + Eq> State<T> {
	fn new(initial: T) -> State<T> {
		State { state: RwLock::new(initial), notify: tokio::sync::Notify::new() }
	}

	fn change<F: FnOnce(T) -> T>(&self, closure: F) -> (T, T) {
		let mut state = self.state.write().unwrap();
		let old_state = state.clone();
		*state = (closure)(old_state.clone());
		self.notify.notify_waiters();
		(old_state, state.clone())
	}

	fn change_from_to(&self, from: T, to: T) -> Result<(), StateError<T>> {
		let mut state = self.state.write().unwrap();
		let old_state = state.clone();
		if old_state == from {
			*state = to;
			self.notify.notify_waiters();
			Ok(())
		} else {
			Err(StateError::UnexpectedState(old_state))
		}
	}

	fn notify(&self) -> &tokio::sync::Notify {
		&self.notify
	}

	fn current(&self) -> T {
		self.state.read().unwrap().clone()
	}
}

#[derive(Clone)]
struct ConnectionParameters {
	timeout: Duration,
	diff_request: Option<bool>,
	diff_produce: Option<bool>,
}

impl ConnectionParameters {
	fn from_url(url: &Url) -> Result<ConnectionParameters, String> {
		let mut timeout = 20.0;
		let mut diff_request = None;
		let mut diff_produce = None;
		if let Some(fragment) = url.fragment() {
			//TODO: split_once
			for key_val in fragment.split(',').map(|key_value| key_value.splitn(2, '=').collect::<Vec<&str>>()).filter(|key_value| key_value.len() == 2) {
				match key_val[0] {
					"timeout" => timeout = key_val[1].parse::<f32>().expect("can't parse 'timeout' parameter's value"),
					"diff-request" =>
						diff_request = Some(match key_val[1] {
							"true" => true,
							"false" => false,
							_ => return Err("'diff-request' parameter can only be set to 'true' or 'false'".to_string()),
						}),
					"diff-produce" =>
						diff_produce = Some(match key_val[1] {
							"true" => true,
							"false" => false,
							_ => return Err("'diff-request' parameter can only be set to 'true' or 'false'".to_string()),
						}),
					other => return Err(format!("unknown parameter ('{}') passed in url fragment", other)),
				}
			}
		};
		Ok(ConnectionParameters { timeout: Duration::from_secs_f32(timeout), diff_request, diff_produce })
	}
}



#[derive(Debug)]
pub enum AddressError<E: std::fmt::Debug> {
	UrlConversionError(E),
	ConnectionParametersParsingError(String),
}

impl<E: std::fmt::Debug> std::fmt::Display for AddressError<E> {
	fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
		write!(f, "{:?}", self)
	}
}

impl<E: std::fmt::Debug> std::error::Error for AddressError<E> {}



pub struct WebsocketListener {
	shared: Arc<WebsocketListenerShared>,
}

struct WebsocketListenerShared {
	local_node: LocalNode,
	state: State<WebsocketListenerState>,
	url: Url,
	parameters: ConnectionParameters,
	connections: Mutex<HashMap<Url, Arc<WebsocketConnection<WebsocketStreamFromAcceptAsync>>>>,
}

#[derive(Eq, PartialEq, Clone, Debug)]
enum WebsocketListenerState {
	Listening,
	Stop,
	Stopped,
}

#[derive(Debug)]
pub enum WebsocketListenerError<E: std::fmt::Debug> {
	UrlConversionError(E),
	ConnectionParametersParsingError(String),
	BindError(std::io::Error),
}

impl<E: std::fmt::Debug> std::fmt::Display for WebsocketListenerError<E> {
	fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
		write!(f, "{:?}", self)
	}
}

impl<E: std::fmt::Debug> std::error::Error for WebsocketListenerError<E> {}


impl WebsocketListener {
	pub async fn new<T: TryInto<Url> + std::fmt::Debug>(local_node: LocalNode, address: T) -> Result<WebsocketListener, WebsocketListenerError<T::Error>>
	where
		T::Error: std::fmt::Debug,
	{
		Ok(WebsocketListener { shared: WebsocketListenerShared::new(local_node, address).await? })
	}

	pub fn url(&self) -> &Url {
		&self.shared.url
	}

	#[doc(hidden)]
	pub fn inspect(&self) -> String {
		let mut ret = format!("┌───────────────────────────────────────────────────────────────────── Tokio::WebsocketListener: {}\n", self.url());
		for (_url, connection) in self.shared.connections.lock().unwrap().iter() {
			ret.push_str(connection.inspect().as_str());
		}
		ret.push_str("└╼");
		ret
	}
}

impl Drop for WebsocketListener {
	fn drop(&mut self) {
		self.shared.state.change(|current| match current {
			WebsocketListenerState::Listening => WebsocketListenerState::Stop,
			WebsocketListenerState::Stop | WebsocketListenerState::Stopped => current,
		});
	}
}

impl WebsocketListenerShared {
	async fn new<T: TryInto<Url> + std::fmt::Debug>(local_node: LocalNode, address: T) -> Result<Arc<WebsocketListenerShared>, WebsocketListenerError<T::Error>>
	where
		T::Error: std::fmt::Debug,
	{
		let mut url: Url = address.try_into().map_err(WebsocketListenerError::UrlConversionError)?;
		let bind_host = url.host_str().ok_or_else(|| WebsocketListenerError::ConnectionParametersParsingError("Missing host".to_string()))?.to_string();
		let bind_port = url.port().ok_or_else(|| WebsocketListenerError::ConnectionParametersParsingError("Missing port".to_string()))?;
		let tcp_listener = TcpListener::bind((bind_host.clone(), bind_port)).await.map_err(WebsocketListenerError::BindError)?;
		let final_address = tcp_listener.local_addr().unwrap();
		let _ = url.set_port(Some(final_address.port())); //TODO: should we really ignore errors?
		let _ = url.set_ip_host(final_address.ip()); //TODO: should we really ignore errors?
		info!("Listening on {}", url.as_str());
		let shared = Arc::new(WebsocketListenerShared {
			local_node,
			state: State::new(WebsocketListenerState::Listening),
			parameters: ConnectionParameters::from_url(&url).map_err(WebsocketListenerError::ConnectionParametersParsingError)?,
			url,
			connections: Mutex::new(HashMap::new()),
		});
		tokio::task::spawn(shared.clone().event_loop(tcp_listener));
		Ok(shared)
	}

	async fn event_loop(self: Arc<Self>, tcp_listener: TcpListener) {
		loop {
			let state_change_notified = self.state.notify().notified();
			if self.state.change_from_to(WebsocketListenerState::Stop, WebsocketListenerState::Stopped).is_ok() {
				break;
			};
			tokio::select! {
				//TODO: is this safe? or do we have to keep this future across loop iterations?
				new_connection = tcp_listener.accept() => match new_connection {
					Ok((stream, address)) => {
						let address = Url::parse(format!("ws://{:?}",address).as_str()).unwrap(); //TODO: is this stupid?
						info!("Incomming connection from {:?}", address.as_str());
						let shared_for_closure = self.clone();
						tokio::task::spawn(async move {
							let node = Arc::new(RemoteNode::new(&shared_for_closure.local_node, false, shared_for_closure.parameters.diff_produce, shared_for_closure.parameters.diff_request));
							//TODO: timeout
							match async_tungstenite::tokio::accept_async(stream).await {
								Ok(ws_stream) => {
									let connection = Arc::new(WebsocketConnection::new(address.clone(),ws_stream,shared_for_closure.parameters.clone(),node));
									let (_old_state, new_state) = shared_for_closure.state.change(|current| {
										if current == WebsocketListenerState::Listening {
											shared_for_closure.connections.lock().unwrap().insert(connection.url.clone(), connection.clone());
										};
										current
									});
									if new_state == WebsocketListenerState::Listening {
										match connection.remote_node_process().await {
											Ok(_) => info!("Disconnected {}", address.as_str()),
											Err(error @ Termination::SeriousError(_)) => error!("Error {} {}",address.as_str(),error),
											Err(error) => info!("Disconnected {} - {}",address.as_str(),error)
										};
										shared_for_closure.connections.lock().unwrap().remove(&connection.url);
									}
								}
								Err(error) => {
									info!("Socket accept error: {:?}", error);
								}
							};
						});
					}
					Err(_) => todo!(), //TODO what errors can happen here?
				},
				_ = state_change_notified => {}
			}
		}
		for (url, connection) in self.connections.lock().unwrap().drain() {
			trace!("Stopping connection {:?}", url);
			connection.stop();
		}
	}
}



#[derive(Clone)]
pub struct WebsocketConnector {
	shared: Arc<WebsocketConnectorShared>,
}

struct WebsocketConnectorShared {
	url: Url,
	state: State<WebsocketConnectorState>,
	connection: Mutex<Option<Arc<WebsocketConnection<WebsocketStreamFromConnectAsync>>>>,
	connection_parameters: ConnectionParameters,
	remote_node: Arc<RemoteNode>,
}

#[derive(Eq, PartialEq, Clone, Debug)]
enum WebsocketConnectorState {
	Connecting,
	Connected,
	Stop,
	Stopped,
}

impl WebsocketConnector {
	#[must_use = "Newly created WebsocketConnector needs to be held on to to keep connection alive"]
	pub fn new<T: TryInto<Url>>(local_node: LocalNode, address: T) -> Result<Self, AddressError<T::Error>>
	where
		T::Error: std::fmt::Debug,
	{
		let url = address.try_into().map_err(AddressError::UrlConversionError)?;
		let connection_parameters = ConnectionParameters::from_url(&url).map_err(AddressError::ConnectionParametersParsingError)?;
		let shared = Arc::new(WebsocketConnectorShared {
			url,
			state: State::new(WebsocketConnectorState::Connecting),
			connection: Mutex::new(None),
			remote_node: Arc::new(RemoteNode::new(&local_node, true, connection_parameters.diff_produce, connection_parameters.diff_request)),
			connection_parameters,
		});
		let shared_for_closure = shared.clone();
		tokio::task::spawn(async move {
			shared_for_closure.event_loop().await;
		});
		Ok(WebsocketConnector { shared })
	}

	pub fn url(&self) -> &Url {
		&self.shared.url
	}

	#[doc(hidden)]
	pub fn inspect(&self) -> String {
		let mut ret = format!("┌───────────────────────────────────────────────────────────────────── Tokio::WebsocketConnector: {}\n", self.url());
		if let Some(connection) = &*self.shared.connection.lock().unwrap() {
			ret.push_str(connection.inspect().as_str());
		}
		ret.push_str("└╼");
		ret
	}
}

impl Drop for WebsocketConnector {
	fn drop(&mut self) {
		self.shared.state.change(|current| match current {
			WebsocketConnectorState::Connecting => WebsocketConnectorState::Stop,
			WebsocketConnectorState::Connected => {
				let _ = self.shared.connection.lock().unwrap().as_ref().unwrap().stop();
				WebsocketConnectorState::Stop
			}
			WebsocketConnectorState::Stopped => WebsocketConnectorState::Stopped,
			_ => panic!(),
		});
	}
}


//TODO: stop, drop
impl WebsocketConnectorShared {
	async fn event_loop(&self) {
		let mut connection_future = None;
		loop {
			//TODO: timeout here
			let notified = self.state.notify().notified();
			if self.state.change_from_to(WebsocketConnectorState::Stop, WebsocketConnectorState::Stopped).is_ok() {
				return;
			};

			// some futures, like connect_async, have side effects, and select will happily drop them on the floor mid-execution (at await point ofc)
			// specifically, connect_async may get terminated after accepting tcp connection, but before handshake is over. leading to connection being dropped.
			if connection_future.is_none() {
				trace!("Connecting to {:?}", self.url.as_str());
				connection_future = Some(Box::pin(connect_async(self.url.to_string())));
			}
			tokio::select! {
				_ = notified => {},
				maybe_ws_stream = connection_future.as_mut().unwrap() => {
					connection_future.take();
					match maybe_ws_stream {
						Ok((ws_stream, _)) => {
							info!("Connected to {:?}", self.url.as_str());
							let connection = Arc::new(WebsocketConnection::new(self.url.clone(), ws_stream, self.connection_parameters.clone(),self.remote_node.clone()));
							*self.connection.lock().unwrap() = Some(connection.clone());
							if self.state.change_from_to(WebsocketConnectorState::Connecting, WebsocketConnectorState::Connected).is_ok() {
								match connection.remote_node_process().await {
									Ok(_) => info!("Disconnected {}", self.url.as_str()),
									Err(error @ Termination::SeriousError(_)) => {
										error!("Error {} {}",self.url.as_str(),error);
										sleep(Duration::from_millis(100)).await;
									},
									Err(error) => {
										info!("Disconnected {} - {}",self.url.as_str(),error);
										sleep(Duration::from_millis(100)).await;
									}
								};
								let _ = self.state.change_from_to(WebsocketConnectorState::Connected, WebsocketConnectorState::Connecting);
							}
							self.connection.lock().unwrap().take();
						}
						Err(error) => {
							info!("Connector error: {:?}", error);
							sleep(Duration::from_millis(100)).await;
						}
					}
				}
			}
		}
	}
}


#[derive(Debug)]
enum Termination {
	SocketClosed,
	BenignError(String),
	SeriousError(String),
}

// should be From trait with WebsocketConnection::T::Error, but:
// https://stackoverflow.com/questions/37347311/how-is-there-a-conflicting-implementation-of-from-when-using-a-generic-type
impl Termination {
	fn from_tungstenite_error(error: tungstenite::Error) -> Self {
		match error {
			tungstenite::Error::ConnectionClosed
			| tungstenite::Error::Io(_)
			| tungstenite::Error::Tls(_) //TODO: hm...
			| tungstenite::Error::Protocol(tungstenite::error::ProtocolError::ResetWithoutClosingHandshake) => Termination::SocketClosed,
			error => Termination::SeriousError(format!("{:?}", error)),
		}
	}

	fn from_other_error(error: impl std::fmt::Debug) -> Self {
		Self::SeriousError(format!("{:?}", error))
	}
}

impl std::fmt::Display for Termination {
	fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
		match self {
			Termination::SocketClosed => write!(f, "Socket closed"),
			Termination::BenignError(text) => write!(f, "Benign error termination: {:?}", text),
			Termination::SeriousError(text) => write!(f, "Serious error termination: {:?}", text),
		}
	}
}

impl std::error::Error for Termination {}


//TODO: simplify this if possible
type WebsocketStreamFromAcceptAsync = async_tungstenite::WebSocketStream<async_tungstenite::tokio::TokioAdapter<tokio::net::TcpStream>>;
type WebsocketStreamFromConnectAsync = async_tungstenite::WebSocketStream<
	async_tungstenite::stream::Stream<
		async_tungstenite::tokio::TokioAdapter<tokio::net::TcpStream>,
		async_tungstenite::tokio::TokioAdapter<tokio_native_tls::TlsStream<tokio::net::TcpStream>>,
	>,
>;


#[derive(Eq, PartialEq, Clone, Debug)]
enum WebsocketConnectionState {
	Created,
	Processing,
	Stop,
	Stopped,
}

struct WebsocketConnection<T>
where
	T: Send + Sync + Unpin + futures::Sink<tungstenite::Message> + futures::Stream<Item = Result<tungstenite::Message, tungstenite::Error>>,
	<T as futures::Sink<async_tungstenite::tungstenite::Message>>::Error: std::fmt::Debug,
{
	url: Url,
	parameters: ConnectionParameters,
	ws_stream_read: tokio::sync::Mutex<Option<SplitStream<T>>>,
	ws_stream_write: tokio::sync::Mutex<Option<SplitSink<T, async_tungstenite::tungstenite::Message>>>,
	last_sent_at: std::sync::Mutex<Instant>,
	last_received_at: std::sync::Mutex<Instant>,
	state: Arc<State<WebsocketConnectionState>>,
	remote_node: Arc<RemoteNode>,
}


impl<T> WebsocketConnection<T>
where
	T: Send
		+ Sync
		+ Unpin
		+ 'static
		+ futures::Sink<tungstenite::Message, Error = async_tungstenite::tungstenite::Error>
		+ futures::Stream<Item = Result<tungstenite::Message, tungstenite::Error>>,
{
	fn new(address: Url, ws_stream: T, parameters: ConnectionParameters, remote_node: Arc<RemoteNode>) -> WebsocketConnection<T> {
		let (write, read) = ws_stream.split();
		WebsocketConnection {
			url: address,
			parameters,
			ws_stream_read: tokio::sync::Mutex::new(Some(read)),
			ws_stream_write: tokio::sync::Mutex::new(Some(write)),
			last_sent_at: std::sync::Mutex::new(Instant::now()),
			last_received_at: std::sync::Mutex::new(Instant::now()),
			state: Arc::new(State::new(WebsocketConnectionState::Created)),
			remote_node,
		}
	}

	async fn remote_node_process(&self) -> Result<(), Termination> {
		if self
			.state
			.change(|state| match state {
				WebsocketConnectionState::Created => WebsocketConnectionState::Processing,
				WebsocketConnectionState::Stop => WebsocketConnectionState::Stopped,
				_ => panic!(),
			})
			.1 == WebsocketConnectionState::Stopped
		{
			return Ok(());
		};
		self.send(self.remote_node.connected().map_err(Termination::from_other_error)?).await?;
		let ret = self.remote_node_event_loop().await;
		self.remote_node.disconnected();
		if let Some(read) = self.ws_stream_read.lock().await.take() {
			let write = self.ws_stream_write.lock().await.take().unwrap();
			let _ = read.reunite(write).unwrap().close().await;
		}
		self.state.change(|state| match state {
			WebsocketConnectionState::Processing | WebsocketConnectionState::Stop => WebsocketConnectionState::Stopped,
			_ => panic!(),
		});
		ret
	}

	async fn remote_node_event_loop(&self) -> Result<(), Termination> {
		let mut local_node_events: EventStream<Descriptor> = self.remote_node.local_node_events.clone().into();
		let mut receive_future = None;
		loop {
			let cloned_state = self.state.clone();
			let notified = cloned_state.notify().notified();

			if receive_future.is_none() {
				receive_future = Some(Box::pin(self.receive()));
			};

			let mut messages_to_send = Vec::new();
			tokio::select! {
				_close = notified => {},
				message_result = receive_future.as_mut().unwrap() => {
					receive_future.take();
					messages_to_send.append(&mut self.remote_node.received_message(message_result?).map_err(Termination::from_other_error)?);
				},
				maybe_event = local_node_events.next() => {
					messages_to_send.append(&mut self.remote_node.received_local_node_event(maybe_event.unwrap()).map_err(Termination::from_other_error)?);
				},
			};
			if self.state.current() == WebsocketConnectionState::Stop {
				return Ok(());
			}
			self.send(messages_to_send).await?;
		}
	}

	fn stop(&self) {
		self.state.change(|state| match state {
			WebsocketConnectionState::Stopped => WebsocketConnectionState::Stopped,
			_ => WebsocketConnectionState::Stop,
		});
	}

	async fn send(&self, messages: Vec<Message>) -> Result<(), Termination> {
		let mut sink_mutex = self.ws_stream_write.lock().await;
		let sink = sink_mutex.as_mut().unwrap();
		if !messages.is_empty() {
			for message in messages {
				let tungstenite_message = tungstenite::Message::Binary(message.serialize().unwrap());
				sink.send(tungstenite_message).await.map_err(Termination::from_tungstenite_error)?;
			}
			*self.last_sent_at.lock().unwrap() = Instant::now(); //FORMER-BUG, REGRESSION-TEST-NEEDED: this was called even if messages vector was empty
		}
		Ok(())
	}

	//TODO: move keep-alives and acks to remote_node_event_loop()
	async fn receive(&self) -> Result<Message, Termination> {
		let mut ws_stream_read = self.ws_stream_read.lock().await;
		loop {
			// sending ack if needed
			if Instant::now() > { *self.last_sent_at.lock().unwrap() } + Duration::from_secs(1) {
				if let Some(message) = self.remote_node.ack() {
					self.send(vec![message]).await?;
				}
			}
			// sending ack or keep-alive if needed
			if Instant::now() > { *self.last_sent_at.lock().unwrap() } + self.parameters.timeout / 2 {
				if let Some(message) = self.remote_node.ack() {
					self.send(vec![message]).await?;
				} else {
					*self.last_sent_at.lock().unwrap() = Instant::now();
					self.ws_stream_write
						.lock()
						.await
						.as_mut()
						.unwrap()
						.send(tungstenite::Message::Ping(vec![]))
						.await
						.map_err(Termination::from_tungstenite_error)?;
				}
			};
			// trying to receive something
			if let Ok(message) = timeout(min(self.parameters.timeout / 4, Duration::from_millis(1000)), ws_stream_read.as_mut().unwrap().next()).await {
				match message {
					Some(Ok(tungstenite::Message::Binary(serialized_message))) => {
						*self.last_received_at.lock().unwrap() = Instant::now();
						if !serialized_message.is_empty() {
							return Message::deserialize(&serialized_message).map_err(Termination::from_other_error);
						}
					}
					Some(Ok(tungstenite::Message::Ping(_payload))) => {
						//trace!("Received ping");
						*self.last_received_at.lock().unwrap() = Instant::now();
					}
					Some(Ok(tungstenite::Message::Pong(_payload))) => {
						//trace!("Received pong");
						*self.last_received_at.lock().unwrap() = Instant::now();
					}
					Some(Ok(tungstenite::Message::Close(_payload))) => {
						return Err(Termination::SocketClosed);
					}
					Some(Ok(other_message)) => {
						todo!("Got unexpected message: {:?}", other_message);
					}
					Some(Err(error)) => return Err(Termination::from_tungstenite_error(error)),
					None => todo!(),
				}
			};
			// checking timeout
			if Instant::now() > { *self.last_received_at.lock().unwrap() } + self.parameters.timeout {
				return Err(Termination::BenignError("Read timeout".to_string()));
			}
		}
	}

	pub(crate) fn inspect(&self) -> String {
		let mut ret = format!("├─────\n│ URL: {:?}\n", self.url.to_string(),);
		ret.push_str(self.remote_node.inspect().as_str());
		ret
	}
}


pub mod ffi {
	#![allow(clippy::not_unsafe_ptr_arg_deref)]


	use super::*;
	use crate::ffi::FFIError;


	pub struct TokioRuntime {
		runtime: tokio::runtime::Runtime,
	}


	#[no_mangle]
	pub extern "C" fn hakuban_tokio_init_multi_thread(worker_threads: usize) -> *mut TokioRuntime {
		let mut builder = tokio::runtime::Builder::new_multi_thread(); //new_current_thread()
		if worker_threads > 0 {
			builder.worker_threads(worker_threads);
		};
		builder.enable_all();
		builder.thread_name("hakuban");
		let runtime = builder.build().unwrap();
		Box::into_raw(Box::new(TokioRuntime { runtime }))
	}

	//TODO: deinit

	#[repr(C)]
	pub struct FFITokioWebsocketConnectorNewResult {
		error: FFIError,
		websocket_connector: *mut WebsocketConnector,
	}


	#[no_mangle]
	pub extern "C" fn hakuban_tokio_websocket_connector_new(
		runtime: *mut TokioRuntime, local_node: *mut LocalNode, address: *const i8,
	) -> FFITokioWebsocketConnectorNewResult {
		let runtime: Box<TokioRuntime> = unsafe { Box::from_raw(runtime) };
		let _runtime_context = runtime.runtime.enter();
		std::mem::forget(runtime);
		let local_node: Box<LocalNode> = unsafe { Box::from_raw(local_node) };
		let address = unsafe { std::ffi::CStr::from_ptr(address).to_string_lossy().into_owned() };
		let connector = match WebsocketConnector::new(*local_node.clone(), address.as_str()) {
			Ok(connector) => FFITokioWebsocketConnectorNewResult { error: FFIError::None, websocket_connector: Box::into_raw(Box::new(connector)) },
			Err(error) => {
				error!("Invalid URL: {}", error);
				FFITokioWebsocketConnectorNewResult { error: FFIError::InvalidURL, websocket_connector: std::ptr::null_mut() }
			}
		};
		std::mem::forget(local_node);
		connector
	}

	#[no_mangle]
	pub extern "C" fn hakuban_tokio_websocket_connector_drop(runtime: *mut TokioRuntime, websocket_connector: *mut WebsocketConnector) {
		let runtime: Box<TokioRuntime> = unsafe { Box::from_raw(runtime) };
		let _runtime_context = runtime.runtime.enter();
		std::mem::forget(runtime);
		drop(unsafe { Box::from_raw(websocket_connector) });
	}

	//TODO: listener
}
