use std::{collections::{HashMap, HashSet}, sync::{Arc, Mutex, RwLock}};

use instant::Instant;

use super::core::TagCore;
use crate::{contract::Contract, events::{EventMerge, EventMergeInput}, expose_contract::ExposeContract, node::core::NodeCore, object::{core::{ObjectCore, ObjectVersion, Synchronized}, expose::ObjectSerializer}, DefaultSerializerError, LocalNode, ObjectDescriptor, TagDescriptor};


/// Represents a wish, a __contract__ to expose any object with specific tag
pub struct TagExpose<T: Send + Sync + 'static, E = DefaultSerializerError> {
	contract: Arc<TagExposeContract>,
	serializer: ObjectSerializer<T, E>,
}

impl<T: Send + Sync + 'static, E> TagExpose<T, E> {
	pub(super) fn new(contract: Arc<TagExposeContract>, serializer: ObjectSerializer<T, E>) -> TagExpose<T, E> {
		TagExpose { contract, serializer }
	}

	pub fn descriptor(&self) -> &TagDescriptor {
		&self.contract.tag_core.descriptor
	}

	pub fn object_descriptors(&self) -> HashSet<ObjectDescriptor> {
		//MISSING-REGRESSION-TEST: this reported active objects instead of assigned objects and no test caught it
		self.contract.assignments.read().unwrap().assigned_objects.keys().cloned().collect()
	}

	pub fn changes(&self) -> EventMerge<ObjectDescriptor> {
		self.contract.assigned_objects_changes()
	}

	pub fn object_assignment(&self, object_descriptor: &ObjectDescriptor) -> Option<u64> {
		self.contract.assignment(object_descriptor)
	}

	pub fn set_object_state<'a>(
		&self, descriptor: impl Into<&'a ObjectDescriptor>, version: impl Into<&'a ObjectVersion>, cooked: impl Into<&'a T>, assignment_id: u64,
	) -> Result<bool, E> {
		//WARNING: there was a deadlock here: keeping lock while calling state_set, leading to deadlocks with unlink_object_observe_contract->recalculate_active->expose_contract_unassign->unassign
		//WARNING: and again, after rewrite, a deadlock on self.contract.assigment, instant self deadlock. this time there is a test which detects this.
		let object_core = self.contract.assignments.read().unwrap().assigned_objects.get(descriptor.into()).map(
			|assignment| assignment.object_core.clone(), //TODO: check if this can cause acquire/release leak
		);
		if let Some(object_core) = object_core {
			let (data_type, raw) = (self.serializer)(cooked.into())?;
			let ret = object_core.state_set_with_assignment_id(version.into(), data_type, raw, assignment_id, Synchronized::Now);
			Ok(ret)
		} else {
			Ok(false)
		}
	}

	pub fn set_object_state_at<'a>(
		&self, descriptor: impl Into<&'a ObjectDescriptor>, version: impl Into<&'a ObjectVersion>, cooked: impl Into<&'a T>, assignment_id: u64,
		last_synchronized_at: Instant,
	) -> Result<bool, E> {
		let object_core = self.contract.assignments.read().unwrap().assigned_objects.get(descriptor.into()).map(
			|assignment| assignment.object_core.clone(), //TODO: check if this can cause acquire/release leak
		);
		if let Some(object_core) = object_core {
			let (data_type, raw) = (self.serializer)(cooked.into())?;
			let ret = object_core.state_set_with_assignment_id(version.into(), data_type, raw, assignment_id, Synchronized::LastAt(last_synchronized_at));
			Ok(ret)
		} else {
			Ok(false)
		}
	}

	pub fn set_object_data<'a>(&self, descriptor: impl Into<&'a ObjectDescriptor>, cooked: impl Into<&'a T>) -> Result<(), E> {
		let object_core = self.contract.assignments.read().unwrap().assigned_objects.get(descriptor.into()).map(
			|assignment| assignment.object_core.clone(), //TODO: check if this can cause acquire/release leak
		);
		if let Some(object_core) = object_core {
			//this is racy by nature
			let (data_type, raw) = (self.serializer)(cooked.into())?;
			object_core.data_set(self.contract.node_core(), data_type, raw);
		};
		Ok(())
	}

	pub fn object_desynchronize<'a>(&self, descriptor: impl Into<&'a ObjectDescriptor>, assignment_id: u64) {
		let object_core = self.contract.assignments.read().unwrap().assigned_objects.get(descriptor.into()).map(
			|assignment| assignment.object_core.clone(), //TODO: check if this can cause acquire/release leak
		);
		if let Some(object_core) = object_core {
			object_core.desynchronize(assignment_id);
		}
	}
}


impl<T: Sync + Send, E> Drop for TagExpose<T, E> {
	fn drop(&mut self) {
		self.contract.destroy();
	}
}

pub(super) struct Assignment {
	pub object_core: Arc<ObjectCore>,
	pub assignment: u64,
}

pub(super) struct Assignments {
	pub counter: u64,
	pub assigned_objects: HashMap<ObjectDescriptor, Assignment>,
}

pub(super) struct TagExposeContract {
	local_node: Option<LocalNode>,
	tag_core: Arc<TagCore>,
	composition_cost: u32,
	assignments: RwLock<Assignments>,
	assigned_objects_changes: EventMerge<ObjectDescriptor>, //TODO: only for tag dir
	assigned_objects_changes_tokens: Mutex<HashMap<ObjectDescriptor, EventMergeInput<ObjectDescriptor>>>, //TODO only for this dir
}

impl TagExposeContract {
	pub(super) fn new(local_node: LocalNode, tag_core: Arc<TagCore>, composition_cost: u32) -> Arc<Self> {
		let contract = Arc::new(TagExposeContract {
			local_node: Some(local_node),
			tag_core,
			composition_cost,
			assignments: RwLock::new(Assignments { counter: 0, assigned_objects: HashMap::new() }),
			assigned_objects_changes: EventMerge::new(),
			assigned_objects_changes_tokens: Mutex::new(HashMap::new()),
		});
		contract.tag_core.link_tag_expose_contract(contract.clone());
		contract
	}

	pub(super) fn destroy(self: &Arc<Self>) {
		let dyn_self: Arc<dyn ExposeContract> = self.clone();
		self.tag_core.unlink_tag_expose_contract(&dyn_self);
		drop(dyn_self);
		assert_eq!(Arc::strong_count(self), 1);
	}

	pub(super) fn assigned_objects_changes(&self) -> EventMerge<ObjectDescriptor> {
		self.assigned_objects_changes.clone()
	}
}

impl Contract for TagExposeContract {
	fn id(&self) -> usize {
		(self as *const Self) as usize
	}

	fn node_core(&self) -> &Arc<NodeCore> {
		self.local_node.as_ref().unwrap().shared.node_core.as_ref().unwrap()
	}
}

impl ExposeContract for TagExposeContract {
	fn assign(&self, object_core: Arc<ObjectCore>) -> bool {
		//TODO: should this happen after locking?
		self.assigned_objects_changes_tokens
			.lock()
			.unwrap()
			.insert(object_core.descriptor.clone(), self.assigned_objects_changes.include(object_core.changes())); //TODO: check if we still exist
		let mut assignments = self.assignments.write().unwrap();
		let descriptor = object_core.descriptor.clone();
		assignments.counter += 1;
		let assignment = Assignment { object_core, assignment: assignments.counter };
		if assignments.assigned_objects.insert(descriptor, assignment).is_some() {
			panic!("Double assignment in TagExposeContract");
		};
		true
	}

	fn unassign(&self, object_descriptor: &ObjectDescriptor) -> bool {
		self.assigned_objects_changes_tokens.lock().unwrap().remove(object_descriptor).unwrap().remove();
		let mut assignments = self.assignments.write().unwrap();
		assignments.assigned_objects.remove(object_descriptor).unwrap();
		true
	}

	fn assignment(&self, object_descriptor: &ObjectDescriptor) -> Option<u64> {
		let assignments = self.assignments.write().unwrap();
		assignments.assigned_objects.get(object_descriptor).map(|assignment| assignment.assignment)
	}

	fn composition_cost(&self) -> u32 {
		self.composition_cost
	}
}

impl std::fmt::Debug for TagExposeContract {
	fn fmt(&self, formatter: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
		formatter.write_str(&format!("TagExposeContract:{}", self.tag_core.descriptor))
	}
}
