use std::{collections::{HashMap, HashSet}, fmt::Debug, hash::{Hash, Hasher}, sync::{Arc, Mutex, RwLock, Weak}};

use events::EventSource;

use crate::{descriptor::ObjectDescriptor, events::{self, EventMerge, EventMergeInput}, expose_contract::ExposeContract, node::{core::NodeCore, LocalNodeShared}, object::core::ObjectCore, observe_contract::ObserveContract, TagDescriptor};

pub(crate) struct TagCore {
	local_node: Arc<LocalNodeShared>,
	pub(crate) descriptor: TagDescriptor,
	links: RwLock<TagLinks>,
	changes: EventSource<TagDescriptor>,
	pub(super) active_objects: RwLock<HashMap<ObjectDescriptor, Arc<ObjectCore>>>, //shared by all TagObserveContracts
	active_objects_changes: EventMerge<ObjectDescriptor>,
	active_objects_changes_tokens: Mutex<HashMap<ObjectDescriptor, EventMergeInput<ObjectDescriptor>>>,
	objects_changes: EventMerge<ObjectDescriptor>,
	objects_changes_tokens: Mutex<HashMap<ObjectDescriptor, EventMergeInput<ObjectDescriptor>>>,
}

struct TagLinks {
	objects: HashMap<ObjectDescriptor, Weak<ObjectCore>>, // used solely to notify objects of inserted/removed tag contracts
	observe_contracts: HashMap<Arc<NodeCore>, HashSet<Arc<dyn ObserveContract>>>,
	expose_contracts: HashMap<Arc<NodeCore>, HashSet<Arc<dyn ExposeContract>>>,
}


impl TagCore {
	pub(crate) fn new(local_node: Arc<LocalNodeShared>, descriptor: TagDescriptor) -> TagCore {
		//TODO: does this reduce the number of hashbrown monomorphizations?
		let hash_builder = std::collections::hash_map::RandomState::new();
		TagCore {
			local_node,
			changes: EventSource::new(descriptor.clone()),
			descriptor,
			links: RwLock::new(TagLinks {
				objects: HashMap::with_hasher(hash_builder.clone()),
				observe_contracts: HashMap::with_hasher(hash_builder.clone()),
				expose_contracts: HashMap::with_hasher(hash_builder.clone()),
			}),
			active_objects: RwLock::new(HashMap::with_hasher(hash_builder.clone())),
			active_objects_changes: EventMerge::new(),
			active_objects_changes_tokens: Mutex::new(HashMap::with_hasher(hash_builder.clone())),
			objects_changes: EventMerge::new(),
			objects_changes_tokens: Mutex::new(HashMap::with_hasher(hash_builder)),
		}
	}

	pub(crate) fn inspect(&self) -> String {
		let links = self.links.read().unwrap();
		format!(
			"├─────\n│ Tag: {:?}\n│ Observe contracts: {:?}\n│ Expose contracts: {:?}\n",
			self.descriptor,
			links.observe_contracts.keys(),
			links.expose_contracts.keys()
		)
	}

	pub(super) fn link_tag_observe_contract(self: &Arc<Self>, contract: Arc<dyn ObserveContract>) {
		let mut links = self.links.write().unwrap();
		links.observe_contracts.entry(contract.node_core().clone()).or_insert_with(HashSet::new).insert(contract.clone());
		let first = links.observe_contracts.len() == 1;
		let objects_to_notify = links.objects.values().filter_map(Weak::upgrade).collect::<Vec<Arc<ObjectCore>>>();
		drop(links);
		for object in objects_to_notify {
			object.tag_changed();
		}
		if first {
			//TODO: really, or always?
			self.changes.change();
		}
	}

	pub(super) fn link_tag_expose_contract(&self, contract: Arc<dyn ExposeContract>) {
		let mut links = self.links.write().unwrap();
		links.expose_contracts.entry(contract.node_core().clone()).or_insert_with(HashSet::new).insert(contract.clone());
		let first = links.expose_contracts.len() == 1;
		let objects_to_notify = links.objects.values().filter_map(Weak::upgrade).collect::<Vec<Arc<ObjectCore>>>();
		drop(links);
		for object in objects_to_notify {
			object.tag_changed();
			//TODO: consider reasigning to this?
		}
		if first {
			self.changes.change();
		}
	}

	pub(super) fn unlink_tag_observe_contract(self: &Arc<Self>, contract: &Arc<dyn ObserveContract>) {
		let mut links = self.links.write().unwrap();
		let observe_contracts = links.observe_contracts.get_mut(contract.node_core()).unwrap();
		observe_contracts.remove(contract);
		if observe_contracts.is_empty() {
			links.observe_contracts.remove(contract.node_core());
		};
		let objects_to_notify = links.objects.values().filter_map(Weak::upgrade).collect::<Vec<Arc<ObjectCore>>>();
		drop(links);
		for object in objects_to_notify {
			object.tag_changed();
		}
		self.changes.change();
	}

	pub(super) fn unlink_tag_expose_contract(self: &Arc<Self>, contract: &Arc<dyn ExposeContract>) {
		let mut links = self.links.write().unwrap();
		let expose_contracts = links.expose_contracts.get_mut(contract.node_core()).unwrap();
		expose_contracts.remove(contract);
		if expose_contracts.is_empty() {
			links.expose_contracts.remove(contract.node_core());
		};
		let objects_to_notify = links.objects.values().filter_map(Weak::upgrade).collect::<Vec<Arc<ObjectCore>>>();
		drop(links);
		for object in objects_to_notify {
			object.expose_contract_unassign(contract);
			object.tag_changed();
		}
		self.changes.change();
	}

	pub(crate) fn active_insert(&self, object_core: Arc<ObjectCore>) {
		let descriptor = object_core.descriptor.clone();
		let object_changes = object_core.changes();
		self.active_objects.write().unwrap().insert(descriptor.clone(), object_core);
		self.active_objects_changes_tokens.lock().unwrap().insert(descriptor, self.active_objects_changes.include(object_changes));
	}

	pub(crate) fn active_remove(&self, object_core: &ObjectCore) {
		self.active_objects.write().unwrap().remove(&object_core.descriptor);
		//TODO: deactivate expose contract?
		self.active_objects_changes_tokens.lock().unwrap().remove(&object_core.descriptor).unwrap().remove();
	}

	pub(crate) fn object_insert(&self, object_core: &Arc<ObjectCore>) {
		self.objects_changes_tokens.lock().unwrap().insert(object_core.descriptor.clone(), self.objects_changes.include(object_core.changes()));
		self.links.write().unwrap().objects.insert(object_core.descriptor.clone(), Arc::downgrade(object_core));
	}

	pub(crate) fn object_remove(&self, descriptor: &ObjectDescriptor) {
		self.links.write().unwrap().objects.remove(descriptor);
		self.objects_changes_tokens.lock().unwrap().remove(descriptor).unwrap().remove();
		assert!(!self.active_objects.read().unwrap().contains_key(descriptor));
	}

	pub(crate) fn has_observe_expose_contracts(&self) -> (bool, bool) {
		let links = self.links.read().unwrap();
		(!links.observe_contracts.is_empty(), !links.expose_contracts.is_empty())
	}

	pub(crate) fn expose_contracts(&self) -> Vec<Arc<dyn ExposeContract>> {
		self.links.read().unwrap().expose_contracts.values().flatten().cloned().collect()
	}

	pub(super) fn active_objects_changes(&self) -> EventMerge<ObjectDescriptor> {
		self.active_objects_changes.clone()
	}

	pub(super) fn objects_changes(&self) -> EventMerge<ObjectDescriptor> {
		self.objects_changes.clone()
	}

	pub(crate) fn changes(&self) -> EventSource<TagDescriptor> {
		self.changes.clone()
	}

	pub(crate) fn has_observer_other_than(self: &Arc<Self>, node_core: &Arc<NodeCore>) -> bool {
		let tag_links = self.links.read().unwrap();
		tag_links.observe_contracts.len() > 1 || (!tag_links.observe_contracts.is_empty() && !tag_links.observe_contracts.contains_key(node_core))
	}

	pub(crate) fn has_exposer_other_than(self: &Arc<Self>, node_core: &Arc<NodeCore>) -> bool {
		let tag_links = self.links.read().unwrap();
		tag_links.expose_contracts.len() > 1 || (!tag_links.expose_contracts.is_empty() && !tag_links.expose_contracts.contains_key(node_core))
	}

	pub(crate) fn is_exposed_by_node(self: &Arc<Self>, node_core: &Arc<NodeCore>) -> bool {
		self.links.read().unwrap().expose_contracts.contains_key(node_core)
	}

	pub(crate) fn is_observed_by_node(self: &Arc<Self>, node_core: &Arc<NodeCore>) -> bool {
		self.links.read().unwrap().observe_contracts.contains_key(node_core)
	}

	//TODO: cache this
	pub(crate) fn minimum_cost_excluding_node(self: &Arc<Self>, node_core: &Arc<NodeCore>) -> Option<u32> {
		let mut lowest_cost = None;

		for expose_contract in self.expose_contracts() {
			if expose_contract.node_core() != node_core {
				let cost = expose_contract.composition_cost();
				if lowest_cost.is_none() || cost < lowest_cost.unwrap() {
					lowest_cost = Some(cost);
				}
			}
		}

		lowest_cost
	}
}



impl Debug for TagCore {
	fn fmt(&self, formatter: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
		formatter.write_str(&format!("Tag:{}", self.descriptor))
	}
}


impl Hash for TagCore {
	fn hash<H: Hasher>(&self, state: &mut H) {
		self.descriptor.hash(state);
	}
}

impl PartialEq for TagCore {
	fn eq(&self, other: &Self) -> bool {
		self.descriptor == other.descriptor
	}
}

impl Eq for TagCore {}


//TEST-NEEDED: removing this doesn't make any test fail
impl Drop for TagCore {
	fn drop(&mut self) {
		self.local_node.tag_release(&self.descriptor);
	}
}
