#![allow(clippy::mutex_atomic)] //because I want the value AND the locking. I think :)

use std::{collections::hash_map::Entry, hash::{Hash, Hasher}, sync::{atomic::{AtomicU32, Ordering}, Arc, Mutex, RwLock}};

use super::{core::{ObjectCore, ObjectRawData, ObjectState, ObjectType, ObjectVersion, Synchronized}, Assignment};
use crate::{contract::Contract, diff, events::EventSource, expose_contract::ExposeContract, node::{core::NodeCore, local::LocalNodeShared}, observe_contract::ObserveContract, ObjectDescriptor};

pub(crate) struct RemoteObject {
	object_interface: Arc<RemoteObjectInterface>,
}


impl RemoteObject {
	pub(crate) fn new(local_node: Arc<LocalNodeShared>, remote_node_core: Arc<NodeCore>, descriptor: &ObjectDescriptor) -> RemoteObject {
		let object_core = local_node.object_acquire(descriptor);
		let object_interface = RemoteObjectInterface::new(object_core, remote_node_core);
		RemoteObject { object_interface }
	}

	pub(crate) fn expose(&self, value: bool, cost: u32) {
		self.object_interface.composition_cost.store(cost, Ordering::SeqCst); //TODO weaken
		let mut expose = self.object_interface.expose.lock().unwrap();
		if *expose != value {
			let dyn_self: Arc<dyn ExposeContract> = self.object_interface.clone();
			if value {
				self.object_interface.object_core.link_object_expose_contract(dyn_self);
			} else {
				self.object_interface.object_core.unlink_object_expose_contract(&dyn_self);
			}
			*expose = value;
		}
	}

	pub(crate) fn observe(&self, value: bool) {
		let mut observe = self.object_interface.observe.lock().unwrap();
		if *observe != value {
			let dyn_self: Arc<dyn ObserveContract> = self.object_interface.clone();
			if value {
				self.object_interface.object_core.link_object_observe_contract(dyn_self);
			} else {
				self.object_interface.object_core.unlink_object_observe_contract(&dyn_self);
			}
			*observe = value;
		}
	}

	pub(crate) fn state(&self) -> Option<ObjectState> {
		self.object_interface.object_core.state_get()
	}

	pub(crate) fn changes(&self) -> EventSource<ObjectDescriptor> {
		self.object_interface.object_core.changes()
	}

	pub(crate) fn is_observed_by_this_node(&self) -> bool {
		self.object_interface.object_core.is_observed_by_node(&self.object_interface.remote_node_core)
	}

	pub(crate) fn is_exposed_by_this_node(&self) -> bool {
		self.object_interface.object_core.is_exposed_by_node(&self.object_interface.remote_node_core)
	}

	pub(crate) fn is_assigned_to_this_node(&self) -> bool {
		self.object_interface.object_core.is_assigned_to_node(&self.object_interface.remote_node_core)
	}

	pub(crate) fn is_assigned_to_other_node(&self) -> bool {
		self.object_interface.object_core.is_assigned_to_node_other_than(&self.object_interface.remote_node_core)
	}

	pub(crate) fn is_observed_by_other_node(&self, check_tags: bool) -> bool {
		self.object_interface.object_core.has_observer_other_than(&self.object_interface.remote_node_core, check_tags)
	}

	pub(crate) fn is_exposed_by_other_node(&self, check_tags: bool) -> bool {
		self.object_interface.object_core.has_exposer_other_than(&self.object_interface.remote_node_core, check_tags)
	}

	#[allow(clippy::ptr_arg)]
	pub(crate) fn state_set(&self, version: &ObjectVersion, raw: ObjectRawData, data_type: ObjectType, synchronized: Synchronized) -> bool {
		self.object_interface.object_core.state_set_with_node(&self.object_interface.remote_node_core, version, data_type, raw, synchronized)
	}

	pub(crate) fn minimum_cost_in_other_nodes(&self) -> Option<u32> {
		self.object_interface.object_core.minimum_cost_excluding_node(&self.object_interface.remote_node_core)
	}

	pub(crate) fn get_diff(
		&self, old_version: Option<&ObjectVersion>, old_data: Option<&ObjectRawData>, new_version: &[i64], new_data: &ObjectRawData,
	) -> Option<ObjectRawData> {
		let diff_mutex = {
			let mut diffs = self.object_interface.object_core.diffs.lock().unwrap();
			while diffs.len() > 1 {
				let key = diffs.keys().min().unwrap().clone();
				diffs.remove(&key);
			}
			match diffs.entry((old_version.cloned(), new_version.into())) {
				Entry::Occupied(entry) => entry.get().clone(),
				Entry::Vacant(entry) => entry.insert(Arc::new(Mutex::new(None))).clone(),
			}
		};
		let mut diff_mutex_lock = diff_mutex.lock().unwrap();

		Some(
			diff_mutex_lock
				.get_or_insert_with(|| {
					let (diff, new_state) = if let Some(old_version) = old_version {
						let state_mutex = {
							let mut diff_states = self.object_interface.object_core.diff_states.lock().unwrap();
							while diff_states.len() > 2 {
								let key = diff_states.keys().min().unwrap().clone();
								diff_states.remove(&key);
							}
							match diff_states.entry(old_version.clone()) {
								Entry::Occupied(entry) => entry.get().clone(),
								Entry::Vacant(entry) => entry.insert(Arc::new(Mutex::new(None))).clone(),
							}
						};
						let mut state_mutex_lock = state_mutex.lock().unwrap();
						let state = state_mutex_lock.get_or_insert_with(|| diff::State::new().diff(&[], old_data.unwrap(), 8, 6).unwrap().1);
						state.diff(old_data.unwrap(), new_data, 8, 6).unwrap()
					} else {
						let state = diff::State::new();
						state.diff(&[], new_data, 8, 6).unwrap()
					};
					let mut diff_states = self.object_interface.object_core.diff_states.lock().unwrap();
					if !diff_states.contains_key(new_version) {
						diff_states.insert(new_version.into(), Arc::new(Mutex::new(Some(new_state))));
					}
					Arc::new(diff)
				})
				.clone(),
		)
	}

	#[doc(hidden)]
	#[allow(dead_code)]
	pub fn inspect(&self) -> String {
		self.object_interface.object_core.inspect(None)
	}
}



impl Drop for RemoteObject {
	fn drop(&mut self) {
		self.object_interface.destroy();
	}
}


struct RemoteObjectInterface {
	remote_node_core: Arc<NodeCore>,
	object_core: Arc<ObjectCore>,
	expose: Mutex<bool>,
	observe: Mutex<bool>,
	composition_cost: AtomicU32,
	assignment: RwLock<Assignment>,
	changes: EventSource<ObjectDescriptor>,
}

impl RemoteObjectInterface {
	pub(super) fn new(object_core: Arc<ObjectCore>, node: Arc<NodeCore>) -> Arc<RemoteObjectInterface> {
		Arc::new(RemoteObjectInterface {
			remote_node_core: node,
			changes: EventSource::new(object_core.descriptor.clone()),
			object_core,
			expose: Mutex::new(false),
			observe: Mutex::new(false),
			composition_cost: AtomicU32::new(0),
			assignment: RwLock::new(Assignment { counter: 0, assigned: false }),
		})
	}
}


impl Hash for RemoteObjectInterface {
	fn hash<H: Hasher>(&self, state: &mut H) {
		self.id().hash(state);
	}
}

impl PartialEq for RemoteObjectInterface {
	fn eq(&self, other: &Self) -> bool {
		self.id() == other.id()
	}
}

impl Eq for RemoteObjectInterface {}

impl RemoteObjectInterface {
	pub(super) fn destroy(self: &Arc<Self>) {
		if *self.observe.lock().unwrap() {
			let dyn_self: Arc<dyn ObserveContract> = self.clone();
			self.object_core.unlink_object_observe_contract(&dyn_self);
		};
		if *self.expose.lock().unwrap() {
			let dyn_self: Arc<dyn ExposeContract> = self.clone();
			self.object_core.unlink_object_expose_contract(&dyn_self);
		};
		assert_eq!(Arc::strong_count(self), 1);
	}
}

impl std::fmt::Debug for RemoteObjectInterface {
	fn fmt(&self, formatter: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
		formatter.write_str(&format!("RemoteObjectInterface:{}", self.object_core.descriptor))
	}
}



impl Contract for RemoteObjectInterface {
	fn id(&self) -> usize {
		(self as *const Self) as usize
	}

	fn node_core(&self) -> &Arc<NodeCore> {
		&self.remote_node_core
	}
}

impl ObserveContract for RemoteObjectInterface {}

//TODO: all the same issues as in expose.rs
impl ExposeContract for RemoteObjectInterface {
	fn composition_cost(&self) -> u32 {
		self.composition_cost.load(Ordering::SeqCst) //TODO: weaken
	}

	fn assign(&self, _object_core: Arc<ObjectCore>) -> bool {
		let mut assignment = self.assignment.write().unwrap();
		if assignment.assigned {
			panic!("Double assign of ObjectExpose");
		};
		assignment.counter += 1;
		assignment.assigned = true;
		self.node_core().load.fetch_add(self.composition_cost(), Ordering::SeqCst); //TODO: maybe weaker?
		self.changes.change();
		true //TODO: do we need to check if we still exist?
	}

	fn unassign(&self, _object_descriptor: &ObjectDescriptor) -> bool {
		//TODO: panic if object doesn't match?
		let mut assignment = self.assignment.write().unwrap();
		if !assignment.assigned {
			panic!("Double unassign of ObjectExpose");
		}
		assignment.assigned = false;
		self.node_core().load.fetch_sub(self.composition_cost(), Ordering::SeqCst); //TODO: maybe weaker?
		self.changes.change(); //FORMER-BUG, REGRESSION-TEST-NEEDED, this was missing and no rust test detected it
		true //TODO: do we need to check if we still exist?
	}

	fn assignment(&self, _object_descriptor: &ObjectDescriptor) -> Option<u64> {
		//TODO: panic if object doesn't match?
		let assignment = self.assignment.read().unwrap();
		if assignment.assigned {
			Some(assignment.counter)
		} else {
			None
		}
	}
}
