use std::sync::{atomic::Ordering, Arc, RwLock};

use instant::Instant;

use super::core::{ObjectCore, ObjectRawData, ObjectType, ObjectVersion, Synchronized};
use crate::{contract::Contract, events::EventSource, expose_contract::ExposeContract, node::core::NodeCore, DefaultSerializerError, LocalNode, ObjectDescriptor};

/// `<T, E> = Box<dyn Send + Sync + Fn(&T) -> Result<(ObjectType, ObjectRawData), E>>`
pub type ObjectSerializer<T, E> = Box<dyn Send + Sync + Fn(&T) -> Result<(ObjectType, ObjectRawData), E>>;

/// Represents a wish, a __contract__ to expose an object
pub struct ObjectExpose<T: Send + Sync + 'static, E = DefaultSerializerError> {
	contract: Arc<ObjectExposeContract>,
	serializer: ObjectSerializer<T, E>,
}


impl<T: Send + Sync + 'static, E> ObjectExpose<T, E> {
	pub(super) fn new(contract: Arc<ObjectExposeContract>, serializer: ObjectSerializer<T, E>) -> ObjectExpose<T, E> {
		ObjectExpose { contract, serializer }
	}

	pub fn descriptor(&self) -> &ObjectDescriptor {
		&self.contract.object_core.descriptor
	}

	pub fn changes(&self) -> EventSource<ObjectDescriptor> {
		self.contract.changes()
	}

	pub fn assignment(&self) -> Option<u64> {
		self.contract.assignment(&self.contract.object_core.descriptor)
	}

	// TODO: should this always take assignment id?
	pub fn set_object_state<'a>(&self, version: impl Into<&'a ObjectVersion>, cooked: impl Into<&'a T>, assignment_id: u64) -> Result<bool, E> {
		let (data_type, raw) = (self.serializer)(cooked.into())?;
		let ret = self.contract.object_core.state_set_with_assignment_id(version.into(), data_type, raw, assignment_id, Synchronized::Now);
		Ok(ret)
	}

	pub fn set_object_state_at<'a>(
		&self, version: impl Into<&'a ObjectVersion>, cooked: impl Into<&'a T>, assignment_id: u64, last_synchronized_at: Instant,
	) -> Result<bool, E> {
		let (data_type, raw) = (self.serializer)(cooked.into())?;
		let ret =
			self.contract.object_core.state_set_with_assignment_id(version.into(), data_type, raw, assignment_id, Synchronized::LastAt(last_synchronized_at));
		Ok(ret)
	}

	pub fn set_object_data<'a>(&self, cooked: impl Into<&'a T>) -> Result<(), E> {
		//NOTE: this one's racy by nature. it's all fine. it's fine.
		let (data_type, raw) = (self.serializer)(cooked.into())?;
		self.contract.object_core.data_set(self.contract.node_core(), data_type, raw);
		Ok(())
	}

	pub fn desynchronize(&self, assignment_id: u64) {
		self.contract.object_core.desynchronize(assignment_id);
	}
}


impl<T: Sync + Send, E> Drop for ObjectExpose<T, E> {
	fn drop(&mut self) {
		self.contract.destroy();
	}
}

pub(super) struct Assignment {
	pub counter: u64,
	pub assigned: bool,
}

pub(super) struct ObjectExposeContract {
	//TODO: unwrap from Option, when destroy can be done less explicitly
	local_node: Option<LocalNode>,
	object_core: Arc<ObjectCore>,
	composition_cost: u32,
	assignment: RwLock<Assignment>,
	changes: EventSource<ObjectDescriptor>,
}

impl ObjectExposeContract {
	pub(super) fn new(local_node: LocalNode, object_core: Arc<ObjectCore>, composition_cost: u32) -> Arc<ObjectExposeContract> {
		let contract = Arc::new(ObjectExposeContract {
			changes: EventSource::new(object_core.descriptor.clone()),
			object_core,
			local_node: Some(local_node),
			composition_cost,
			assignment: RwLock::new(Assignment { counter: 0, assigned: false }),
		});
		contract.object_core.link_object_expose_contract(contract.clone());
		contract
	}

	pub(super) fn destroy(self: &Arc<Self>) {
		//TODO: is this retarded?
		let dyn_self: Arc<dyn ExposeContract> = self.clone();
		self.object_core.unlink_object_expose_contract(&dyn_self);
		drop(dyn_self);
		assert_eq!(Arc::strong_count(self), 1);
	}

	pub(super) fn changes(&self) -> EventSource<ObjectDescriptor> {
		self.changes.clone()
	}
}



impl Contract for ObjectExposeContract {
	fn id(&self) -> usize {
		(self as *const Self) as usize
	}

	fn node_core(&self) -> &Arc<NodeCore> {
		self.local_node.as_ref().unwrap().shared.node_core.as_ref().unwrap()
	}
}


impl ExposeContract for ObjectExposeContract {
	fn composition_cost(&self) -> u32 {
		self.composition_cost
	}

	fn assign(&self, _object_core: Arc<ObjectCore>) -> bool {
		let mut assignment = self.assignment.write().unwrap();
		if assignment.assigned {
			panic!("Double assign of ObjectExpose");
		};
		assignment.counter += 1;
		assignment.assigned = true;
		self.node_core().load.fetch_add(self.composition_cost(), Ordering::SeqCst); //TODO: maybe weaker?
		self.changes.change();
		true //TODO: do we need to check if we still exist?
	}

	fn unassign(&self, _object_descriptor: &ObjectDescriptor) -> bool {
		//TODO: panic if object doesn't match?
		let mut assignment = self.assignment.write().unwrap();
		if !assignment.assigned {
			panic!("Double unassign of ObjectExpose");
		}
		assignment.assigned = false;
		self.node_core().load.fetch_sub(self.composition_cost(), Ordering::SeqCst); //TODO: maybe weaker?
		self.changes.change(); //FORMER-BUG, REGRESSION-TEST-NEEDED, this was missing and no rust test detected it
		true //TODO: do we need to check if we still exist?
	}

	fn assignment(&self, _object_descriptor: &ObjectDescriptor) -> Option<u64> {
		//TODO: panic if object doesn't match?
		let assignment = self.assignment.read().unwrap();
		if assignment.assigned {
			Some(assignment.counter)
		} else {
			None
		}
	}
}


impl std::fmt::Debug for ObjectExposeContract {
	fn fmt(&self, formatter: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
		formatter.write_str(&format!("ObjectExposeContract:{}", self.object_core.descriptor))
	}
}
