use std::{cmp::{max, Ordering}, collections::{HashMap, HashSet}, hash::{Hash, Hasher}, sync::{Arc, Mutex, RwLock, RwLockWriteGuard}, time::SystemTime};

use instant::Instant;

use crate::{descriptor::ObjectDescriptor, diff, events::EventSource, expose_contract::ExposeContract, node::{core::NodeCore, LocalNodeShared}, observe_contract::ObserveContract, tag::core::TagCore};

/// `= Vec<i64>`
pub type ObjectVersion = Vec<i64>;
/// `= Vec<String>`
pub type ObjectType = Vec<String>;
/// `= Arc<Vec<u8>>`
pub type ObjectRawData = Arc<Vec<u8>>;

type VersionFromTo = (Option<ObjectVersion>, ObjectVersion);

pub(crate) struct ObjectCore {
	local_node: Arc<LocalNodeShared>,
	pub descriptor: ObjectDescriptor,
	pub(super) diffs: Mutex<HashMap<VersionFromTo, Arc<Mutex<Option<ObjectRawData>>>>>,
	pub(super) diff_states: Mutex<HashMap<ObjectVersion, Arc<Mutex<Option<diff::State>>>>>,
	tags: Vec<Arc<TagCore>>,
	state: RwLock<Option<ObjectState>>,
	links: RwLock<ObjectLinks>,
	changes: EventSource<ObjectDescriptor>,
}

pub(crate) struct ObjectLinks {
	observe_contracts: HashMap<Arc<NodeCore>, HashSet<Arc<dyn ObserveContract>>>,
	expose_contracts: HashMap<Arc<NodeCore>, HashSet<Arc<dyn ExposeContract>>>,
	active: bool,
	assigned_expose_contract: Option<Arc<dyn ExposeContract>>,
}

#[derive(Clone, Debug, PartialEq, Eq)]
pub(crate) enum Synchronized {
	//Never,
	LastAt(Instant), // data was exposed by somebody but "Now" conditions were not met. TODO: what should happen if newest version gets uploaded by non-current-exposer?
	Now,             // (some, not necessarily current?) data was exposed by currently assigned exposer while exposer was aware of the most recent assignment id
}

impl Synchronized {
	pub fn micros_ago(&self) -> u64 {
		match self {
			Synchronized::LastAt(timestamp) => max(1, (Instant::now().duration_since(*timestamp)).as_micros() as u64),
			Synchronized::Now => 0,
		}
	}
}


#[derive(Clone)]
pub(crate) struct ObjectState {
	pub raw: ObjectRawData,
	pub data_type: ObjectType,
	pub version: ObjectVersion,
	pub synchronized: Synchronized,
}


impl ObjectLinks {
	fn recalculate_active(&mut self, object_core: &Arc<ObjectCore>) -> bool {
		let mut has_tag_observe_contracts = false;
		let mut has_tag_expose_contracts = false;
		for tag in object_core.tags.iter() {
			let (tag_observe_contracts, tag_expose_contracts) = tag.has_observe_expose_contracts();
			has_tag_observe_contracts = has_tag_observe_contracts || tag_observe_contracts;
			has_tag_expose_contracts = has_tag_expose_contracts || tag_expose_contracts;
			if has_tag_observe_contracts && has_tag_expose_contracts {
				break;
			}
		}

		let previous_active = self.active;
		self.active = !self.observe_contracts.is_empty() && (!self.expose_contracts.is_empty() || has_tag_expose_contracts)
			|| !self.expose_contracts.is_empty() && (!self.observe_contracts.is_empty() || has_tag_observe_contracts);

		if self.active != previous_active {
			for tag in object_core.tags.iter() {
				if self.active {
					tag.active_insert(object_core.clone());
				} else {
					tag.active_remove(object_core);
				}
			}
		};

		if self.active && self.assigned_expose_contract.is_none() {
			self.expose_contract_assign(object_core);
		} else if !self.active && self.assigned_expose_contract.is_some() {
			self.current_expose_contract_unassign(object_core);
		};

		//this is always triggered, not only when if active has changed, because despite object not changing, the tags may have changed
		//maybe stakeholders should react to tags changes instead?
		object_core.changes.change();

		self.active != previous_active
	}

	fn expose_contract_assign(&mut self, object_core: &Arc<ObjectCore>) {
		//TODO if all observe contracts are on a single node, and that node also has an expose contract - strongly prefer composing there
		loop {
			let mut best_expected_score = (true, u32::MAX); // (downstream, expected load)
			let mut best_expose_contract = None;

			for expose_contracts in self.expose_contracts.values() {
				for candidate_expose_contract in expose_contracts {
					if let Some(load) = candidate_expose_contract.expected_proportional_load_with() {
						let candidate_score = (!candidate_expose_contract.node_core().upstream, load);
						if candidate_score < best_expected_score {
							best_expected_score = candidate_score;
							best_expose_contract = Some(candidate_expose_contract.clone() as Arc<dyn ExposeContract>);
						}
					}
				}
			}

			for tag in object_core.tags.iter() {
				for candidate_expose_contract in tag.expose_contracts() {
					if let Some(load) = candidate_expose_contract.expected_proportional_load_with() {
						let candidate_score = (!candidate_expose_contract.node_core().upstream, load);
						if candidate_score < best_expected_score {
							best_expected_score = candidate_score;
							best_expose_contract = Some(candidate_expose_contract.clone() as Arc<dyn ExposeContract>);
						}
					}
				}
			}
			//TODO: unassign before assignment?
			if let Some(selected_expose_contract) = &best_expose_contract {
				if selected_expose_contract.assign(object_core.clone()) {
					self.assigned_expose_contract = best_expose_contract;
					break;
				};
			} else {
				break;
			}
		}
	}

	fn potentially_reassign_to(&mut self, object_core: &Arc<ObjectCore>, candidate_contract: Arc<dyn ExposeContract>) {
		if let Some(current_expose_contract) = &self.assigned_expose_contract {
			let current_score = (!current_expose_contract.node_core().upstream, current_expose_contract.expected_proportional_load_without());
			let candidate_score = (!candidate_contract.node_core().upstream, candidate_contract.expected_proportional_load_without());
			if current_score > candidate_score && candidate_contract.assign(object_core.clone()) {
				Self::expose_contract_unassign(object_core, current_expose_contract);
				self.assigned_expose_contract = Some(candidate_contract);
			}
		} else if candidate_contract.assign(object_core.clone()) {
			self.assigned_expose_contract = Some(candidate_contract);
		}
	}

	fn expose_contract_unassign(object_core: &ObjectCore, contract_to_unassign: &Arc<dyn ExposeContract>) {
		contract_to_unassign.unassign(&object_core.descriptor);
		if let Some(object_state) = &mut *object_core.state.write().unwrap() {
			object_state.synchronized = Synchronized::LastAt(Instant::now());
		};
	}

	fn current_expose_contract_unassign(&mut self, object_core: &ObjectCore) -> bool {
		match self.assigned_expose_contract.as_ref() {
			Some(assigned_expose_contract) => {
				Self::expose_contract_unassign(object_core, assigned_expose_contract);
				self.assigned_expose_contract = None;
				true
			}
			None => false,
		}
	}
}

impl Hash for ObjectCore {
	fn hash<H: Hasher>(&self, state: &mut H) {
		self.descriptor.hash(state);
	}
}

impl PartialEq for ObjectCore {
	fn eq(&self, other: &Self) -> bool {
		self.descriptor == other.descriptor
	}
}

impl Eq for ObjectCore {}


impl ObjectCore {
	pub(crate) fn new(local_node: Arc<LocalNodeShared>, descriptor: ObjectDescriptor, tags: Vec<Arc<TagCore>>) -> ObjectCore {
		ObjectCore {
			local_node,
			descriptor: descriptor.clone(),
			state: RwLock::new(None),
			links: RwLock::new(ObjectLinks {
				observe_contracts: HashMap::new(),
				expose_contracts: HashMap::new(),
				active: false,
				assigned_expose_contract: None,
			}),
			tags,
			changes: EventSource::new(descriptor),
			diffs: Mutex::new(HashMap::new()),
			diff_states: Mutex::new(HashMap::new()),
		}
	}

	pub(crate) fn inspect(&self, links: Option<RwLockWriteGuard<ObjectLinks>>) -> String {
		let links = links.unwrap_or_else(|| self.links.write().unwrap());
		let state = self.state.read().unwrap();
		format!(
			"├─────\n│ Object: {:?}\n│ Active: {:?}\n│ Observe contracts: {:?}\n│ Expose contracts: {:?}\n│ Tags: {:?}\n│ Assigned: {:?}\n| Diffs stored: {}  diff states stored: {}\n",
			self.descriptor,
			links.active,
			links.observe_contracts.keys(),
			links.expose_contracts.keys(),
			self.tags,
			links.assigned_expose_contract.as_ref().map(|contract| contract.node_core()),
			self.diffs.lock().unwrap().len(),
			self.diff_states.lock().unwrap().len(),
		) + &match &*state {
			Some(data) => {
				format!("│ Data: {:?} {:?} {:?}B {:?}\n", data.version, data.data_type, data.raw.len(), data.synchronized)
			}
			None => "│ Data: None\n".to_string(),
		}
	}

	pub(super) fn link_object_observe_contract(self: &Arc<Self>, contract: Arc<dyn ObserveContract>) {
		let mut links = self.links.write().unwrap();
		links.observe_contracts.entry(contract.node_core().clone()).or_insert_with(HashSet::new).insert(contract.clone());
		links.recalculate_active(self);
		if links.observe_contracts.len() == 1 {
			self.changes.change();
		}
	}

	pub(super) fn link_object_expose_contract(self: &Arc<Self>, contract: Arc<dyn ExposeContract>) {
		let mut links = self.links.write().unwrap();
		links.expose_contracts.entry(contract.node_core().clone()).or_insert_with(HashSet::new).insert(contract.clone());
		links.potentially_reassign_to(self, contract.clone() as Arc<dyn ExposeContract>);
		links.recalculate_active(self);
		if links.expose_contracts.len() == 1 {
			self.changes.change(); //TODO: is this
		};
	}

	pub(super) fn unlink_object_expose_contract(self: &Arc<Self>, contract: &Arc<dyn ExposeContract>) {
		let mut links = self.links.write().unwrap();
		let node_expose_contracts = links.expose_contracts.get_mut(contract.node_core()).unwrap();
		node_expose_contracts.remove(contract);
		if node_expose_contracts.is_empty() {
			links.expose_contracts.remove(contract.node_core());
		};
		if let Some(ref expose_contract) = links.assigned_expose_contract {
			if expose_contract.id() == contract.id() {
				links.current_expose_contract_unassign(self);
			}
		};
		links.recalculate_active(self);
		//TODO: notify?
	}

	pub(super) fn unlink_object_observe_contract(self: &Arc<Self>, contract: &Arc<dyn ObserveContract>) {
		let mut links = self.links.write().unwrap();
		let node_observe_contracts = links.observe_contracts.get_mut(contract.node_core()).unwrap();
		node_observe_contracts.remove(contract);
		if node_observe_contracts.is_empty() {
			links.observe_contracts.remove(contract.node_core());
		};
		links.recalculate_active(self);
		//TODO: notify?
	}

	pub(crate) fn expose_contract_unassign(self: &Arc<Self>, contract: &Arc<dyn ExposeContract>) {
		let mut links = self.links.write().unwrap();
		if let Some(ref expose_contract) = links.assigned_expose_contract {
			if expose_contract.id() == contract.id() {
				links.current_expose_contract_unassign(self);
			}
		}
	}

	pub(crate) fn state_get(&self) -> Option<ObjectState> {
		let locked_state = self.state.read().unwrap();
		(*locked_state).clone()
	}

	pub(crate) fn version_get(&self) -> Option<ObjectVersion> {
		let locked_state = &self.state.read().unwrap();
		locked_state.as_ref().map(|data| data.version.clone())
	}

	#[allow(clippy::ptr_arg)]
	pub fn state_set_with_precalculated_synchronized(
		&self, mut locked_state: RwLockWriteGuard<Option<ObjectState>>, version: &ObjectVersion, data_type: ObjectType, raw: ObjectRawData,
		new_synchronized: Synchronized,
	) -> bool {
		let submitted_version_is = locked_state.as_ref().map(|state| version.cmp(&state.version)).unwrap_or(Ordering::Greater);
		if submitted_version_is == Ordering::Greater {
			*locked_state = Some(ObjectState { raw, data_type, version: version.clone(), synchronized: new_synchronized });
			self.changes.change();
			return true;
		};

		if let Some(state) = &mut *locked_state {
			if state.synchronized != new_synchronized {
				state.synchronized = new_synchronized;
				self.changes.change();
			}
		} else {
			panic!();
		}

		false
	}

	#[allow(clippy::ptr_arg)]
	pub(crate) fn state_set_with_node(
		&self, node: &Arc<NodeCore>, version: &ObjectVersion, data_type: ObjectType, raw: ObjectRawData, synchronized: Synchronized,
	) -> bool {
		let links = self.links.read().unwrap();
		let locked_state = self.state.write().unwrap();
		let new_synchronized = self.new_synchronized_state(
			Some(node.id()) == links.assigned_expose_contract.as_ref().map(|assigned| assigned.node_core().id()),
			&locked_state,
			synchronized,
		);
		self.state_set_with_precalculated_synchronized(locked_state, version, data_type, raw, new_synchronized)
	}

	#[allow(clippy::ptr_arg)]
	pub(crate) fn state_set_with_assignment_id(
		&self, version: &ObjectVersion, data_type: ObjectType, raw: ObjectRawData, assignment_id: u64, synchronized: Synchronized,
	) -> bool {
		let links = self.links.read().unwrap();
		let locked_state = self.state.write().unwrap();
		let new_synchronized = self.new_synchronized_state(
			Some(Some(assignment_id)) == links.assigned_expose_contract.as_ref().map(|assigned| assigned.assignment(&self.descriptor)),
			&locked_state,
			synchronized,
		);
		self.state_set_with_precalculated_synchronized(locked_state, version, data_type, raw, new_synchronized)
	}

	fn new_synchronized_state(
		&self, assigned: bool, locked_state: &RwLockWriteGuard<Option<ObjectState>>, proposed_synchronized: Synchronized,
	) -> Synchronized {
		//TODO: check if sync timestamp is not in the future
		match locked_state.as_ref().map(|state| state.synchronized.clone()) {
			None => match proposed_synchronized {
				Synchronized::LastAt(_new_timestamp) => proposed_synchronized,
				Synchronized::Now =>
					if assigned {
						Synchronized::Now
					} else {
						Synchronized::LastAt(Instant::now())
					},
			},
			Some(Synchronized::LastAt(old_timestamp)) => match proposed_synchronized {
				Synchronized::LastAt(new_timestamp) => Synchronized::LastAt(max(new_timestamp, old_timestamp)),
				Synchronized::Now =>
					if assigned {
						Synchronized::Now
					} else {
						Synchronized::LastAt(max(Instant::now(), old_timestamp)) //hmmm...
					},
			},
			Some(Synchronized::Now) =>
				if assigned {
					proposed_synchronized //TEST-MISSING: this branch did not exist, always leaving synchronized as Now. router failed to accept and propagate desynchronize coming from wasm downstream.
				} else {
					Synchronized::Now
				},
		}
	}

	pub(crate) fn desynchronize(&self, assignment_id: u64) {
		let links = self.links.read().unwrap();
		let mut locked_state = self.state.write().unwrap();
		if let Some(ref mut state) = *locked_state {
			if Some(Some(assignment_id)) == links.assigned_expose_contract.as_ref().map(|assigned| assigned.assignment(&self.descriptor)) {
				state.synchronized = Synchronized::LastAt(Instant::now());
				self.changes.change();
			}
		}
	}

	//TODO: should we check contract too, in addition to node?
	pub(crate) fn data_set(self: &Arc<Self>, node: &Arc<NodeCore>, data_type: ObjectType, raw: ObjectRawData) {
		let links = self.links.read().unwrap();
		let locked_state = self.state.write().unwrap();
		let version = Self::version_next(locked_state.as_ref().map(|state| state.version.clone()));
		let new_synchronized = self.new_synchronized_state(
			Some(node.id()) == links.assigned_expose_contract.as_ref().map(|assigned| assigned.node_core().id()),
			&locked_state,
			Synchronized::Now,
		);
		self.state_set_with_precalculated_synchronized(locked_state, &version, data_type, raw, new_synchronized);
	}

	//TODO: maybe create a PROPER Version type, and put this in it?
	fn version_next(previous: Option<ObjectVersion>) -> ObjectVersion {
		match previous {
			Some(mut elements) => {
				if let Some(last_number) = elements.pop() {
					elements.push(last_number + 1)
				} else {
					panic!();
				};
				elements
			}
			None => {
				let duration_since_epoch = SystemTime::now().duration_since(SystemTime::UNIX_EPOCH).expect("SystemTime before UNIX EPOCH!");
				vec![1, duration_since_epoch.as_secs() as i64, duration_since_epoch.subsec_nanos() as i64, 0]
			}
		}
	}

	pub(crate) fn synchronized(&self) -> Option<Synchronized> {
		self.state.read().unwrap().as_ref().map(|state| state.synchronized.clone())
	}

	pub(crate) fn tag_changed(self: &Arc<Self>) {
		self.links.write().unwrap().recalculate_active(self);
	}

	pub(crate) fn changes(&self) -> EventSource<ObjectDescriptor> {
		self.changes.clone()
	}

	pub(super) fn has_observer_other_than(self: &Arc<Self>, node: &Arc<NodeCore>, check_tags: bool) -> bool {
		let object_links = self.links.read().unwrap();
		object_links.observe_contracts.len() > 1
			|| (!object_links.observe_contracts.is_empty() && !object_links.observe_contracts.contains_key(node))
			|| (check_tags && self.tags.iter().any(|tag| tag.has_observer_other_than(node)))
	}

	pub(super) fn has_exposer_other_than(self: &Arc<Self>, node: &Arc<NodeCore>, check_tags: bool) -> bool {
		let object_links = self.links.read().unwrap();
		object_links.expose_contracts.len() > 1
			|| (!object_links.expose_contracts.is_empty() && !object_links.expose_contracts.contains_key(node))
			|| (check_tags && self.tags.iter().any(|tag| tag.has_exposer_other_than(node)))
	}

	pub(super) fn is_assigned_to_node(self: &Arc<Self>, node: &Arc<NodeCore>) -> bool {
		let object_links = self.links.read().unwrap();
		if let Some(ref expose_contract) = object_links.assigned_expose_contract {
			expose_contract.node_core() == node
		} else {
			false
		}
	}

	pub(super) fn is_assigned_to_node_other_than(self: &Arc<Self>, node: &Arc<NodeCore>) -> bool {
		let object_links = self.links.read().unwrap();
		if let Some(ref expose_contract) = object_links.assigned_expose_contract {
			expose_contract.node_core() != node
		} else {
			false
		}
	}

	pub(super) fn is_exposed_by_node(self: &Arc<Self>, node: &Arc<NodeCore>) -> bool {
		let object_links = self.links.read().unwrap();
		object_links.expose_contracts.contains_key(node) || self.tags.iter().any(|tag| tag.is_exposed_by_node(node))
	}

	pub(super) fn is_observed_by_node(self: &Arc<Self>, node: &Arc<NodeCore>) -> bool {
		let object_links = self.links.read().unwrap();
		object_links.observe_contracts.contains_key(node) || self.tags.iter().any(|tag| tag.is_observed_by_node(node))
	}

	pub(super) fn minimum_cost_excluding_node(self: &Arc<Self>, node: &Arc<NodeCore>) -> Option<u32> {
		let object_links = self.links.read().unwrap();

		let mut lowest_cost = None;

		for expose_contracts in object_links.expose_contracts.values() {
			for expose_contract in expose_contracts {
				if expose_contract.node_core() != node {
					let cost = expose_contract.composition_cost();
					if lowest_cost.is_none() || cost < lowest_cost.unwrap() {
						lowest_cost = Some(cost);
					}
				}
			}
		}

		for tag in self.tags.iter() {
			if let Some(cost) = tag.minimum_cost_excluding_node(node) {
				if lowest_cost.is_none() || cost < lowest_cost.unwrap() {
					lowest_cost = Some(cost);
				}
			}
		}

		lowest_cost
	}
}

impl Drop for ObjectCore {
	fn drop(&mut self) {
		self.local_node.object_release(&self.descriptor);
	}
}
