/*! Communication primitives; only for tests and custom transports
*/

use serde::{Deserialize, Serialize};

use crate::{object::core::{ObjectRawData, ObjectVersion}, ObjectType};

#[derive(Debug, Serialize, Deserialize, PartialEq, Clone)]
pub struct Message {
	pub sequence_number: u64,
	pub acknowledgment_number: u64,
	pub changes: Vec<Change>,
}


// this was just plain enum before, with struct-like fields.
// but serde doesn't seem to handle well missing fields in such enum values
// and it's less painful to create and mutate nested structs than such pseudo-structs
#[derive(Debug, Serialize, Deserialize, PartialEq, Clone)]
pub enum Change {
	Options(OptionsChange),
	Object(ObjectChange),
	Tag(TagChange),
}

#[derive(Debug, Serialize, Deserialize, PartialEq, Clone, Copy)]
pub enum Preference {
	PreferYes = 1,
	PreferNo = 2,
	NoPreference = 3,
}

#[derive(Debug, Serialize, Deserialize, PartialEq, Clone)]
pub struct OptionsChange {
	pub name: Option<String>,
	pub custom: Option<String>, // JSON
	pub expose_capacity: Option<u32>,
	pub expose_load_limit: Option<u32>,
	pub diff_request: Option<Preference>,
}


#[derive(Serialize, Deserialize, Eq, PartialEq, Clone)]
pub enum DataChange {
	Data { data: ObjectRawData },
	Diff { old_data_version: Option<ObjectVersion>, diff: ObjectRawData },
}

#[derive(Debug, Serialize, Deserialize, Eq, Clone)]
pub struct ObjectChange {
	pub descriptor_tags: Vec<String>,
	pub descriptor_json: String, // JSON/ObjectDescriptor
	pub observer: Option<bool>,
	pub exposer: Option<bool>,
	pub data: Option<DataChange>,
	pub data_type: Option<ObjectType>,
	pub data_version: Option<ObjectVersion>,
	pub data_last_synchronized_microseconds_ago: Option<u64>,
	pub cost: Option<u32>,
}

//ONLY FOR TESTS
impl PartialEq for ObjectChange {
	fn eq(&self, other: &Self) -> bool {
		self.descriptor_tags == other.descriptor_tags
			&& self.descriptor_json == other.descriptor_json
			&& self.observer == other.observer
			&& self.exposer == other.exposer
			&& self.data == other.data
			&& self.data_type == other.data_type
			&& self.data_version == other.data_version
			&& self.cost == other.cost
			&& ((self.data_last_synchronized_microseconds_ago.is_none() && other.data_last_synchronized_microseconds_ago.is_none())
				|| (self.data_last_synchronized_microseconds_ago.is_some()
					&& other.data_last_synchronized_microseconds_ago.is_some()
					&& ((self.data_last_synchronized_microseconds_ago.unwrap() == 0 && other.data_last_synchronized_microseconds_ago.unwrap() == 0)
						|| (self.data_last_synchronized_microseconds_ago.unwrap() != 0 && other.data_last_synchronized_microseconds_ago.unwrap() != 0)
							&& ((self.data_last_synchronized_microseconds_ago.unwrap() as i64
								- other.data_last_synchronized_microseconds_ago.unwrap() as i64)
								.abs() < 100000))))
	}
}


#[derive(Debug, Serialize, Deserialize, Eq, PartialEq, Clone)]
pub struct TagChange {
	pub descriptor_json: String,
	pub observer: Option<bool>,
	pub exposer: Option<bool>,
	pub cost: Option<u32>,
}

impl ObjectChange {
	pub fn is_empty(&self) -> bool {
		!(self.observer.is_some()
			|| self.exposer.is_some()
			|| self.data.is_some()
			|| self.data_type.is_some()
			|| self.data_version.is_some()
			|| self.data_last_synchronized_microseconds_ago.is_some()
			|| self.cost.is_some())
	}
}

impl TagChange {
	pub fn is_empty(&self) -> bool {
		!(self.observer.is_some() || self.exposer.is_some())
	}
}

#[derive(Debug)]
pub struct MessageSerializationError {
	pub message: String,
}

impl Message {
	pub fn serialize(&self) -> Result<Vec<u8>, MessageSerializationError> {
		bincode::serialize(self).map_err(|error| MessageSerializationError { message: error.to_string() })
	}

	pub fn deserialize(buffer: &[u8]) -> Result<Message, MessageSerializationError> {
		bincode::deserialize::<Message>(buffer).map_err(|error| MessageSerializationError { message: error.to_string() })
	}

	pub fn inspect(&self) -> String {
		let mut ret = format!("┌───── Message seq:{} ack:{} changes:\n", self.sequence_number, self.acknowledgment_number);
		for change in &self.changes {
			ret.push_str(change.inspect().as_str());
		}
		ret.push_str("└╼");
		ret
	}
}

impl Change {
	pub fn inspect(&self) -> String {
		match &self {
			Change::Options(change) => format!(
				"│ Options -> name: {:?}  custom: {:?}  expose_capacity: {:?}  expose_load_limit: {:?}  diff_request: {:?}\n",
				change.name, change.custom, change.expose_capacity, change.expose_load_limit, change.diff_request
			),
			Change::Object(change) => format!(
				"│ Object -> {:?} {:?}  observer: {:?}  exposer: {:?}  data_type: {:?}  data_version: {:?} data: {:?} cost: {:?} last_synchronized: {:?}\n",
				change.descriptor_tags,
				change.descriptor_json,
				change.observer,
				change.exposer,
				change.data_type,
				change.data_version,
				change.data,
				change.cost,
				change.data_last_synchronized_microseconds_ago
			),
			Change::Tag(change) =>
				format!("│ Tag -> {:?}  observer: {:?}  exposer: {:?}  cost: {:?}\n", change.descriptor_json, change.observer, change.exposer, change.cost),
		}
	}
}

impl std::fmt::Debug for DataChange {
	fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
		match self {
			DataChange::Data { data } => f.write_fmt(format_args!("Data:{}B", data.len())),
			DataChange::Diff { old_data_version: _, diff } => f.write_fmt(format_args!("Diff:{}B", diff.len())),
		}
	}
}

impl From<Option<bool>> for Preference {
	fn from(value: Option<bool>) -> Self {
		match value {
			Some(true) => Preference::PreferYes,
			Some(false) => Preference::PreferNo,
			None => Preference::NoPreference,
		}
	}
}

impl Preference {
	pub fn or(&self, other: Preference) -> Preference {
		match self {
			Preference::PreferYes | Preference::PreferNo => *self,
			Preference::NoPreference => other,
		}
	}

	pub fn unwrap_or(&self, default: bool) -> bool {
		match self {
			Preference::PreferYes => true,
			Preference::PreferNo => false,
			Preference::NoPreference => default,
		}
	}
}


#[cfg(test)]
mod tests {

	use std::sync::Arc;

	use super::*;

	#[test]
	fn can_serialize_and_deserialize_message() {
		let message = Message {
			sequence_number: 1,
			acknowledgment_number: 2,
			changes: vec![Change::Object(ObjectChange {
				descriptor_tags: vec!["{}".to_string()],
				descriptor_json: "{}".to_string(),
				observer: Some(true),
				exposer: Some(true),
				data: Some(DataChange::Data { data: Arc::new(b"aaaaa".to_vec()) }),
				data_type: None,
				data_version: None,
				data_last_synchronized_microseconds_ago: None,
				cost: None,
			})],
		};

		//JSON
		let serialized = serde_json::to_string(&message).unwrap();
		println!("{}", serialized);
		let deserialized: Message = serde_json::from_str(&serialized).unwrap();
		assert_eq!(message, deserialized);

		//YAML
		let serialized = serde_yaml::to_string(&message).unwrap();
		println!("{}", serialized);
		let deserialized: Message = serde_yaml::from_str(&serialized).unwrap();
		assert_eq!(message, deserialized);

		//Wire proto
		let serialized = message.serialize().unwrap();
		println!("{:?}", &serialized);
		let deserialized = Message::deserialize(&serialized).unwrap();
		assert_eq!(message, deserialized);
	}
}
