#![allow(clippy::not_unsafe_ptr_arg_deref)]

use std::{convert::TryFrom, ffi::{c_void, CString}, intrinsics::transmute, mem::forget, ops::DerefMut, sync::{Arc, Mutex}};

use super::FFIObjectDescriptor;
use crate::{events::{Callback, CallbackRegistry, Event, EventSource}, ffi::FFIError, object::{core::{ObjectRawData, ObjectType}, observe::ObjectObserveState}, LocalNode, ObjectDescriptor, ObjectObserve};

#[allow(clippy::ptr_arg)] //it's less powerful but more readable
pub(super) fn raw_deserialize(_data_type: &ObjectType, data: &ObjectRawData) -> Result<ObjectRawData, ()> {
	Ok(data.clone())
}


//TODO: implement Try for this and other results
//TODO: better error api
#[repr(C)]
pub struct FFIObjectObserveResult {
	pub error: FFIError,
	pub object_observe_pointer: *mut ObjectObserve<ObjectRawData, ()>,
}

#[no_mangle]
pub extern "C" fn hakuban_object_observe_new(local_node: *mut LocalNode, descriptor: FFIObjectDescriptor) -> FFIObjectObserveResult {
	let descriptor: ObjectDescriptor = match ObjectDescriptor::try_from(&descriptor) {
		Ok(descriptor) => descriptor,
		Err(error) => return FFIObjectObserveResult { error, object_observe_pointer: std::ptr::null_mut() },
	};
	let local_node = unsafe { Box::from_raw(local_node) };
	let object_observe_pointer = Box::into_raw(Box::new(local_node.object(descriptor).with_deserializer(raw_deserialize).observe()));
	forget(local_node);
	FFIObjectObserveResult { error: FFIError::None, object_observe_pointer }
}


#[no_mangle]
pub extern "C" fn hakuban_object_observe_drop(object_ptr: *mut ObjectObserve<ObjectRawData, ()>) {
	drop(unsafe { Box::from_raw(object_ptr) });
}


//Option's are for laziness
pub struct FFIObjectObserveState {
	pub(super) state: ObjectObserveState<Arc<Vec<u8>>>,
	pub(super) data_type_c_strings: Option<(Vec<CString>, Vec<*const i8>)>,
}



#[no_mangle]
pub extern "C" fn hakuban_object_observe_state_borrow(object_observe_pointer: *mut ObjectObserve<ObjectRawData, ()>) -> *const Mutex<FFIObjectObserveState> {
	let object_observe = unsafe { Box::from_raw(object_observe_pointer) };
	let ret = if let Some(state) = object_observe.object_state().unwrap() {
		Box::into_raw(Box::new(Mutex::new(FFIObjectObserveState { state, data_type_c_strings: None })))
	} else {
		std::ptr::null()
	};
	forget(object_observe);
	ret
}

#[no_mangle]
pub extern "C" fn hakuban_object_observe_state_return(state_mutex_ptr: *mut Mutex<FFIObjectObserveState>) {
	let state_mutex = unsafe { Box::from_raw(state_mutex_ptr) };
	drop(state_mutex);
}

#[no_mangle]
pub extern "C" fn hakuban_object_observe_state_get_synchronized(state_mutex_ptr: *mut Mutex<FFIObjectObserveState>) -> u64 {
	let state_mutex = unsafe { Box::from_raw(state_mutex_ptr) };
	let state = state_mutex.lock().unwrap();
	let ret = state.state.synchronized;
	drop(state);
	forget(state_mutex);
	ret
}

#[repr(C)]
pub struct FFIObjectObserveStateGetVersionResult {
	pub version_length: usize,
	pub version_elements: *const i64,
}

#[no_mangle]
pub extern "C" fn hakuban_object_observe_state_get_data_version(state_mutex_ptr: *mut Mutex<FFIObjectObserveState>) -> FFIObjectObserveStateGetVersionResult {
	let state_mutex = unsafe { Box::from_raw(state_mutex_ptr) };
	let state = state_mutex.lock().unwrap();
	let ret = FFIObjectObserveStateGetVersionResult { version_length: state.state.data_version.len(), version_elements: state.state.data_version.as_ptr() };
	drop(state);
	forget(state_mutex);
	ret
}

#[repr(C)]
pub struct FFIObjectObserveStateGetDataTypeResult {
	pub type_length: usize,
	pub type_elements: *const *const i8,
}


#[no_mangle]
pub extern "C" fn hakuban_object_observe_state_get_data_type(state_mutex_ptr: *mut Mutex<FFIObjectObserveState>) -> FFIObjectObserveStateGetDataTypeResult {
	let state_mutex = unsafe { Box::from_raw(state_mutex_ptr) };
	let mut state_lock = state_mutex.lock().unwrap();
	let state = state_lock.deref_mut();
	let type_elements = state.data_type_c_strings.get_or_insert_with(|| {
		let strings: Vec<CString> = state.state.data_type.iter().map(|rust_string| CString::new(rust_string.clone()).unwrap()).collect();
		let array = strings.iter().map(|cstring| cstring.as_ptr()).collect();
		(strings, array)
	});
	let ret = FFIObjectObserveStateGetDataTypeResult { type_length: state.state.data_type.len(), type_elements: type_elements.1.as_ptr() };
	drop(state_lock);
	forget(state_mutex);
	ret
}


#[repr(C)]
pub struct FFIObjectObserveStateGetDataResult {
	pub data_length: usize,
	pub data_bytes: *const u8,
}

#[no_mangle]
pub extern "C" fn hakuban_object_observe_state_get_data(state_mutex_ptr: *mut Mutex<FFIObjectObserveState>) -> FFIObjectObserveStateGetDataResult {
	let state_mutex = unsafe { Box::from_raw(state_mutex_ptr) };
	let state = state_mutex.lock().unwrap();
	let ret = FFIObjectObserveStateGetDataResult { data_length: state.state.data.len(), data_bytes: state.state.data.as_ptr() };
	drop(state);
	forget(state_mutex);
	ret
}


pub type ObjectEventCallbackFn = extern "C" fn(*const c_void, object_descriptor: FFIObjectDescriptor, action: u8);

pub(super) fn make_object_descriptor_callback(callback: ObjectEventCallbackFn, userdata: *const c_void) -> Box<impl Fn(Event<ObjectDescriptor>)> {
	let pointers_are_not_send: usize = userdata as usize;
	Box::new(move |event: Event<ObjectDescriptor>| {
		let ffi_descriptor: FFIObjectDescriptor = (&event.key).into();
		(callback)(unsafe { transmute(pointers_are_not_send) }, ffi_descriptor.clone(), event.action as u8);
		ffi_descriptor.free();
	})
}

#[no_mangle]
pub extern "C" fn hakuban_object_observe_events_get(
	object_observe_ptr: *mut ObjectObserve<ObjectRawData, ()>,
) -> *mut Box<dyn CallbackRegistry<Params = Event<ObjectDescriptor>>> {
	let object_observe = unsafe { Box::from_raw(object_observe_ptr) };
	let ret = Box::new(Box::new(object_observe.changes()) as Box<dyn CallbackRegistry<Params = Event<ObjectDescriptor>>>);
	forget(object_observe);
	Box::into_raw(ret)
}

#[no_mangle]
pub extern "C" fn hakuban_object_descriptor_events_callback_register(
	events_ptr: *mut Box<dyn CallbackRegistry<Params = Event<ObjectDescriptor>>>, callback: ObjectEventCallbackFn, userdata: *const c_void,
) -> *mut Box<dyn Callback<Event<ObjectDescriptor>>> {
	let events = unsafe { Box::from_raw(events_ptr) };
	let ret = Box::new(events.register(make_object_descriptor_callback(callback, userdata)));
	forget(events);
	Box::into_raw(ret)
}

#[no_mangle]
pub extern "C" fn hakuban_object_descriptor_events_callback_unregister(callback: *mut Box<dyn Callback<Event<ObjectDescriptor>>>) {
	drop(unsafe { Box::from_raw(callback) });
}

#[no_mangle]
pub extern "C" fn hakuban_object_descriptor_events_return(events_ptr: *mut EventSource<ObjectDescriptor>) {
	drop(unsafe { Box::from_raw(events_ptr) });
}
