/*!
Event delivery pipeline

Synchronous (callbacks, Iterator) and asynchronous (futures::Stream) event delivery pipeline.

TO-DOCUMENT:
* Pipeline diagram
* Reason
* CallbackRegistry
* Buffers logic

*/

// TODO: split this into multiple files

use std::{collections::HashMap, fmt::Debug, hash::Hash, sync::{Arc, Condvar, Mutex, RwLock}, task::Waker};


pub trait CallbackRegistry: Send + Sync {
	type Params;
	// Callback is dyn because otherwise storing different EventEmitters together doesn't work (dyn EventEmitter wants associated type of Callback defined)
	#[must_use = "register returns an object representing registered callback. the callback gets unregistered when that object is dropped. hold on to it or call forever() on it."]
	fn register(&self, function: Box<dyn Fn(Self::Params) + Send + Sync>) -> Box<dyn Callback<Self::Params>>;
	fn call(&self, params: Self::Params);
}

pub trait Callback<P>: Send + Sync + 'static {
	// self is boxed because: 1) we want to take over the value completely, not just borrow, and 2) we need unregister to be callable on dyn Callback, so we can't use plain self
	fn unregister(self: Box<Self>) -> Box<dyn Fn(P) + Send + Sync>;
	fn forever(self: Box<Self>);
}


//trait Key = Eq + Hash + Clone + Send + Sync + 'static;
pub trait Key: Eq + Hash + Clone + Send + Sync + Debug + 'static {}
impl<K: Eq + Hash + Clone + Send + Sync + Debug + 'static> Key for K {}

#[derive(Copy, Clone, Debug, PartialEq, Eq, serde::Deserialize, serde::Serialize)]
pub enum Action {
	Insert,
	Change,
	Remove,
}

#[derive(Clone, Debug, PartialEq, Eq, serde::Deserialize, serde::Serialize)] //deserialize is for integration tests, serialize if for wasm js binding
pub struct Event<K: Key> {
	pub key: K,
	pub action: Action,
}



pub trait EventEmitter: CallbackRegistry<Params = Event<Self::Key>> + 'static {
	type Key: Key;

	fn keys(&self) -> Vec<Self::Key>; //TODO: should return an iterator maybe?
}



struct AddressableSet<T> {
	sequence: usize,
	map: HashMap<usize, T>,
}

impl<T> AddressableSet<T> {
	pub fn new() -> AddressableSet<T> {
		AddressableSet { sequence: 0, map: HashMap::new() }
	}

	#[must_use]
	pub fn insert(&mut self, value: T) -> usize {
		self.sequence += 1;
		self.map.insert(self.sequence, value);
		self.sequence
	}

	pub fn iter(&self) -> impl Iterator<Item = &T> {
		self.map.values()
	}

	pub fn remove(&mut self, id: &usize) -> T {
		self.map.remove(id).unwrap()
	}
}


type EventCallbackFn<T> = Box<dyn Fn(Event<T>) + Sync + Send>;

struct EventSourceShared<K: Key> {
	functions: RwLock<AddressableSet<EventCallbackFn<K>>>,
	key: K,
}

#[derive(Clone)]
pub struct EventSource<K: Key> {
	shared: Arc<EventSourceShared<K>>,
}

pub struct EventSourceCallback<K: Key> {
	source: Option<EventSource<K>>,
	callback_id: usize,
	unregister_on_drop: bool,
}

impl<K: Key> EventSource<K> {
	pub fn new(key: K) -> EventSource<K> {
		EventSource { shared: Arc::new(EventSourceShared { functions: RwLock::new(AddressableSet::new()), key }) }
	}

	pub fn key(&self) -> &K {
		&self.shared.key
	}

	pub fn change(&self) {
		//println!("{:?}", Backtrace::new());
		self.call(Event { key: self.key().clone(), action: Action::Change });
	}
}

impl<K: Key> CallbackRegistry for EventSource<K> {
	type Params = Event<K>;

	#[must_use]
	fn register(&self, function: Box<dyn Fn(Event<K>) + Send + Sync>) -> Box<dyn Callback<Self::Params>> {
		(function)(Event { key: self.shared.key.clone(), action: Action::Insert });
		Box::new(EventSourceCallback {
			source: Some(self.clone()),
			callback_id: self.shared.functions.write().unwrap().insert(function),
			unregister_on_drop: true,
		})
	}

	fn call(&self, params: Self::Params) {
		for callback in self.shared.functions.read().unwrap().iter() {
			(callback)(params.clone());
		}
	}
}

impl<K: Key> Callback<Event<K>> for EventSourceCallback<K> {
	fn unregister(mut self: Box<Self>) -> Box<dyn Fn(Event<K>) + Send + Sync> {
		self.unregister_ref()
	}

	fn forever(mut self: Box<Self>) {
		self.unregister_on_drop = false;
	}
}

impl<K: Key> EventSourceCallback<K> {
	fn unregister_ref(&mut self) -> Box<dyn Fn(Event<K>) + Send + Sync> {
		let callbacks = self.source.take().unwrap();
		let function = callbacks.shared.functions.write().unwrap().remove(&self.callback_id);
		(function)(Event { key: callbacks.shared.key.clone(), action: Action::Remove });
		function
	}
}

impl<K: Key> Drop for EventSourceCallback<K> {
	fn drop(&mut self) {
		if self.unregister_on_drop && self.source.is_some() {
			let _ = self.unregister_ref();
		}
	}
}

impl<K: Key> std::fmt::Debug for EventSource<K> {
	fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
		f.debug_struct("Source").finish()
	}
}

impl<K: Key> EventEmitter for EventSource<K> {
	type Key = K;

	fn keys(&self) -> Vec<Self::Key> {
		vec![self.shared.key.clone()]
	}
}

impl<K: Key> IntoIterator for EventSource<K> {
	type IntoIter = EventIterator<K>;
	type Item = Event<K>;

	fn into_iter(self) -> Self::IntoIter {
		let buffer = EventIterator::new();
		buffer.input.include(self).forever();
		buffer
	}
}


struct EventMergeShared<K: Key> {
	functions: RwLock<AddressableSet<EventCallbackFn<K>>>,
	#[allow(clippy::type_complexity)] //ignoring for now. we'll have to rewrite this anyway to support auto-dropping of inputs.
	inputs: RwLock<AddressableSet<(Box<dyn EventEmitter<Key = K, Params = Event<K>>>, Box<dyn Callback<Event<K>>>)>>,
	keys: RwLock<HashMap<K, usize>>,
}

#[derive(Clone)]
pub struct EventMerge<K: Key> {
	shared: Arc<EventMergeShared<K>>,
}

pub struct EventMergeCallback<K: Key> {
	shared: Option<Arc<EventMergeShared<K>>>,
	callback_id: usize,
	unregister_on_drop: bool,
}

pub struct EventMergeInput<K: Key> {
	shared: Option<Arc<EventMergeShared<K>>>,
	input_id: usize,
	remove_on_drop: bool,
}

impl<K: Key> EventMerge<K> {
	pub fn new() -> EventMerge<K> {
		EventMerge {
			shared: Arc::new(EventMergeShared {
				functions: RwLock::new(AddressableSet::new()),
				inputs: RwLock::new(AddressableSet::new()),
				keys: RwLock::new(HashMap::new()),
			}),
		}
	}

	#[must_use]
	pub fn include<E: EventEmitter<Key = K, Params = Event<K>>>(&self, emitter: E) -> EventMergeInput<K> {
		let self_for_closure = self.clone();
		let token = emitter.register(Box::new(move |mut event| {
			let keys = &mut *self_for_closure.shared.keys.write().unwrap();
			let count = *keys.get(&event.key).unwrap_or(&0);
			match event.action {
				Action::Insert if count == 0 => {
					keys.insert(event.key.clone(), count + 1);
				}
				Action::Insert if count > 0 => {
					*keys.get_mut(&event.key).unwrap() = count + 1;
					event.action = Action::Change; //hmm
				}
				Action::Change if count == 0 => {
					panic!();
				}
				Action::Change if count > 0 => {}
				Action::Remove if count == 1 => {
					keys.remove(&event.key);
				}
				Action::Remove if count > 1 => {
					keys.insert(event.key.clone(), count - 1);
					event.action = Action::Change;
				}
				_ => unreachable!(),
			};
			self_for_closure.call(event)
		}));
		EventMergeInput {
			shared: Some(self.shared.clone()),
			input_id: self.shared.inputs.write().unwrap().insert((Box::new(emitter), token)),
			remove_on_drop: true,
		}
	}
}

impl<K: Key> Default for EventMerge<K> {
	fn default() -> Self {
		Self::new()
	}
}

impl<K: Key> EventMergeInput<K> {
	pub fn remove(mut self) -> Option<Box<dyn EventEmitter<Key = K, Params = Event<K>>>> {
		self.remove_ref()
	}

	pub fn forever(mut self) {
		self.remove_on_drop = false;
	}

	fn remove_ref(&mut self) -> Option<Box<dyn EventEmitter<Key = K, Params = Event<K>>>> {
		if let Some(shared) = self.shared.take() {
			let mut emitters = shared.inputs.write().unwrap();
			let (emitter, callback) = emitters.remove(&self.input_id);
			let _ = callback.unregister();
			Some(emitter)
		} else {
			None
		}
	}
}

impl<K: Key> Drop for EventMergeInput<K> {
	fn drop(&mut self) {
		if self.remove_on_drop && self.shared.is_some() {
			let _ = self.remove_ref();
		}
	}
}

impl<K: Key> CallbackRegistry for EventMerge<K> {
	type Params = Event<K>;

	#[must_use]
	fn register(&self, function: Box<dyn Fn(Event<K>) + Send + Sync>) -> Box<dyn Callback<Self::Params>> {
		for key in self.shared.keys.read().unwrap().keys() {
			(function)(Event { key: key.clone(), action: Action::Insert });
		}
		Box::new(EventMergeCallback {
			shared: Some(self.shared.clone()),
			callback_id: self.shared.functions.write().unwrap().insert(function),
			unregister_on_drop: true,
		})
	}

	fn call(&self, params: Self::Params) {
		for callback in self.shared.functions.read().unwrap().iter() {
			(callback)(params.clone());
		}
	}
}

impl<K: Key> Callback<Event<K>> for EventMergeCallback<K> {
	fn unregister(mut self: Box<Self>) -> Box<dyn Fn(Event<K>) + Send + Sync> {
		self.unregister_ref()
	}

	fn forever(mut self: Box<Self>) {
		self.unregister_on_drop = false;
	}
}

impl<K: Key> EventMergeCallback<K> {
	fn unregister_ref(&mut self) -> Box<dyn Fn(Event<K>) + Send + Sync> {
		let callbacks = self.shared.take().unwrap();
		let function = callbacks.functions.write().unwrap().remove(&self.callback_id);
		for key in callbacks.keys.read().unwrap().keys() {
			(function)(Event { key: key.clone(), action: Action::Remove });
		}
		function
	}
}

impl<K: Key> Drop for EventMergeCallback<K> {
	fn drop(&mut self) {
		if self.unregister_on_drop && self.shared.is_some() {
			let _ = self.unregister_ref();
		}
	}
}



impl<K: Key> EventEmitter for EventMerge<K> {
	type Key = K;

	fn keys(&self) -> Vec<Self::Key> {
		self.shared.keys.read().unwrap().keys().cloned().collect()
	}
}

impl<K: Key> std::fmt::Debug for EventMerge<K> {
	fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
		f.debug_struct("Merge").finish()
	}
}

impl<K: Key> IntoIterator for EventMerge<K> {
	type IntoIter = EventIterator<K>;
	type Item = Event<K>;

	fn into_iter(self) -> Self::IntoIter {
		let buffer = EventIterator::new();
		buffer.input.include(self).forever();
		buffer
	}
}


struct EventMapShared<I: Key, O: Key, E: EventEmitter<Key = I, Params = Event<I>>, F: Fn(I) -> O + 'static + Sync + Send> {
	functions: RwLock<AddressableSet<EventCallbackFn<O>>>,
	pub input: E, //weird that this is exposed. should we deref instead?
	mapper: F,
}

#[derive(Clone)]
pub struct EventMap<I: Key, O: Key, E: EventEmitter<Key = I, Params = Event<I>>, F: Fn(I) -> O + 'static + Sync + Send> {
	shared: Arc<EventMapShared<I, O, E, F>>,
}

pub struct EventMapCallback<I: Key, O: Key> {
	input_callback: Option<Box<dyn Callback<Event<I>>>>,
	function: Option<Arc<EventCallbackFn<O>>>, //arc box uh oh
	unregister_on_drop: bool,
}

impl<I: Key, O: Key, E: EventEmitter<Key = I, Params = Event<I>>, F: Fn(I) -> O + 'static + Sync + Send> EventEmitter for EventMap<I, O, E, F> {
	type Key = O;

	fn keys(&self) -> Vec<Self::Key> {
		self.shared.input.keys().into_iter().map(|input_key| (self.shared.mapper)(input_key)).collect()
	}
}

impl<I: Key, O: Key> EventMapCallback<I, O> {
	fn unregister_ref(&mut self) -> Box<dyn Fn(Event<O>) + Send + Sync> {
		drop(self.input_callback.take());
		Arc::try_unwrap(self.function.take().unwrap()).ok().unwrap()
	}
}

impl<I: Key, O: Key> Callback<Event<O>> for EventMapCallback<I, O> {
	fn unregister(mut self: Box<Self>) -> Box<dyn Fn(Event<O>) + Send + Sync> {
		self.unregister_ref()
	}

	fn forever(mut self: Box<Self>) {
		self.unregister_on_drop = false;
	}
}


impl<I: Key, O: Key> Drop for EventMapCallback<I, O> {
	fn drop(&mut self) {
		if self.unregister_on_drop && self.input_callback.is_some() {
			let _ = self.unregister_ref();
		}
	}
}

//TODO: drop of Map and others!

impl<I: Key, O: Key, E: EventEmitter<Key = I, Params = Event<I>>, F: Fn(I) -> O + 'static + Sync + Send> EventMap<I, O, E, F> {
	pub fn new(input: E, mapper: F) -> EventMap<I, O, E, F> {
		EventMap { shared: Arc::new(EventMapShared { functions: RwLock::new(AddressableSet::new()), input, mapper }) }
	}

	pub fn input(&self) -> &E {
		&self.shared.input
	}
}

impl<I: Key, O: Key, E: EventEmitter<Key = I, Params = Event<I>>, F: Fn(I) -> O + 'static + Sync + Send> CallbackRegistry for EventMap<I, O, E, F> {
	type Params = Event<O>;

	#[must_use]
	fn register(&self, function: Box<dyn Fn(Event<O>) + Send + Sync + 'static>) -> Box<dyn Callback<Self::Params>> {
		let function = Arc::new(function);
		let function_for_lambda = Arc::clone(&function);
		let mut callback = Box::new(EventMapCallback { input_callback: None, function: Some(function), unregister_on_drop: true });
		let shared_for_lambda = self.shared.clone();
		callback.input_callback = Some(
			self.shared
				.input
				.register(Box::new(move |event| (function_for_lambda)(Event { key: (shared_for_lambda.mapper)(event.key), action: event.action }))),
		);
		callback
	}

	fn call(&self, params: Self::Params) {
		for callback in self.shared.functions.read().unwrap().iter() {
			(callback)(params.clone());
		}
	}
}

impl<I: Key, O: Key, E: EventEmitter<Key = I, Params = Event<I>>, F: Fn(I) -> O + 'static + Sync + Send> IntoIterator for EventMap<I, O, E, F> {
	type IntoIter = EventIterator<O>;
	type Item = Event<O>;

	fn into_iter(self) -> Self::IntoIter {
		let buffer = EventIterator::new();
		buffer.input.include(self).forever();
		buffer
	}
}



trait EventBufferShared<K: Key> {
	fn events(&self) -> &Mutex<HashMap<K, Action>>;
	fn event(&self, event: Event<K>) -> bool {
		let mut notifications = self.events().lock().unwrap();
		let (wake, insert, remove) = if let Some(previous_action) = notifications.get_mut(&event.key) {
			(false, false, match (previous_action, &event.action) {
				(Action::Insert, Action::Insert) => panic!(),
				(Action::Insert, Action::Change) => false,
				(Action::Insert, Action::Remove) => true,
				(Action::Change, Action::Insert) => panic!(),
				(Action::Change, Action::Change) => false,
				(entry @ Action::Change, Action::Remove) => {
					*entry = Action::Remove;
					false
				}
				(entry @ Action::Remove, Action::Insert) => {
					*entry = Action::Change;
					false
				}
				(Action::Remove, Action::Change) => panic!(),
				(Action::Remove, Action::Remove) => panic!(),
			})
		} else {
			(true, true, false)
		};
		if remove {
			notifications.remove(&event.key);
		} else if insert {
			notifications.insert(event.key, event.action);
		}
		wake
	}
}

struct EventStreamShared<K: Key> {
	events: Mutex<HashMap<K, Action>>,
	waker: RwLock<Option<Waker>>,
}

impl<K: Key> EventBufferShared<K> for EventStreamShared<K> {
	fn events(&self) -> &Mutex<HashMap<K, Action>> {
		&self.events
	}
}

struct EventIteratorShared<K: Key> {
	events: Mutex<HashMap<K, Action>>,
	condvar: Condvar,
}

impl<K: Key> EventBufferShared<K> for EventIteratorShared<K> {
	fn events(&self) -> &Mutex<HashMap<K, Action>> {
		&self.events
	}
}

pub struct EventIterator<K: Key> {
	shared: Arc<EventIteratorShared<K>>,
	input: EventMerge<K>, // another exosed input
	_input_callback: Box<dyn Callback<Event<K>>>,
}

impl<K: Key> EventIterator<K> {
	pub fn new() -> EventIterator<K> {
		let shared = Arc::new(EventIteratorShared { events: Mutex::new(HashMap::new()), condvar: Condvar::new() });
		let shared_for_closure = shared.clone();
		let inputs = EventMerge::new();
		let inputs_callback = inputs.register(Box::new(move |event| {
			if shared_for_closure.event(event) {
				shared_for_closure.condvar.notify_one()
			}
		}));
		EventIterator { input: inputs, _input_callback: inputs_callback, shared }
	}

	pub fn input(&self) -> &EventMerge<K> {
		&self.input
	}

	pub fn clear(&self) {
		self.shared.events.lock().unwrap().clear();
	}

	pub fn ready(&self) -> Option<Event<K>> {
		let mut notifications = self.shared.events.lock().unwrap();
		if let Some(key) = notifications.keys().next().cloned() {
			let action = notifications.remove(&key).unwrap();
			Some(Event { key, action })
		} else {
			None
		}
	}
}

impl<K: Key> Default for EventIterator<K> {
	fn default() -> Self {
		Self::new()
	}
}

pub struct EventStream<K: Key> {
	shared: Arc<EventStreamShared<K>>,
	input: EventMerge<K>, // another exosed input
	_input_callback: Box<dyn Callback<Event<K>>>,
}

impl<K: Key> EventStream<K> {
	pub fn new() -> EventStream<K> {
		let shared = Arc::new(EventStreamShared { events: Mutex::new(HashMap::new()), waker: RwLock::new(None) });
		let shared_for_closure = shared.clone();
		let inputs = EventMerge::new();
		let inputs_callback = inputs.register(Box::new(move |event| {
			if shared_for_closure.event(event) {
				if let Some(waker) = shared_for_closure.waker.write().unwrap().take() {
					waker.wake();
				};
			}
		}));
		EventStream { input: inputs, _input_callback: inputs_callback, shared }
	}

	pub fn input(&self) -> &EventMerge<K> {
		&self.input
	}

	pub fn clear(&self) {
		self.shared.events.lock().unwrap().clear();
	}

	pub fn ready(&self) -> Option<Event<K>> {
		let mut notifications = self.shared.events.lock().unwrap();
		if let Some(key) = notifications.keys().next().cloned() {
			let action = notifications.remove(&key).unwrap();
			Some(Event { key, action })
		} else {
			None
		}
	}
}

impl<K: Key> Default for EventStream<K> {
	fn default() -> Self {
		Self::new()
	}
}



impl<K: Key, E: EventEmitter<Key = K> + Clone> From<E> for EventIterator<K> {
	fn from(emitter: E) -> Self {
		let buffer = EventIterator::new();
		buffer.input.include(emitter).forever();
		buffer
	}
}


impl<K: Key, E: EventEmitter<Key = K> + Clone> From<E> for EventStream<K> {
	fn from(emitter: E) -> Self {
		let buffer = EventStream::new();
		buffer.input.include(emitter).forever();
		buffer
	}
}


impl<K: Key> Iterator for EventIterator<K> {
	type Item = Event<K>;

	fn next(&mut self) -> Option<Self::Item> {
		let mut notifications = self.shared.events.lock().unwrap();
		while notifications.len() == 0 {
			notifications = self.shared.condvar.wait(notifications).unwrap();
		}
		let key = notifications.keys().next().cloned().unwrap();
		let action = notifications.remove(&key).unwrap();
		Some(Event { key, action })
	}
}


#[cfg(feature = "async")]
impl<K: Key> futures::stream::Stream for EventStream<K> {
	type Item = Event<K>;

	fn poll_next(self: std::pin::Pin<&mut Self>, cx: &mut std::task::Context<'_>) -> std::task::Poll<Option<Self::Item>> {
		let mut notifications = self.shared.events.lock().unwrap();
		if let Some(key) = notifications.keys().next().cloned() {
			*self.shared.waker.write().unwrap() = None;
			return std::task::Poll::Ready(Some(Event { action: notifications.remove(&key).unwrap(), key }));
		};
		*self.shared.waker.write().unwrap() = Some(cx.waker().clone());
		std::task::Poll::Pending
	}
}


#[cfg(test)]
mod tests {

	use futures::StreamExt;

	use super::*;


	fn assert_events<K: Key>(events_received: Vec<Event<K>>, events_expected: Vec<Event<K>>) {
		let events_received: HashMap<K, Event<K>> = events_received.into_iter().map(|event| (event.key.clone(), event)).collect();
		let events_expected: HashMap<K, Event<K>> = events_expected.into_iter().map(|event| (event.key.clone(), event)).collect();
		assert_eq!(events_expected.len(), events_received.len());
		for (key_expected, event_expected) in events_expected {
			assert_eq!(&event_expected, events_received.get(&key_expected).unwrap());
		}
	}


	#[tokio::test]
	async fn source_emits_events_into_own_async_buffer() {
		let source: EventSource<u8> = EventSource::new(1);
		let buffer: EventStream<u8> = source.clone().into();
		let mut stream = buffer.ready_chunks(2);
		let events = stream.next().await.unwrap();
		assert_events(events, vec![Event { key: 1, action: Action::Insert }]);
		source.change();
		let events = stream.next().await.unwrap();
		assert_events(events, vec![Event { key: 1, action: Action::Change }]);
	}

	#[test]
	fn source_emits_events_into_own_sync_buffer() {
		let source: EventSource<u8> = EventSource::new(1);
		let buffer: EventIterator<u8> = source.clone().into();
		let mut stream = buffer;
		let event = stream.next().unwrap();
		assert_events(vec![event], vec![Event { key: 1, action: Action::Insert }]);
		assert_eq!(stream.ready(), None);
		source.change();
		let event = stream.next().unwrap();
		assert_events(vec![event], vec![Event { key: 1, action: Action::Change }]);
		assert_eq!(stream.ready(), None);
	}
}
