import * as hakuban_js from "../pkg/hakuban.js"
import * as manager from "./manager.coffee"

logger_initialized = false

export initialize_wasm = (wasm)->
		await hakuban_js.default(wasm)

export logger_initialize = (log_level, skip_if_already_initialized=false) ->
		if logger_initialized and skip_if_already_initialized
			return true

		if 0 == (error = await hakuban_js.hakuban_logger_initialize(log_level))
			logger_initialized = true
		else
			console.error "Failed to initialize logger, error " + error  
			false


raise_if_error = (error_code)->
	if error_code != 0
		throw "Hakuban error: "+error_code


export JSON_serialize = (data_type, cooked)->
	{
		data_type: ["JSON"].concat(data_type)
		data: new TextEncoder().encode(JSON.stringify(cooked))
	}

export JSON_deserialize = (data_type, raw)-> 
	throw "Expected JSON data_type, got: "+data_type  if data_type.length == 0 or data_type[0] != "JSON"
	{
		data_type: data_type.slice(1)
		data: JSON.parse(new TextDecoder().decode(raw))
	}


export class ObjectDescriptor
	constructor: (tags, @json) -> @tags = ((if tag instanceof TagDescriptor then tag else new TagDescriptor(tag)) for tag in tags)
	as_json: ()->JSON.stringify(tags: @tags.map((tag)->tag.json), json: @json)
	hash: ()-> 
		if not @_hash?
			@_hash = @tags.map((tag)->tag.hash().toString()).sort().join(",") + "|"
			ret = hakuban_js.hakuban_json_hash(JSON.stringify(@json))
			raise_if_error(ret.error)
			@_hash += ret.hash.toString()
		@_hash
	


export class TagDescriptor
	constructor: (@json) -> null
	as_json: ()->JSON.stringify(@json)
	hash: ()-> 
		if not @_hash?
			ret = hakuban_js.hakuban_json_hash(JSON.stringify(@json))
			raise_if_error(ret.error)
			@_hash = ret.hash
		@_hash




LocalNode_finalization_registry = new FinalizationRegistry (pointer)=>
	hakuban_js.hakuban_local_node_drop(pointer)

export class LocalNode
	constructor: (@name="wasm") ->
		logger_initialize("warn", true)
		result = hakuban_js.hakuban_local_node_new(@name)
		raise_if_error(result.error)
		@pointer = result.local_node_pointer
		LocalNode_finalization_registry.register(@, @pointer, @)
		@with_default_serializer(JSON_serialize)
		@with_default_deserializer(JSON_deserialize)

	destroy: () ->
		hakuban_js.hakuban_local_node_drop(@pointer)
		LocalNode_finalization_registry.unregister(@)

	object: (tags, json) ->
		new ObjectBuilder(@, @default_serializer, @default_deserializer, tags, json)

	tag: (json) ->
		new TagBuilder(@, @default_serializer, @default_deserializer, json)

	with_default_serializer: (@default_serializer) -> @
	with_default_deserializer: (@default_deserializer) -> @


class ObjectBuilder
	constructor: (@local_node, @serializer, @deserializer, @tags, @json) -> null
	observe: () -> new ObjectObserve(@local_node, new ObjectDescriptor(@tags, @json), @deserializer)
	expose: () -> new ObjectExpose(@local_node, new ObjectDescriptor(@tags, @json), @serializer)
	with_serializer: (@serializer) -> @
	with_deserializer: (@deserializer) -> @

class TagBuilder
	constructor: (@local_node, @serializer, @deserializer, @json) -> null
	observe: () -> new TagObserve(@local_node, new TagDescriptor(@json), @deserializer)
	expose: () -> new TagExpose(@local_node, new TagDescriptor(@json), @serializer)
	with_serializer: (@serializer) -> @
	with_deserializer: (@deserializer) -> @



ObjectDescriptorEvents_finalization_registry = new FinalizationRegistry (pointer)=>
	hakuban_js.hakuban_object_descriptor_events_drop(pointer)

class ObjectDescriptorEvents

	constructor: (@pointer)->
		ObjectDescriptorEvents_finalization_registry.register(@, @pointer, @)

	wait: ()=>next()
	next: ()=>
		@wait_released = new Promise (@wait_release)=>null
		cancel_waiting_promise = new Promise (resolve)=> @cancel_waiting_resolve = resolve
		next_event_promise = hakuban_js.hakuban_object_descriptor_events_next(@pointer, cancel_waiting_promise)
		next_event_promise.cancel = (value)=>
			@cancel_waiting_resolve.resolve(value)
			await next_event_promise
		next_event_promise.then (value)=>
			@wait_release()
			@wait_released = null
			@cancel_waiting_resolve = null
			parsed = JSON.parse(value)
			{ action: parsed.action, descriptor: new ObjectDescriptor(parsed.key.tags, parsed.key.json) }

	drop: ()=>
		if @cancel_waiting_resolve?
			@cancel_waiting_resolve()
			await @wait_released
		ObjectDescriptorEvents_finalization_registry.unregister(@)
		hakuban_js.hakuban_object_descriptor_events_drop(@pointer)
		@pointer = null



ObjectObserve_finalization_registry = new FinalizationRegistry (pointer)=>
	hakuban_js.hakuban_object_observe_drop(pointer)

class ObjectObserve

	constructor: (@local_node, @descriptor, @deserializer) ->
		result = hakuban_js.hakuban_object_observe_new(@local_node.pointer, @descriptor.as_json())
		raise_if_error(result.error)
		@pointer = result.object_observe_pointer
		ObjectObserve_finalization_registry.register(@, @pointer, @)
		@dropped = false
	
	drop: ()=>
		@dropped = true
		hakuban_js.hakuban_object_observe_drop(@pointer)
		ObjectObserve_finalization_registry.unregister(@)
		@pointer = null

	object_state: ()=>
		return null  if not @pointer?
		state_pointer = hakuban_js.hakuban_object_observe_state_borrow(@pointer)
		return null  if state_pointer == 0
		synchronized = hakuban_js.hakuban_object_observe_state_synchronized(state_pointer)
		version = JSON.parse(hakuban_js.hakuban_object_observe_state_data_version(state_pointer))
		data_type = JSON.parse(hakuban_js.hakuban_object_observe_state_data_type(state_pointer))
		raw = hakuban_js.hakuban_object_observe_state_data(state_pointer)
		hakuban_js.hakuban_object_observe_state_return(state_pointer)
		deserialized = @deserializer(data_type, raw)
		{
			version: version
			data: deserialized.data
			data_type: deserialized.data_type
			synchronized: synchronized
		}

	events: ()=>
		new ObjectDescriptorEvents(hakuban_js.hakuban_object_observe_events_get(@pointer))

	async: (lambda)=>	
		class ObservedObject extends manager.ManagedObject
			state: ()->
				@contract?.object_state()
			data: ()->
				@contract?.object_state()?.data

		new manager.ObjectManager(@, ObservedObject, lambda)



ObjectExpose_finalization_registry = new FinalizationRegistry (pointer)=>
	hakuban_js.hakuban_object_expose_drop(pointer)

class ObjectExpose
	constructor: (@local_node, @descriptor, @serializer)->
		result = hakuban_js.hakuban_object_expose_new(@local_node.pointer, @descriptor.as_json())
		raise_if_error(result.error)
		@pointer = result.object_expose_pointer
		ObjectExpose_finalization_registry.register(@, @pointer, @)
		@dropped = false

	drop: ()=>
		@dropped = true
		hakuban_js.hakuban_object_expose_drop(@pointer)
		ObjectExpose_finalization_registry.unregister(@)

	set_object_state: (version, cooked_data, data_type=[], assignment=0)=>
		serialized = @serializer(data_type, cooked_data)
		assignment = BigInt(assignment)  if not (typeof assignment == 'bigint')
		result = hakuban_js.hakuban_object_expose_set_state(@pointer, JSON.stringify(version), JSON.stringify(serialized.data_type), serialized.data, assignment)
		raise_if_error(result.error)
		result.changed

	assignment: ()=>
		hakuban_js.hakuban_object_expose_assignment(@pointer)

	assigned: ()=>
		@assignment() > 0

	desynchronize: (assignment)=>
		assignment = BigInt(assignment)  if not (typeof assignment == 'bigint')
		hakuban_js.hakuban_object_expose_desynchronize(@pointer, assignment)

	events: ()=>
		new ObjectDescriptorEvents(hakuban_js.hakuban_object_expose_events_get(@pointer))

	async: (lambda)=>	
		class ExposedObject extends manager.ManagedObject
			constructor: (contract, descriptor)->
				super(contract, descriptor)
				@_assignment = contract.assignment()
			run: (handler)=>
				await super(handler)
				@contract?.desynchronize(@_assignment)
			do_change: (change)=>
				@_assignment = @contract?.assignment()
				super(change)
			assignment: ()=>
				@contract?.assignment()
			assigned: ()=>
				@contract?.assigned()
			set_state: (version, cooked_data, data_type=[])=>
				@contract?.set_object_state(version, cooked_data, data_type, @_assignment)
			set_data: (data)=>
				timestamp = new Date().getTime()
				@set_state([1, Math.floor(timestamp/1000), timestamp - Math.floor(timestamp/1000)*1000, 0], data)

		new manager.ObjectManager(@, ExposedObject, lambda)



TagObserve_finalization_registry = new FinalizationRegistry (pointer)=>
	hakuban_js.hakuban_tag_observe_drop(pointer)

class TagObserve
	constructor: (@local_node, @descriptor, @deserializer) ->
		result = hakuban_js.hakuban_tag_observe_new(@local_node.pointer, @descriptor.as_json())
		raise_if_error(result.error)
		@pointer = result.tag_observe_pointer
		TagObserve_finalization_registry.register(@, @pointer, @)
		@dropped = false
	
	drop: ()=>
		@dropped = true
		hakuban_js.hakuban_tag_observe_drop(@pointer)
		TagObserve_finalization_registry.unregister(@)

	object_descriptors: ()=>
		result = hakuban_js.hakuban_tag_observe_object_descriptors(@pointer)
		for descriptor_as_json in JSON.parse(result)
			new ObjectDescriptor(descriptor_as_json.tags, descriptor_as_json.json)

	object_state: (object_descriptor)=>
		result = hakuban_js.hakuban_tag_observe_object_state_borrow(@pointer, object_descriptor.as_json())
		return null  if result.error == 4
		return null  if result.state_pointer == 0
		synchronized = hakuban_js.hakuban_object_observe_state_synchronized(result.state_pointer)
		version = JSON.parse(hakuban_js.hakuban_object_observe_state_data_version(result.state_pointer))
		data_type = JSON.parse(hakuban_js.hakuban_object_observe_state_data_type(result.state_pointer))
		raw = hakuban_js.hakuban_object_observe_state_data(result.state_pointer)
		hakuban_js.hakuban_object_observe_state_return(result.state_pointer)
		deserialized = @deserializer(data_type, raw)
		{
			version: version
			data: deserialized.data
			data_type: deserialized.data_type
			synchronized: synchronized
		}

	events: ()->
		new ObjectDescriptorEvents(hakuban_js.hakuban_tag_observe_events_get(@pointer))

	async: (lambda)=>	
		class ObservedObject extends manager.ManagedObject
			state: ()->
				@contract?.object_state(@descriptor)
			data: ()->
				@contract?.object_state(@descriptor)?.data

		new manager.ObjectManager(@, ObservedObject, lambda)

	

TagExpose_finalization_registry = new FinalizationRegistry (pointer)=>
	hakuban_js.hakuban_tag_expose_drop(pointer)

class TagExpose
	constructor: (@local_node, @descriptor, @serializer)->
		result = hakuban_js.hakuban_tag_expose_new(@local_node.pointer, @descriptor.as_json())
		raise_if_error(result.error)
		@pointer = result.tag_expose_pointer
		TagExpose_finalization_registry.register(@, @pointer, @)
		@dropped = false
	
	drop: ()=>
		@dropped = true
		hakuban_js.hakuban_tag_expose_drop(@pointer)
		TagExpose_finalization_registry.unregister(@)

	object_descriptors: ()=>
		result = hakuban_js.hakuban_tag_expose_object_descriptors(@pointer)
		for descriptor_as_json in JSON.parse(result)
			new ObjectDescriptor(descriptor_as_json.tags, descriptor_as_json.json)

	set_object_state: (object_descriptor, version, cooked_data, data_type=[], assignment=0)=>
		serialized = @serializer(data_type, cooked_data)
		assignment = BigInt(assignment)  if not (typeof assignment == 'bigint')
		result = hakuban_js.hakuban_tag_expose_set_object_state(@pointer, object_descriptor.as_json(), JSON.stringify(version), JSON.stringify(serialized.data_type), serialized.data, assignment)
		raise_if_error(result.error)
		result.changed

	assignment: (object_descriptor)=>
		result = hakuban_js.hakuban_tag_expose_assignment(@pointer, object_descriptor.as_json())
		raise_if_error(result.error)
		result.assignment

	assigned: (object_descriptor)=>
		@assignment(object_descriptor) > 0

	desynchronize: (object_descriptor, assignment)=>
		assignment = BigInt(assignment)  if not (typeof assignment == 'bigint')
		error = hakuban_js.hakuban_tag_expose_desynchronize(@pointer, object_descriptor.as_json(), assignment)
		raise_if_error(error)

	events: ()=>
		new ObjectDescriptorEvents(hakuban_js.hakuban_tag_expose_events_get(@pointer))

	async: (lambda)=>	
		class ExposedObject extends manager.ManagedObject
			constructor: (contract, descriptor)->
				super(contract, descriptor)
				@_assignment = contract.assignment(@descriptor)
			run: (handler)=>
				await super(handler)
				@contract?.desynchronize(@descriptor, @_assignment)
			do_change: (change)=>
				@_assignment = @contract?.assignment(@descriptor)
				super(change)
			assignment: ()=>
				@contract?.assignment(@descriptor)
			assigned: ()=>
				@contract?.assigned(@descriptor)
			set_state: (version, cooked_data, data_type=[])=>
				@contract?.set_object_state(@descriptor, version, cooked_data, data_type, @_assignment)
			set_data: (data)=>
				timestamp = new Date().getTime()
				@set_state([1, Math.floor(timestamp/1000), timestamp - Math.floor(timestamp/1000)*1000, 0], data)

		new manager.ObjectManager(@, ExposedObject, lambda)



WebsocketConnector_finalization_registry = new FinalizationRegistry (pointer)=>
	hakuban_js.hakuban_remote_node_drop(pointer)

#TODO: hide private methods
#TODO: handle errors somehow
export class WebsocketConnector

	constructor: (@local_node, @uri, @options={})->
		@connected = false
		@socket = undefined
		@error = undefined
		@reconnection_timer = undefined
		@reconnect_on_close = true
		@ack_timer = undefined
		result = hakuban_js.hakuban_remote_node_new(@local_node.pointer, true, true, true)
		raise_if_error(result.error)		
		@remote_node_pointer = result.remote_node_pointer
		@event_loop_stop = new Promise (resolve) => @event_loop_stop_resolve = resolve
		@event_loop_promise = @event_loop()
		WebsocketConnector_finalization_registry.register(@, @remote_node_pointer, @)
		@connect()

	destroy: () ->
		@event_loop_stop_resolve(["stop"])
		await @event_loop_promise
		@reconnect_on_close = false
		if @keep_alive_interval
			clearInterval(@keep_alive_interval)  
		if @reconnection_timer?
			clearTimeout(@reconnection_timer)
			@reconnection_timer = null
		if @socket?
			@socket.onclose = null
			@socket.onerror = null
			@socket.onmessage = null
			@socket.close()
			@socket_onclose()
		if @ack_timer?
			clearTimeout(@ack_timer)
			@ack_timer = null
		hakuban_js.hakuban_remote_node_drop(@remote_node_pointer)
		WebsocketConnector_finalization_registry.unregister(@)

		
	connect: ()=>
		@reconnection_timer = undefined

		try
			@socket = new WebSocket(@uri)
		catch error
			@error =  { while: "connecting", error: error }
			@onstatuschange_proc(@)  if @onstatuschange_proc?
			return false

		@socket.binaryType = "arraybuffer"

		@socket.onopen = ()=>
			console.debug('Connected to hakuban remote node')  if @options.debug
			@connected = true
			@error = undefined
			@onstatuschange_proc(@)  if @onstatuschange_proc?
			bytes_to_send = hakuban_js.hakuban_remote_node_connected(@remote_node_pointer)
			@send(bytes_to_send)

		@socket.onclose = @socket_onclose = ()=>
			console.debug('Disconnected from hakuban remote node')  if @options.debug
			clearInterval(@keep_alive_interval)  if @keep_alive_interval
			@connected = false
			@socket = undefined
			hakuban_js.hakuban_remote_node_disconnected(@remote_node_pointer)
			@onstatuschange_proc(@)  if @onstatuschange_proc?
			clearTimeout(@reconnection_timer)  if @reconnection_timer?
			@reconnection_timer = null
			@reconnection_timer = setTimeout(@connect, 1000)  if @reconnect_on_close

		@socket.onerror = (error)=>
			console.debug('Hakuban socket error', error)  if @options.debug
			@error = { while: "connected", error: error }

		@socket.onmessage = (event) =>
			bytes_to_send = hakuban_js.hakuban_remote_node_received_message(@remote_node_pointer, new Uint8Array(event.data))
			@send(bytes_to_send)
			clearTimeout(@ack_timer)  if @ack_timer?
			@ack_timer = setTimeout(@ack, 1000)
			null

	ping: ()=>
		@socket.send(new Uint8Array())  if @socket

	send: (bytes_to_send)=>
		if bytes_to_send.length > 0
			@socket.send(bytes_to_send)
			clearInterval(@keep_alive_interval)  if @keep_alive_interval
			@keep_alive_interval = setInterval(@ping, 10000)

	ack: ()=>
		if @socket?
			bytes_to_send = hakuban_js.hakuban_remote_node_ack(@remote_node_pointer)
			@send(bytes_to_send)  if bytes_to_send.length > 0

	event_loop: ()->
		next_local_node_event = false
		cancel_waiting_resolve = null
		cancel_waiting_promise = new Promise (resolve) -> cancel_waiting_resolve = resolve
		while true
			next_local_node_event = hakuban_js.hakuban_remote_node_next_local_node_event(@remote_node_pointer, cancel_waiting_promise).then((event)->["local_node_event", event])  if not next_local_node_event
			[promise, event] = await Promise.any([@event_loop_stop, next_local_node_event])
			if promise == "local_node_event"
				next_local_node_event = false
				setTimeout(((event)=>
					if @connected
						bytes_to_send = hakuban_js.hakuban_remote_node_received_local_node_event(@remote_node_pointer, event)
						@send(bytes_to_send)
				), 0, event)
			else
				break
		if next_local_node_event
			cancel_waiting_resolve(true)
			await next_local_node_event

	onstatuschange: (proc)->
		@onstatuschange_proc = proc
		@onstatuschange_proc()  if @onstatuschange_proc?
		@
