use std::{collections::HashMap, sync::{atomic::Ordering, Arc}, time::Duration};

extern crate hakuban;

use async_tungstenite::{tokio::TokioAdapter, tungstenite, WebSocketStream};
use futures::{SinkExt, StreamExt};
use hakuban::{events::EventStream, message::Message, tokio::WebsocketConnector, LocalNode, ObjectDescriptor, ObjectRawData, ObjectType, TagDescriptor, TagExpose};
use tokio::{net::TcpListener, sync::mpsc, task::JoinHandle, time::timeout};
use tracing::Level;
use url::Url;

use crate::scene::*;

pub struct RealClient;

impl RealClient {
	pub fn spawn(scene: Arc<Scene>, name: String) -> (mpsc::Sender<Order>, JoinHandle<()>) {
		let (tx, mut rx) = mpsc::channel::<Order>(1000);
		let join_handle = tokio::spawn(async move {
			let proxy_port = crate::director::PORT.fetch_add(1, Ordering::SeqCst);
			let proxy_address = Url::parse(format!("ws://127.0.0.1:{}", proxy_port).as_str()).unwrap();
			let proxy_listener = TcpListener::bind(format!("127.0.0.1:{}", proxy_port)).await.unwrap(); //.await.expect("Couldn't bind listening socket"); //TODO handle error gracefully //TODO accept full url as address

			let mut observed_objects = HashMap::new();
			let mut observed_objects_event_buffers = HashMap::new();
			let mut exposed_objects = HashMap::new();
			let mut exposed_objects_event_buffers = HashMap::new();
			let mut observed_tags = HashMap::new();
			let mut observed_tags_object_event_buffers: HashMap<TagDescriptor, EventStream<ObjectDescriptor>> = HashMap::new();
			let mut exposed_tags: HashMap<TagDescriptor, TagExpose<serde_json::Value, serde_json::Error>> = HashMap::new();
			let mut exposed_tags_object_event_buffers: HashMap<TagDescriptor, EventStream<ObjectDescriptor>> = HashMap::new();

			let local_node = LocalNode::builder().with_name(&name).build();
			let mut client_stream: Option<WebSocketStream<TokioAdapter<tokio::net::TcpStream>>> = None;
			let mut target_stream = None;
			let mut target_address = None;
			let mut connector = None;
			//TODO: maybe verify these
			let sequence_number = 1;
			let acknowledgment_number = 0;

			let span = tracing::span!(Level::TRACE, "actor", name = name.as_str());

			while let Some(order) = rx.recv().await {
				match order {
					Order::Event(_frame, event, done) => {
						match event {
							TestAction::Connect { connect_to } => {
								connector = Some(WebsocketConnector::new(local_node.clone(), proxy_address.as_str()).unwrap());

								let (tcp_client_stream, _client_address) = proxy_listener.accept().await.unwrap();
								client_stream = Some(async_tungstenite::tokio::accept_async(tcp_client_stream).await.unwrap());

								target_address = Some(format!("ws://127.0.0.1:{}", scene.ports.read().unwrap().get(&connect_to.clone()).unwrap()));
								target_stream = Some(async_tungstenite::tokio::connect_async(target_address.as_ref().unwrap().clone()).await.unwrap().0);
							}
							TestAction::Disconnect => {
								drop(connector.take());
								match client_stream.as_mut().unwrap().next().await.unwrap().unwrap() {
									tungstenite::Message::Close(_) => {}
									message => panic!("Got message instea of Close: {:?}", message),
								}
								target_stream.take().unwrap().close(None).await.unwrap();
							}
							TestAction::NetPartition => {
								client_stream.take().unwrap().close(None).await.unwrap();
								target_stream.take().unwrap().close(None).await.unwrap();
							}
							TestAction::NetJoin => {
								let (tcp_client_stream, _client_address) = proxy_listener.accept().await.unwrap();
								client_stream = Some(async_tungstenite::tokio::accept_async(tcp_client_stream).await.unwrap());
								target_stream = Some(async_tungstenite::tokio::connect_async(target_address.as_ref().unwrap().clone()).await.unwrap().0);
							}
							TestAction::Send(expected_message) => match client_stream.as_mut().unwrap().next().await.unwrap() {
								Ok(tungstenite::Message::Binary(message_msgpack)) => {
									let client_changes = rmp_serde::from_read_ref::<Vec<u8>, Message>(&message_msgpack).unwrap().changes;
									assert_eq!(expected_message.changes.len(), client_changes.len());
									for (expected_change, client_change) in expected_message.changes.iter().zip(client_changes.iter()) {
										assert_eq!(
											expected_change, client_change,
											"\n❌ERROR: {} sent change different than expected. Left is expected, Right is received:",
											name
										);
									}
									let tungstenite_message = tungstenite::Message::Binary(
										rmp_serde::to_vec(&Message { changes: client_changes, sequence_number, acknowledgment_number }).unwrap(),
									);
									target_stream.as_mut().unwrap().send(tungstenite_message).await.unwrap();
								}
								Ok(message) => panic!("Unexpected message: {:?}", message),
								Err(error) => panic!("{:?}", error),
							},
							TestAction::SendSomeMessages { count } =>
								for _ in 0..count {
									match client_stream.as_mut().unwrap().next().await.unwrap() {
										Ok(tungstenite::Message::Binary(message_msgpack)) => {
											let client_changes = rmp_serde::from_read_ref::<Vec<u8>, Message>(&message_msgpack).unwrap().changes;
											let tungstenite_message = tungstenite::Message::Binary(
												rmp_serde::to_vec(&Message { changes: client_changes, sequence_number, acknowledgment_number }).unwrap(),
											);
											target_stream.as_mut().unwrap().send(tungstenite_message).await.unwrap();
										}
										Ok(message) => panic!("Unexpected message: {:?}", message),
										Err(error) => panic!("{:?}", error),
									};
								},
							TestAction::Receive(mut expected_messages) => {
								match target_stream {
									Some(ref mut stream) => {
										let mut to_send = vec![];
										while !expected_messages.is_empty() {
											tokio::select! {
												next_message = stream.next() => match next_message {
													Some(Ok(tungstenite::Message::Binary(message_msgpack))) => {
														let received_changes = rmp_serde::from_read_ref::<Vec<u8>, Message>(&message_msgpack).unwrap().changes;
														let found_at = expected_messages
															.iter()
															.enumerate()
															.find(|(_, expected_message)| {
																if expected_message.changes.len() != received_changes.len() {
																	return false
																}
																for (expected_change, target_change) in expected_message.changes.iter().zip(received_changes.iter()) {
																	if expected_change != target_change {
																		return false;
																	};
																}
																true
															})
															.unwrap_or_else(|| panic!("❌ERROR: {} received message different than expected.\nExpected:\n{:?}\nReceived:\n{:?}\n", name, expected_messages, received_changes))
															.0;
														expected_messages.remove(found_at);
														to_send.push(message_msgpack);
													},
													Some(Err(error)) => panic!("Connection error: {:?}", error),
													_ => panic!()
												},
												_ = tokio::time::sleep(Duration::from_secs_f32(1.0)) => {
													panic!("\n❌{} - TIMEOUT waiting for messages {:?}", name, expected_messages);
												}
											};
										}
										for message_msgpack in to_send {
											let tungstenite_message = tungstenite::Message::Binary(message_msgpack);
											client_stream.as_mut().unwrap().send(tungstenite_message).await.unwrap();
										}
									}
									None => panic!(),
								};
							}
							TestAction::ReceiveSomeMessages { count } =>
								for _ in 0..count {
									match target_stream {
										Some(ref mut stream) => {
											let mut to_send = vec![];
											match stream.next().await {
												Some(Ok(tungstenite::Message::Binary(message_msgpack))) => {
													to_send.push(message_msgpack);
												}
												Some(Err(error)) => panic!("Connection error: {:?}", error),
												_ => panic!(),
											}
											for message_msgpack in to_send {
												let tungstenite_message = tungstenite::Message::Binary(message_msgpack);
												client_stream.as_mut().unwrap().send(tungstenite_message).await.unwrap();
											}
										}
										None => panic!(),
									}
								},
							TestAction::GetDisconnected => loop {
								match timeout(Duration::from_secs(1), target_stream.as_mut().unwrap().next()).await {
									Ok(Some(Ok(tungstenite::Message::Ping(_)))) => {}
									Ok(Some(Ok(tungstenite::Message::Close(_)))) => {
										break;
									}
									Ok(Some(Ok(received))) =>
										panic!("\n❌ERROR: {} received message while expecting disconnect. message: {:?}", name, received),
									Ok(Some(Err(error))) => {
										panic!("\n❌ERROR: {} received error while expecting disconnect. error: {:?}", name, error)
									}
									Ok(None) => panic!(),
									Err(_) => panic!("\n❌ERROR: {} got timeout while waiting to get disconnected", &name),
								};
							},
							TestAction::ReceiveNothing(duration) =>
								match timeout(Duration::from_secs_f32(duration), target_stream.as_mut().unwrap().next()).await {
									Ok(Some(Ok(tungstenite::Message::Binary(received)))) => panic!(
										"\n❌ERROR: {} received message, while it should not. Message: {:?}",
										&name,
										rmp_serde::from_read_ref::<Vec<u8>, Message>(&received).unwrap().changes
									),
									Ok(Some(Ok(received))) => panic!("\n❌ERROR: {} received message, while it should not. Message: {:?}", &name, &received),
									Ok(Some(Err(error))) => panic!("\n❌ERROR: {} receive errored out, while it should not. Error: {:?}", &name, &error),
									Ok(None) => panic!(),
									Err(_) => {}
								},
							TestAction::SendNothing(duration) => match timeout(Duration::from_secs_f32(duration), client_stream.as_mut().unwrap().next()).await
							{
								Ok(Some(Ok(sent))) => panic!("\n❌ERROR: {} sent message, while it should not. Message: {:?}", &name, &sent),
								Ok(Some(Err(error))) => panic!("\n❌ERROR: {} send errored out, while it should not. Error: {:?}", &name, &error),
								Ok(None) => panic!(),
								Err(_) => {}
							},
							TestAction::Sleep(duration) => {
								tokio::time::sleep(Duration::from_secs_f32(duration)).await;
							}
							TestAction::ObjectObserveNew { descriptor } => {
								let _enter = span.enter();
								let object_builder = local_node.object(descriptor.clone()).with_deserializer(
									|data_type: &ObjectType, raw_data: &ObjectRawData| -> Result<serde_json::Value, serde_json::Error> {
										if *data_type != vec!["JSON".to_string()] {
											panic!()
										};
										serde_json::from_slice::<serde_json::Value>(raw_data)
									},
								);
								let object = object_builder.observe();
								let watch = EventStream::new();
								watch.input().include(object.changes()).forever();
								observed_objects.insert(descriptor.clone(), object);
								observed_objects_event_buffers.insert(descriptor, watch);
							}
							TestAction::ObjectExposeNew { descriptor } => {
								let _enter = span.enter();
								let object_builder = local_node.object(descriptor.clone()).with_serializer(
									|data: &serde_json::Value| -> Result<(ObjectType, ObjectRawData), serde_json::Error> {
										let serialized = serde_json::to_vec(data).unwrap();
										Ok((vec!["JSON".to_string()], Arc::new(serialized)))
									},
								);
								let object = object_builder.expose();
								let watch = EventStream::new();
								watch.input().include(object.changes()).forever();
								exposed_objects.insert(descriptor.clone(), object);
								exposed_objects_event_buffers.insert(descriptor, watch);
							}
							TestAction::ObjectObserveData { descriptor, data } => {
								let _enter = span.enter();
								let object = observed_objects.get(&descriptor).expect("Unknown descriptor. Broken test?");
								assert_eq!(object.object_data().unwrap().unwrap(), data);
							}
							TestAction::ObjectObserveEvent { action, descriptor, version, data, data_type, synchronized } => {
								let _enter = span.enter();
								let received_event =
									observed_objects_event_buffers.get_mut(&descriptor).expect("Descriptor unknown. Test error?").next().await.unwrap();
								assert_eq!(received_event.action, action);
								assert_eq!(received_event.key, descriptor);
								let received_key = received_event.key;
								if let Some(version) = version {
									let object = observed_objects.get(&received_key).unwrap();
									assert_eq!(version, object.object_state().unwrap().data_version.unwrap());
								};
								if let Some(data_type) = data_type {
									let object = observed_objects.get(&received_key).unwrap();
									assert_eq!(data_type, object.object_state().unwrap().data_type.unwrap());
								};
								if let Some(data) = data {
									let object = observed_objects.get(&received_key).unwrap();
									assert_eq!(data, object.object_state().unwrap().data.unwrap());
								};
								if let Some(synchronized) = synchronized {
									let object = observed_objects.get(&received_key).unwrap();
									assert_eq!(synchronized, object.synchronized());
								};
							}
							TestAction::TagObjectObserveEvent(events) => {
								let _enter = span.enter();
								let events = events.clone();
								let mut events_by_tag = HashMap::new();
								for event in &events {
									events_by_tag.entry(event.tag_descriptor.clone()).or_insert_with(Vec::new).push(event.clone());
								}
								let mut matched_events_count: usize = 0;
								for (tag, mut tag_events) in events_by_tag {
									println!("{} {:?}", tag, tag_events);
									while !tag_events.is_empty() {
										let received_event = observed_tags_object_event_buffers.get_mut(&tag).unwrap().next().await.unwrap();
										let received_key = received_event.key.clone();
										if let Some((i, _)) = tag_events.iter().enumerate().find(|(_i, expected_event)| {
											received_key == expected_event.object_descriptor && received_event.action == expected_event.action
										}) {
											tag_events.remove(i);
											matched_events_count += 1;
										} else {
											panic!();
										}
									}
								}
								assert_eq!(events.len(), matched_events_count);
								//TODO: verify stuff
							}
							TestAction::ObjectExposeEvent { action, descriptor, assigned } => {
								let _enter = span.enter();
								let received_event = exposed_objects_event_buffers.get_mut(&descriptor).unwrap().next().await.unwrap();
								let received_key = received_event.key.clone();
								assert_eq!(received_event.action, action);
								if received_key != descriptor {
									panic!("wrong object changed. expected {:?}, changed {:?}", received_key, descriptor);
								};
								if let Some(assigned) = assigned {
									let object = exposed_objects.get(&received_key).unwrap();
									assert_eq!(assigned, object.assigned());
								};
							}
							TestAction::TagObjectExposeEvent(events) => {
								let _enter = span.enter();
								let events = events.clone();
								let mut events_by_tag = HashMap::new();
								for event in &events {
									events_by_tag.entry(event.tag_descriptor.clone()).or_insert_with(Vec::new).push(event.clone());
								}
								let mut matched_events_count: usize = 0;
								for (tag, mut tag_events) in events_by_tag {
									println!("{} {:?}", tag, tag_events);
									while !tag_events.is_empty() {
										let received_event = exposed_tags_object_event_buffers.get_mut(&tag).unwrap().next().await.unwrap();
										let received_key = received_event.key.clone();
										if let Some((i, _)) = tag_events.iter().enumerate().find(|(_i, expected_event)| {
											received_key == expected_event.object_descriptor && received_event.action == expected_event.action
										}) {
											tag_events.remove(i);
											matched_events_count += 1;
										} else {
											panic!();
										}
									}
								}
								assert_eq!(events.len(), matched_events_count);
							}
							TestAction::ObjectVersionExposed { .. } => {}
							TestAction::ObjectExposeNewVersion { descriptor, data_version, data, data_type } => {
								let _enter = span.enter();
								let object = exposed_objects.get(&descriptor).expect("Unknown descriptor. Broken test?");
								if data_type != ["JSON"] {
									panic!("we don't support any other serialization than JSON rn");
								};
								if let Some(data_version) = data_version {
									object.set_object_state(&data_version, &data).unwrap();
								} else {
									object.set_object_data(&data).unwrap();
								}
							}
							TestAction::ObjectExposeAssigned { descriptor, assigned } => {
								let _enter = span.enter();
								let object = exposed_objects.get(&descriptor).expect("Unknown descriptor. Broken test?");
								assert_eq!(assigned, object.assigned());
							}
							TestAction::TagObjectExposeNewVersion { tag, descriptor, data_version, data, data_type } => {
								let _enter = span.enter();
								let tag = exposed_tags.get(&tag).unwrap();
								if data_type != ["JSON"] {
									panic!("we don't support any other serialization than JSON rn");
								};
								tag.set_object_state(&descriptor, &data_version, &data).unwrap();
							}
							TestAction::TagObserveNew { descriptor } => {
								let _enter = span.enter();
								let tag_builder = local_node.tag(descriptor.clone()).with_deserializer(
									|data_type: &ObjectType, raw_data: &ObjectRawData| -> Result<serde_json::Value, serde_json::Error> {
										if *data_type != ["JSON"] {
											panic!()
										};
										serde_json::from_slice::<serde_json::Value>(raw_data)
									},
								);
								let tag = tag_builder.observe();
								let watch = EventStream::new();
								watch.input().include(tag.changes()).forever();
								observed_tags.insert(descriptor.clone(), tag);
								observed_tags_object_event_buffers.insert(descriptor, watch);
							}
							TestAction::TagExposeNew { descriptor } => {
								let _enter = span.enter();
								let tag_builder = local_node.tag(descriptor.clone()).with_serializer(
									|data: &serde_json::Value| -> Result<(ObjectType, ObjectRawData), serde_json::Error> {
										let serialized = serde_json::to_vec(data)?;
										Ok((vec!["JSON".to_string()], Arc::new(serialized)))
									},
								);
								let tag = tag_builder.expose();
								let watch = EventStream::new();
								watch.input().include(tag.changes()).forever();
								exposed_tags.insert(descriptor.clone(), tag);
								exposed_tags_object_event_buffers.insert(descriptor, watch);
							}
							TestAction::TagObserveData { tag_descriptor, object_descriptor, data } => {
								let _enter = span.enter();
								let object = observed_tags.get(&tag_descriptor).expect("Unknown descriptor. Broken test?");
								assert_eq!(object.object_data(&object_descriptor).unwrap().unwrap(), data);
							}
							TestAction::ObjectExposeData { descriptor, data } => {
								let _enter = span.enter();
								let object = exposed_objects.get(&descriptor).expect("Unknown descriptor. Broken test?");
								object.set_object_data(&data).unwrap();
							}
							TestAction::TagExposeData { tag_descriptor, object_descriptor, data } => {
								let _enter = span.enter();
								let tag = exposed_tags.get(&tag_descriptor).expect("Unknown descriptor. Broken test?");
								tag.set_object_data(&object_descriptor, &data).unwrap();
							}
							TestAction::ObjectExposeVersion { descriptor, data_version } => {
								let _enter = span.enter();
								let object = exposed_objects.get(&descriptor).expect("Unknown descriptor. Broken test?");
								assert!(object.set_object_version(data_version));
							}
							TestAction::TagExposeVersion { tag_descriptor, object_descriptor, data_version } => {
								let _enter = span.enter();
								let tag = exposed_tags.get(&tag_descriptor).expect("Unknown descriptor. Broken test?");
								assert!(tag.set_object_version(&object_descriptor, data_version));
							}
							TestAction::ObjectObserveVersion { descriptor, data_version } => {
								let _enter = span.enter();
								let object = observed_objects.get(&descriptor).expect("Unknown descriptor. Broken test?");
								assert_eq!(object.object_version().unwrap(), data_version);
							}
							TestAction::TagObserveVersion { tag_descriptor, object_descriptor, data_version } => {
								let _enter = span.enter();
								let tag = observed_tags.get(&tag_descriptor).expect("Unknown descriptor. Broken test?");
								assert_eq!(tag.object_version(&object_descriptor).unwrap(), data_version);
							}
							TestAction::ObjectObserveDrop { descriptor } => {
								let _enter = span.enter();
								observed_objects.remove(&descriptor);
								observed_objects_event_buffers.remove(&descriptor);
							}
							TestAction::ObjectExposeDrop { descriptor } => {
								let _enter = span.enter();
								exposed_objects.remove(&descriptor);
								exposed_objects_event_buffers.remove(&descriptor);
							}
							TestAction::TagObserveDrop { descriptor } => {
								let _enter = span.enter();
								observed_tags.remove(&descriptor);
								observed_tags_object_event_buffers.remove(&descriptor);
							}
							TestAction::TagExposeDrop { descriptor } => {
								let _enter = span.enter();
								exposed_tags.remove(&descriptor);
								exposed_tags_object_event_buffers.remove(&descriptor);
							}
							_ => panic!("RealClient can't do: {:?}", event),
						};
						done.send(Ok(())).unwrap();
					}
					Order::Exit => return,
				}
			}
		});
		(tx, join_handle)
	}
}
