#![allow(clippy::not_unsafe_ptr_arg_deref)]
use std::{mem::forget, sync::Arc};

use futures::StreamExt;
use log::error;
use wasm_bindgen::prelude::*;
use wasm_bindgen_futures::JsFuture;

use super::{raw_deserialize, WasmError};
use crate::{events::EventStream, object::{core::ObjectRawData, ObjectObserveState}, LocalNode, ObjectDescriptor, TagDescriptor, TagObserve};


#[wasm_bindgen]
pub struct WasmTagObserve {
	tag_observe: TagObserve<ObjectRawData, ()>,
	events: EventStream<ObjectDescriptor>,
}


#[wasm_bindgen]
pub struct WasmTagObserveResult {
	pub error: WasmError,
	pub tag_observe_pointer: *mut WasmTagObserve,
}


#[wasm_bindgen]
pub fn hakuban_tag_observe_new(local_node: *mut LocalNode, descriptor: String) -> WasmTagObserveResult {
	let descriptor: TagDescriptor = match serde_json::from_str::<TagDescriptor>(&descriptor) {
		Ok(descriptor) => descriptor,
		Err(error) => {
			error!("Invalid object descriptor json: {:?}", error);
			return WasmTagObserveResult { error: WasmError::InvalidJSON, tag_observe_pointer: std::ptr::null_mut() };
		}
	};
	let local_node = unsafe { Box::from_raw(local_node) };
	let tag_observe = local_node.tag(descriptor).with_deserializer(raw_deserialize).observe();
	let wasm_tag_observe_pointer = Box::into_raw(Box::new(WasmTagObserve { events: tag_observe.changes().into(), tag_observe }));
	forget(local_node);
	WasmTagObserveResult { error: WasmError::None, tag_observe_pointer: wasm_tag_observe_pointer }
}


#[wasm_bindgen]
pub fn hakuban_tag_observe_drop(wasm_tag_observe_pointer: *mut WasmTagObserve) {
	drop(unsafe { Box::from_raw(wasm_tag_observe_pointer) });
}



#[wasm_bindgen]
pub struct WasmTagObserveObjectStateBorrowResult {
	pub error: WasmError,
	pub state_pointer: *mut ObjectObserveState<Arc<Vec<u8>>>,
}


#[wasm_bindgen]
pub fn hakuban_tag_observe_object_state_borrow(
	wasm_tag_observe_pointer: *mut WasmTagObserve, object_descriptor: String,
) -> WasmTagObserveObjectStateBorrowResult {
	let object_descriptor: ObjectDescriptor = match serde_json::from_str::<ObjectDescriptor>(&object_descriptor) {
		Ok(descriptor) => descriptor,
		Err(error) => {
			error!("Invalid object descriptor json: {:?}", error);
			return WasmTagObserveObjectStateBorrowResult { error: WasmError::InvalidJSON, state_pointer: std::ptr::null_mut() };
		}
	};
	let wasm_tag_observe = unsafe { Box::from_raw(wasm_tag_observe_pointer) };
	let ret = match wasm_tag_observe.tag_observe.object_state(&object_descriptor).unwrap() {
		Some(state) => WasmTagObserveObjectStateBorrowResult { error: WasmError::None, state_pointer: Box::into_raw(Box::new(state)) },
		None => WasmTagObserveObjectStateBorrowResult { error: WasmError::ObjectNotFound, state_pointer: std::ptr::null_mut() },
	};
	forget(wasm_tag_observe);
	ret
}


#[wasm_bindgen]
pub fn hakuban_tag_observe_object_descriptors(wasm_tag_observe_pointer: *mut WasmTagObserve) -> String {
	let wasm_tag_observe = unsafe { Box::from_raw(wasm_tag_observe_pointer) };
	let descriptors: Vec<ObjectDescriptor> = wasm_tag_observe.tag_observe.object_descriptors().into_iter().collect();
	forget(wasm_tag_observe);
	serde_json::to_string(&descriptors).unwrap()
}



#[wasm_bindgen]
pub async fn hakuban_tag_observe_next_object_event(wasm_tag_observe_pointer: *mut WasmTagObserve, cancel: js_sys::Promise) -> JsValue {
	let cancel: JsFuture = cancel.into();
	let mut wasm_tag_observe = unsafe { Box::from_raw(wasm_tag_observe_pointer) };
	let ret = match futures::future::select(cancel, wasm_tag_observe.events.next()).await {
		futures::future::Either::Left((Ok(cancel), _)) => cancel,
		futures::future::Either::Left((Err(error), _)) => {
			error!("Error in cancel promise: {:?}", error);
			JsValue::NULL
		}
		futures::future::Either::Right((event, _)) => event.map(|event| serde_json::to_string(&event).unwrap()).into(),
	};
	forget(wasm_tag_observe);
	ret
}
