#![allow(clippy::not_unsafe_ptr_arg_deref)]
use std::mem::forget;

use futures::StreamExt;
use log::error;
use wasm_bindgen::prelude::*;
use wasm_bindgen_futures::JsFuture;

use super::{raw_serialize, WasmError};
use crate::{events::EventStream, LocalNode, ObjectDescriptor, TagDescriptor, TagExpose};



#[wasm_bindgen]
pub struct WasmTagExpose {
	tag_expose: TagExpose<(Vec<String>, Vec<u8>), ()>,
	events: EventStream<ObjectDescriptor>,
}


#[wasm_bindgen]
pub struct WasmTagExposeResult {
	pub error: WasmError,
	pub tag_expose_pointer: *mut WasmTagExpose,
}


#[wasm_bindgen]
pub fn hakuban_tag_expose_new(local_node: *mut LocalNode, descriptor: String) -> WasmTagExposeResult {
	let descriptor: TagDescriptor = match serde_json::from_str::<TagDescriptor>(&descriptor) {
		Ok(descriptor) => descriptor,
		Err(error) => {
			error!("Invalid object descriptor json: {:?}", error);
			return WasmTagExposeResult { error: WasmError::InvalidJSON, tag_expose_pointer: std::ptr::null_mut() };
		}
	};
	let local_node = unsafe { Box::from_raw(local_node) };
	let tag_expose = local_node.tag(descriptor).with_serializer(raw_serialize).expose();
	let wasm_tag_expose_pointer = Box::into_raw(Box::new(WasmTagExpose { events: tag_expose.changes().into(), tag_expose }));
	forget(local_node);
	WasmTagExposeResult { error: WasmError::None, tag_expose_pointer: wasm_tag_expose_pointer }
}

#[wasm_bindgen]
pub fn hakuban_tag_expose_drop(wasm_tag_expose_pointer: *mut WasmTagExpose) {
	drop(unsafe { Box::from_raw(wasm_tag_expose_pointer) });
}


#[wasm_bindgen]
pub struct WasmTagExposeObjectStateResult {
	pub error: WasmError,
	pub changed: bool,
}


#[wasm_bindgen]
pub fn hakuban_tag_expose_object_state(
	wasm_tag_expose_pointer: *mut WasmTagExpose, object_descriptor: String, data_version: String, data_type: String, data: Vec<u8>,
) -> WasmTagExposeObjectStateResult {
	let object_descriptor: ObjectDescriptor = match serde_json::from_str::<ObjectDescriptor>(&object_descriptor) {
		Ok(descriptor) => descriptor,
		Err(error) => {
			error!("Invalid object descriptor json: {:?}", error);
			return WasmTagExposeObjectStateResult { error: WasmError::InvalidJSON, changed: false };
		}
	};
	let data_version = match serde_json::from_str(&data_version) {
		Ok(state) => state,
		Err(error) => {
			error!("Can't parse data_version string as JSON: {:?}", error);
			return WasmTagExposeObjectStateResult { error: WasmError::InvalidJSON, changed: false };
		}
	};
	let data_type = match serde_json::from_str(&data_type) {
		Ok(state) => state,
		Err(error) => {
			error!("Can't parse data_type string as JSON: {:?}", error);
			return WasmTagExposeObjectStateResult { error: WasmError::InvalidJSON, changed: false };
		}
	};
	let cooked = (data_type, data);
	let wasm_tag_expose = unsafe { Box::from_raw(wasm_tag_expose_pointer) };
	let changed = wasm_tag_expose.tag_expose.set_object_state(&object_descriptor, &data_version, &cooked).unwrap();
	forget(wasm_tag_expose);
	forget(cooked.1);
	forget(data_version);
	WasmTagExposeObjectStateResult { error: WasmError::None, changed }
}


#[wasm_bindgen]
pub fn hakuban_tag_expose_object_descriptors(wasm_tag_expose_pointer: *mut WasmTagExpose) -> String {
	let wasm_tag_expose = unsafe { Box::from_raw(wasm_tag_expose_pointer) };
	let descriptors: Vec<ObjectDescriptor> = wasm_tag_expose.tag_expose.object_descriptors().into_iter().collect();
	forget(wasm_tag_expose);
	serde_json::to_string(&descriptors).unwrap()
}



#[wasm_bindgen]
pub async fn hakuban_tag_expose_next_object_event(wasm_tag_expose_pointer: *mut WasmTagExpose, cancel: js_sys::Promise) -> JsValue {
	let cancel: JsFuture = cancel.into();
	let mut wasm_tag_expose = unsafe { Box::from_raw(wasm_tag_expose_pointer) };
	let ret = match futures::future::select(cancel, wasm_tag_expose.events.next()).await {
		futures::future::Either::Left((Ok(cancel), _)) => cancel,
		futures::future::Either::Left((Err(error), _)) => {
			error!("Error in cancel promise: {:?}", error);
			JsValue::NULL
		}
		futures::future::Either::Right((event, _)) => event.map(|event| serde_json::to_string(&event).unwrap()).into(),
	};
	forget(wasm_tag_expose);
	ret
}
