#![allow(clippy::not_unsafe_ptr_arg_deref)]
use std::mem::forget;

use futures::StreamExt;
use log::{error, trace};
use wasm_bindgen::prelude::*;
use wasm_bindgen_futures::JsFuture;

use super::WasmError;
use crate::{events::{Event, EventStream}, message::Message, node::RemoteNode, Descriptor, LocalNode};


#[wasm_bindgen]
pub struct WasmRemoteNodeNewResult {
	pub error: WasmError,
	pub remote_node_pointer: *mut WasmRemoteNode,
}

#[wasm_bindgen]
pub struct WasmRemoteNode {
	remote_node: RemoteNode,
	local_node_events: EventStream<Descriptor>,
}


#[wasm_bindgen]
pub fn hakuban_remote_node_new(
	local_node_pointer: *mut LocalNode, upstream: bool, diff_produce: Option<bool>, diff_request: Option<bool>,
) -> WasmRemoteNodeNewResult {
	trace!("Constructing remote node");
	let local_node = unsafe { Box::from_raw(local_node_pointer) };
	let remote_node = RemoteNode::new(&local_node, upstream, diff_produce, diff_request);
	let wasm_remote_node = Box::new(WasmRemoteNode { local_node_events: remote_node.local_node_events.clone().into(), remote_node });
	forget(local_node);
	WasmRemoteNodeNewResult { error: WasmError::None, remote_node_pointer: Box::into_raw(wasm_remote_node) }
}


#[wasm_bindgen]
pub fn hakuban_remote_node_drop(remote_node_pointer: *mut WasmRemoteNode) {
	let wasm_remote_node = unsafe { Box::from_raw(remote_node_pointer) };
	trace!("Dropping remote node {:?}", wasm_remote_node.remote_node.name());
	drop(wasm_remote_node);
}

#[wasm_bindgen]
pub fn hakuban_remote_node_connected(remote_node_pointer: *mut WasmRemoteNode) -> Result<Vec<u8>, JsValue> {
	let wasm_remote_node = unsafe { Box::from_raw(remote_node_pointer) };
	let ret = match wasm_remote_node.remote_node.connected() {
		Ok(messages) => {
			let mut ret: Vec<u8> = vec![];
			for message in messages {
				ret.append(&mut rmp_serde::to_vec(&message).unwrap());
			}
			Ok(ret)
		}
		Err(error) => {
			let message = format!("Remote node connect fail: {:?}", error);
			error!("{}", message);
			Err(JsValue::from_str(&message))
		}
	};
	forget(wasm_remote_node);
	ret
}


#[wasm_bindgen]
pub fn hakuban_remote_node_disconnected(remote_node_pointer: *mut WasmRemoteNode) {
	let wasm_remote_node = unsafe { Box::from_raw(remote_node_pointer) };
	wasm_remote_node.remote_node.disconnected();
	forget(wasm_remote_node);
}


#[wasm_bindgen]
pub fn hakuban_remote_node_received_message(remote_node_pointer: *mut WasmRemoteNode, data: Vec<u8>) -> Result<Vec<u8>, JsValue> {
	let wasm_remote_node = unsafe { Box::from_raw(remote_node_pointer) };
	let ret = match rmp_serde::from_read_ref::<Vec<u8>, Message>(&data) {
		Ok(message) => match wasm_remote_node.remote_node.received_message(message) {
			Ok(messages) => {
				let mut ret: Vec<u8> = vec![];
				for message in messages {
					ret.append(&mut rmp_serde::to_vec(&message).unwrap());
				}
				Ok(ret)
			}
			Err(error) => {
				let message = format!("Error processing incomming message: {:?}", error);
				error!("{}", message);
				Err(JsValue::from_str(&message))
			}
		},
		Err(error) => {
			let message = format!("Can't parse message: {:?}", error);
			error!("{}", message);
			Err(JsValue::from_str(&message))
		}
	};
	forget(wasm_remote_node);
	ret
}


#[wasm_bindgen]
pub async fn hakuban_remote_node_next_local_node_event(remote_node_pointer: *mut WasmRemoteNode, cancel: js_sys::Promise) -> JsValue {
	let cancel: JsFuture = cancel.into();
	let mut wasm_remote_node = unsafe { Box::from_raw(remote_node_pointer) };
	let ret = match futures::future::select(cancel, wasm_remote_node.local_node_events.next()).await {
		futures::future::Either::Left((_cancel, _)) => None,
		futures::future::Either::Right((event, _)) => event.map(|event| serde_json::to_string(&event).unwrap()),
	};
	forget(wasm_remote_node);
	ret.into()
}


#[wasm_bindgen]
pub fn hakuban_remote_node_received_local_node_event(remote_node_pointer: *mut WasmRemoteNode, event: String) -> Result<Vec<u8>, JsValue> {
	let wasm_remote_node = unsafe { Box::from_raw(remote_node_pointer) };
	let ret = match serde_json::from_str::<Event<Descriptor>>(event.as_str()) {
		Ok(event) => match wasm_remote_node.remote_node.received_local_node_event(event) {
			Ok(messages) => {
				let mut ret: Vec<u8> = vec![];
				for message in messages {
					ret.append(&mut rmp_serde::to_vec(&message).unwrap());
				}
				Ok(ret)
			}
			Err(error) => {
				let message = format!("Error processing incomming message: {:?}", error);
				error!("{}", message);
				Err(JsValue::from_str(&message))
			}
		},
		Err(error) => {
			let message = format!("Can't parse event string: {:?}", error);
			error!("{}", message);
			Err(JsValue::from_str(&message))
		}
	};
	forget(wasm_remote_node);
	ret
}

#[wasm_bindgen]
pub fn hakuban_remote_node_ack(remote_node_pointer: *mut WasmRemoteNode) -> Vec<u8> {
	let wasm_remote_node = unsafe { Box::from_raw(remote_node_pointer) };
	let ret = if let Some(message) = wasm_remote_node.remote_node.ack() { rmp_serde::to_vec(&message).unwrap() } else { vec![] };
	forget(wasm_remote_node);
	ret
}
