#![allow(clippy::not_unsafe_ptr_arg_deref)]

use std::{mem::forget, sync::Arc};

use futures::StreamExt;
use log::error;
use wasm_bindgen::prelude::*;
use wasm_bindgen_futures::JsFuture;

use super::WasmError;
use crate::{events::EventStream, object::{core::{ObjectRawData, ObjectType}, observe::ObjectObserveState}, LocalNode, ObjectDescriptor, ObjectObserve};

#[allow(clippy::ptr_arg)] //it's less powerful but more readable
pub(super) fn raw_deserialize(_data_type: &ObjectType, data: &ObjectRawData) -> Result<ObjectRawData, ()> {
	Ok(data.clone())
}


#[wasm_bindgen]
pub struct WasmObjectObserve {
	object_observe: ObjectObserve<ObjectRawData, ()>,
	events: EventStream<ObjectDescriptor>,
}


#[wasm_bindgen]
pub struct WasmObjectObserveResult {
	pub error: WasmError,
	pub object_observe_pointer: *mut WasmObjectObserve,
}


#[wasm_bindgen]
pub fn hakuban_object_observe_new(local_node: *mut LocalNode, descriptor: String) -> WasmObjectObserveResult {
	let descriptor: ObjectDescriptor = match serde_json::from_str::<ObjectDescriptor>(&descriptor) {
		Ok(descriptor) => descriptor,
		Err(error) => {
			error!("Invalid object descriptor json: {:?}", error);
			return WasmObjectObserveResult { error: WasmError::InvalidJSON, object_observe_pointer: std::ptr::null_mut() };
		}
	};
	let local_node = unsafe { Box::from_raw(local_node) };
	let object_observe = local_node.object(descriptor).with_deserializer(raw_deserialize).observe();
	let wasm_object_observe_pointer = Box::into_raw(Box::new(WasmObjectObserve { events: object_observe.changes().into(), object_observe }));
	forget(local_node);
	WasmObjectObserveResult { error: WasmError::None, object_observe_pointer: wasm_object_observe_pointer }
}


#[wasm_bindgen]
pub fn hakuban_object_observe_drop(object_ptr: *mut WasmObjectObserve) {
	drop(unsafe { Box::from_raw(object_ptr) });
}


#[wasm_bindgen]
pub fn hakuban_object_observe_state_borrow(wasm_object_observe_pointer: *mut WasmObjectObserve) -> *const ObjectObserveState<Arc<Vec<u8>>> {
	let wasm_object_observe = unsafe { Box::from_raw(wasm_object_observe_pointer) };
	let state = Box::new(wasm_object_observe.object_observe.object_state().unwrap());
	forget(wasm_object_observe);
	Box::into_raw(state)
}

//TODO: synchronized

#[wasm_bindgen]
pub fn hakuban_object_observe_state_initialized(state_ptr: *mut ObjectObserveState<Arc<Vec<u8>>>) -> bool {
	let state = unsafe { Box::from_raw(state_ptr) };
	let ret = state.data.is_some();
	forget(state);
	ret
}

#[wasm_bindgen]
pub fn hakuban_object_observe_state_data_version(state_ptr: *mut ObjectObserveState<Arc<Vec<u8>>>) -> String {
	let state = unsafe { Box::from_raw(state_ptr) };
	let ret = serde_json::to_string(&state.data_version).unwrap();
	forget(state);
	ret
}

#[wasm_bindgen]
pub fn hakuban_object_observe_state_data_type(state_ptr: *mut ObjectObserveState<Arc<Vec<u8>>>) -> String {
	let state = unsafe { Box::from_raw(state_ptr) };
	let ret = serde_json::to_string(&state.data_type).unwrap();
	forget(state);
	ret
}

#[wasm_bindgen]
pub fn hakuban_object_observe_state_data(state_ptr: *mut ObjectObserveState<Arc<Vec<u8>>>) -> Vec<u8> {
	let state = unsafe { Box::from_raw(state_ptr) };
	if state.data.as_ref().unwrap().len() > u32::MAX as usize {
		error!("Can't handle over u32::MAX of data");
		panic!();
	}
	let ret = (**state.data.as_ref().unwrap()).clone();
	forget(state);
	ret
}

#[wasm_bindgen]
pub fn hakuban_object_observe_state_return(state_ptr: *mut ObjectObserveState<Arc<Vec<u8>>>) {
	let state = unsafe { Box::from_raw(state_ptr) };
	drop(state);
}

#[wasm_bindgen]
pub async fn hakuban_object_observe_next_event(wasm_object_observe_pointer: *mut WasmObjectObserve, cancel: js_sys::Promise) -> JsValue {
	let cancel: JsFuture = cancel.into();
	let mut wasm_object_observe = unsafe { Box::from_raw(wasm_object_observe_pointer) };
	let ret = match futures::future::select(cancel, wasm_object_observe.events.next()).await {
		futures::future::Either::Left((Ok(cancel), _)) => cancel,
		futures::future::Either::Left((Err(error), _)) => {
			error!("Error in cancel promise: {:?}", error);
			JsValue::NULL
		}
		futures::future::Either::Right((event, _)) => event.map(|event| serde_json::to_string(&event).unwrap()).into(),
	};
	forget(wasm_object_observe);
	ret
}
