#![allow(clippy::not_unsafe_ptr_arg_deref)]
use std::{mem::forget, sync::Arc};

use futures::StreamExt;
use log::error;
use wasm_bindgen::prelude::*;
use wasm_bindgen_futures::JsFuture;

use super::WasmError;
use crate::{events::EventStream, object::core::{ObjectRawData, ObjectType}, LocalNode, ObjectDescriptor, ObjectExpose};


pub(super) fn raw_serialize(cooked: &(ObjectType, Vec<u8>)) -> Result<(ObjectType, ObjectRawData), ()> {
	Ok((cooked.0.clone(), Arc::new(cooked.1.clone())))
}


#[wasm_bindgen]
pub struct WasmObjectExpose {
	object_expose: ObjectExpose<(Vec<String>, Vec<u8>), ()>,
	events: EventStream<ObjectDescriptor>,
}


#[wasm_bindgen]
pub struct WasmObjectExposeResult {
	pub error: WasmError,
	pub object_expose_pointer: *mut WasmObjectExpose,
}


#[wasm_bindgen]
pub fn hakuban_object_expose_new(local_node: *mut LocalNode, descriptor: String) -> WasmObjectExposeResult {
	let descriptor: ObjectDescriptor = match serde_json::from_str::<ObjectDescriptor>(&descriptor) {
		Ok(descriptor) => descriptor,
		Err(error) => {
			error!("Invalid object descriptor json: {:?}", error);
			return WasmObjectExposeResult { error: WasmError::InvalidJSON, object_expose_pointer: std::ptr::null_mut() };
		}
	};
	let local_node = unsafe { Box::from_raw(local_node) };
	let object_expose = local_node.object(descriptor).with_serializer(raw_serialize).expose();
	let wasm_object_expose_pointer = Box::into_raw(Box::new(WasmObjectExpose { events: object_expose.changes().into(), object_expose }));
	forget(local_node);
	WasmObjectExposeResult { error: WasmError::None, object_expose_pointer: wasm_object_expose_pointer }
}


#[wasm_bindgen]
pub fn hakuban_object_expose_drop(wasm_object_expose_pointer: *mut WasmObjectExpose) {
	drop(unsafe { Box::from_raw(wasm_object_expose_pointer) });
}


#[wasm_bindgen]
pub struct WasmObjectExposeStateResult {
	pub error: WasmError,
	pub changed: bool,
}


#[wasm_bindgen]
pub fn hakuban_object_expose_state(
	wasm_object_expose_pointer: *mut WasmObjectExpose, data_version: String, data_type: String, data: Vec<u8>,
) -> WasmObjectExposeStateResult {
	let data_version = match serde_json::from_str(&data_version) {
		Ok(state) => state,
		Err(error) => {
			error!("Can't parse data_version string as JSON: {:?}", error);
			return WasmObjectExposeStateResult { error: WasmError::InvalidJSON, changed: false };
		}
	};
	let data_type = match serde_json::from_str(&data_type) {
		Ok(state) => state,
		Err(error) => {
			error!("Can't parse data_type string as JSON: {:?}", error);
			return WasmObjectExposeStateResult { error: WasmError::InvalidJSON, changed: false };
		}
	};
	let cooked = (data_type, data);
	let wasm_object_expose = unsafe { Box::from_raw(wasm_object_expose_pointer) };
	let changed = wasm_object_expose.object_expose.set_object_state(&data_version, &cooked).unwrap();
	forget(wasm_object_expose);
	forget(cooked.1);
	forget(data_version);
	WasmObjectExposeStateResult { error: WasmError::None, changed }
}


#[wasm_bindgen]
pub fn hakuban_object_expose_assigned(wasm_object_expose_pointer: *mut WasmObjectExpose) -> bool {
	let wasm_object_expose = unsafe { Box::from_raw(wasm_object_expose_pointer) };
	let ret = wasm_object_expose.object_expose.assigned();
	forget(wasm_object_expose);
	ret
}


#[wasm_bindgen]
pub async fn hakuban_object_expose_next_event(wasm_object_expose_pointer: *mut WasmObjectExpose, cancel: js_sys::Promise) -> JsValue {
	let cancel: JsFuture = cancel.into();
	let mut wasm_object_expose = unsafe { Box::from_raw(wasm_object_expose_pointer) };
	let ret = match futures::future::select(cancel, wasm_object_expose.events.next()).await {
		futures::future::Either::Left((Ok(cancel), _)) => cancel,
		futures::future::Either::Left((Err(error), _)) => {
			error!("Error in cancel promise: {:?}", error);
			JsValue::NULL
		}
		futures::future::Either::Right((event, _)) => event.map(|event| serde_json::to_string(&event).unwrap()).into(),
	};
	forget(wasm_object_expose);
	ret
}
