use std::{marker::PhantomData, sync::Arc};

use super::{expose::{TagExpose, TagExposeContract}, observe::{TagObserve, TagObserveContract}};
use crate::{object::{builder::DefaultSerializerError, core::ObjectRawData, expose::ObjectSerializer, observe::ObjectDeserializer}, LocalNode, ObjectType, TagDescriptor};

#[must_use = "Call .expose() or .observe() at the end of the build chain."]
pub struct TagContractBuilder {
	local_node: LocalNode,
	descriptor: TagDescriptor,
}

#[must_use = "Call .expose() or .observe() at the end of the build chain."]
pub struct TagContractBuilderTyped<T: Sync + Send + 'static> {
	tag_contract_builder: TagContractBuilder,
	_object_data_type: std::marker::PhantomData<T>,
}

#[must_use = "Call .expose() or .observe() at the end of the build chain."]
pub struct TagContractBuilderWithDeserializer<T: Sync + Send + 'static, E> {
	typed_tag_contract_builder: TagContractBuilderTyped<T>,
	deserializer: ObjectDeserializer<T, E>,
}

#[must_use = "Call .expose() or .observe() at the end of the build chain."]
pub struct TagContractBuilderWithSerializer<T: Sync + Send + 'static, E> {
	typed_tag_contract_builder: TagContractBuilderTyped<T>,
	serializer: ObjectSerializer<T, E>,
}


impl TagContractBuilder {
	pub(crate) fn new(local_node: LocalNode, descriptor: TagDescriptor) -> TagContractBuilder {
		TagContractBuilder { local_node, descriptor }
	}

	pub fn with_objects_of_type<T: Sync + Send + 'static>(self) -> TagContractBuilderTyped<T> {
		TagContractBuilderTyped { tag_contract_builder: self, _object_data_type: PhantomData }
	}

	pub fn with_deserializer<T: Sync + Send + 'static, E, F: 'static + Send + Sync + Fn(&ObjectType, &ObjectRawData) -> Result<T, E>>(
		self, deserializer: F,
	) -> TagContractBuilderWithDeserializer<T, E> {
		TagContractBuilderWithDeserializer {
			typed_tag_contract_builder: TagContractBuilderTyped { tag_contract_builder: self, _object_data_type: std::marker::PhantomData },
			deserializer: Box::new(deserializer),
		}
	}

	pub fn with_serializer<T: Sync + Send + 'static, E, F: 'static + Send + Sync + Fn(&T) -> Result<(ObjectType, ObjectRawData), E>>(
		self, serializer: F,
	) -> TagContractBuilderWithSerializer<T, E> {
		TagContractBuilderWithSerializer {
			typed_tag_contract_builder: TagContractBuilderTyped { tag_contract_builder: self, _object_data_type: std::marker::PhantomData },
			serializer: Box::new(serializer),
		}
	}

	pub fn observe<T: Sync + Send + 'static + serde::de::DeserializeOwned>(self) -> TagObserve<T, DefaultSerializerError> {
		self.with_objects_of_type().observe()
	}

	pub fn expose<T: Sync + Send + serde::Serialize>(self) -> TagExpose<T, DefaultSerializerError> {
		self.with_objects_of_type().expose()
	}
}

impl<T: Sync + Send + 'static> TagContractBuilderTyped<T> {
	pub fn with_deserializer<E, F: 'static + Send + Sync + Fn(&ObjectType, &ObjectRawData) -> Result<T, E>>(
		self, deserializer: F,
	) -> TagContractBuilderWithDeserializer<T, E> {
		TagContractBuilderWithDeserializer { typed_tag_contract_builder: self, deserializer: Box::new(deserializer) }
	}

	pub fn with_serializer<E, F: 'static + Send + Sync + Fn(&T) -> Result<(ObjectType, ObjectRawData), E>>(
		self, serializer: F,
	) -> TagContractBuilderWithSerializer<T, E> {
		TagContractBuilderWithSerializer { typed_tag_contract_builder: self, serializer: Box::new(serializer) }
	}
}

impl<T: Sync + Send + 'static + serde::de::DeserializeOwned> TagContractBuilderTyped<T> {
	pub fn observe(self) -> TagObserve<T, DefaultSerializerError> {
		self.with_deserializer(|data_type: &ObjectType, raw_data: &ObjectRawData| -> Result<T, DefaultSerializerError> {
			if data_type.len() != 1 || data_type[0] != "JSON" {
				return Err(DefaultSerializerError::WrongDataType { data_type: data_type.clone() });
			}
			serde_json::from_slice::<T>(raw_data).map_err(DefaultSerializerError::SerdeJsonDeserializerError)
		})
		.observe()
	}
}

impl<T: Sync + Send + serde::Serialize> TagContractBuilderTyped<T> {
	pub fn expose(self) -> TagExpose<T, DefaultSerializerError> {
		self.with_serializer(|data: &T| -> Result<(ObjectType, ObjectRawData), DefaultSerializerError> {
			let mut serialized: Vec<u8> = vec![];
			data.serialize(&mut serde_json::Serializer::new(&mut serialized)).map_err(DefaultSerializerError::SerdeJsonSerializerError)?;
			Ok((vec!["JSON".to_string()], Arc::new(serialized)))
		})
		.expose()
	}
}

impl<T: Sync + Send + 'static, E> TagContractBuilderWithDeserializer<T, E> {
	pub fn observe(self) -> TagObserve<T, E> {
		let tag = self
			.typed_tag_contract_builder
			.tag_contract_builder
			.local_node
			.shared
			.tag_acquire(&self.typed_tag_contract_builder.tag_contract_builder.descriptor);
		let contract = TagObserveContract::new(self.typed_tag_contract_builder.tag_contract_builder.local_node, tag);
		TagObserve::new(contract, self.deserializer)
	}
}

impl<T: Sync + Send + 'static, E> TagContractBuilderWithSerializer<T, E> {
	pub fn expose(self) -> TagExpose<T, E> {
		let tag = self
			.typed_tag_contract_builder
			.tag_contract_builder
			.local_node
			.shared
			.tag_acquire(&self.typed_tag_contract_builder.tag_contract_builder.descriptor);
		let contract = TagExposeContract::new(self.typed_tag_contract_builder.tag_contract_builder.local_node, tag, 1000); //TODO function
		TagExpose::new(contract, self.serializer)
	}
}
