use std::{collections::{HashMap, HashSet}, hash::{Hash, Hasher}, sync::{Arc, Mutex, RwLock, RwLockWriteGuard}, time::SystemTime};

use crate::{descriptor::ObjectDescriptor, diff, events::EventSource, expose_contract::ExposeContract, node::{core::NodeCore, LocalNodeShared}, observe_contract::ObserveContract, tag::core::TagCore, TagDescriptor};

/// `= Vec<i64>`
pub type ObjectVersion = Vec<i64>;
/// `= Vec<String>`
pub type ObjectType = Vec<String>;
/// `= Arc<Vec<u8>>`
pub type ObjectRawData = Arc<Vec<u8>>;

type VersionFromTo = (Option<ObjectVersion>, ObjectVersion);

pub(crate) struct ObjectCore {
	local_node: Arc<LocalNodeShared>,
	pub descriptor: ObjectDescriptor,
	pub(super) diffs: Mutex<HashMap<VersionFromTo, Arc<Mutex<Option<ObjectRawData>>>>>,
	pub(super) diff_states: Mutex<HashMap<ObjectVersion, Arc<Mutex<Option<diff::State>>>>>,
	tags: HashMap<TagDescriptor, Arc<TagCore>>,
	state: RwLock<ObjectState>,
	links: RwLock<ObjectLinks>,
	changes: EventSource<ObjectDescriptor>,
}

pub(crate) struct ObjectLinks {
	observe_contracts: HashMap<Arc<NodeCore>, HashSet<Arc<dyn ObserveContract>>>,
	expose_contracts: HashMap<Arc<NodeCore>, HashSet<Arc<dyn ExposeContract>>>,
	active: bool,
	assigned_expose_contract: Option<Arc<dyn ExposeContract>>,
}

pub(crate) struct ObjectState {
	data: Option<RawData>,
	synchronized: bool,
}

struct RawData {
	raw: ObjectRawData,
	data_type: ObjectType,
	version: ObjectVersion,
}



impl ObjectLinks {
	fn recalculate_active(&mut self, object_core: &Arc<ObjectCore>) -> bool {
		let mut has_tag_observe_contracts = false;
		let mut has_tag_expose_contracts = false;
		for tag in object_core.tags.values() {
			let (tag_observe_contracts, tag_expose_contracts) = tag.has_observe_expose_contracts();
			has_tag_observe_contracts = has_tag_observe_contracts || tag_observe_contracts;
			has_tag_expose_contracts = has_tag_expose_contracts || tag_expose_contracts;
			if has_tag_observe_contracts && has_tag_expose_contracts {
				break;
			}
		}

		let previous_active = self.active;
		self.active = !self.observe_contracts.is_empty() && (!self.expose_contracts.is_empty() || has_tag_expose_contracts)
			|| !self.expose_contracts.is_empty() && (!self.observe_contracts.is_empty() || has_tag_observe_contracts);

		if self.active != previous_active {
			for tag in object_core.tags.values() {
				if self.active {
					tag.active_insert(object_core.clone());
				} else {
					tag.active_remove(object_core);
				}
			}
		};

		if self.active && self.assigned_expose_contract.is_none() {
			self.expose_contract_assign(object_core);
		} else if !self.active && self.assigned_expose_contract.is_some() {
			self.expose_contract_unassign(object_core);
		};

		//this is always triggered, not only when if active has changed, because despite object not changing, the tags may have changed
		//maybe stakeholders should react to tags changes instead?
		object_core.changes.change();

		self.active != previous_active
	}

	fn expose_contract_assign(&mut self, object_core: &Arc<ObjectCore>) {
		//TODO if all observe contracts are on a single node, and that node also has an expose contract - strongly prefer composing there
		loop {
			let mut best_expected_score = (true, u32::MAX); // (downstream, expected load)
			let mut best_expose_contract = None;

			for expose_contracts in self.expose_contracts.values() {
				for candidate_expose_contract in expose_contracts {
					if let Some(load) = candidate_expose_contract.expected_proportional_load_with() {
						let candidate_score = (!candidate_expose_contract.node_core().upstream, load);
						if candidate_score < best_expected_score {
							best_expected_score = candidate_score;
							best_expose_contract = Some(candidate_expose_contract.clone() as Arc<dyn ExposeContract>);
						}
					}
				}
			}

			for tag in object_core.tags.values() {
				for candidate_expose_contract in tag.expose_contracts() {
					if let Some(load) = candidate_expose_contract.expected_proportional_load_with() {
						let candidate_score = (!candidate_expose_contract.node_core().upstream, load);
						if candidate_score < best_expected_score {
							best_expected_score = candidate_score;
							best_expose_contract = Some(candidate_expose_contract.clone() as Arc<dyn ExposeContract>);
						}
					}
				}
			}
			//TODO: unassign before assignment?
			if let Some(selected_expose_contract) = &best_expose_contract {
				if selected_expose_contract.assign(object_core.clone()) {
					self.assigned_expose_contract = best_expose_contract;
					break;
				};
			} else {
				break;
			}
		}
	}

	fn potentially_reassign_to(&mut self, object_core: &Arc<ObjectCore>, candidate_contract: Arc<dyn ExposeContract>) {
		if let Some(current_expose_contract) = &self.assigned_expose_contract {
			let current_score = (!current_expose_contract.node_core().upstream, current_expose_contract.expected_proportional_load_without());
			let candidate_score = (!candidate_contract.node_core().upstream, candidate_contract.expected_proportional_load_without());
			if current_score > candidate_score && candidate_contract.assign(object_core.clone()) {
				current_expose_contract.unassign(&object_core.descriptor);
				self.assigned_expose_contract = Some(candidate_contract);
			}
		} else if candidate_contract.assign(object_core.clone()) {
			self.assigned_expose_contract = Some(candidate_contract);
		}
	}

	fn expose_contract_unassign(&mut self, object_core: &ObjectCore) -> bool {
		match &self.assigned_expose_contract {
			Some(assigned_expose_contract) => {
				assigned_expose_contract.unassign(&object_core.descriptor);
				self.assigned_expose_contract = None;
				object_core.state.write().unwrap().synchronized = false;
				true
			}
			None => false,
		}
	}

	//TODO: this already exists somewhere else
	fn is_assigned_to_node(&self, node: &Arc<NodeCore>) -> bool {
		if let Some(ref expose_contract) = self.assigned_expose_contract {
			expose_contract.node_core() == node
		} else {
			false
		}
	}
}

impl Hash for ObjectCore {
	fn hash<H: Hasher>(&self, state: &mut H) {
		self.descriptor.hash(state);
	}
}

impl PartialEq for ObjectCore {
	fn eq(&self, other: &Self) -> bool {
		self.descriptor == other.descriptor
	}
}

impl Eq for ObjectCore {}


impl ObjectCore {
	pub(crate) fn new(local_node: Arc<LocalNodeShared>, descriptor: ObjectDescriptor, tags: HashMap<TagDescriptor, Arc<TagCore>>) -> ObjectCore {
		ObjectCore {
			local_node,
			descriptor: descriptor.clone(),
			state: RwLock::new(ObjectState { data: None, synchronized: false }),
			links: RwLock::new(ObjectLinks {
				observe_contracts: HashMap::new(),
				expose_contracts: HashMap::new(),
				active: false,
				assigned_expose_contract: None,
			}),
			tags,
			changes: EventSource::new(descriptor),
			diffs: Mutex::new(HashMap::new()),
			diff_states: Mutex::new(HashMap::new()),
		}
	}

	pub(crate) fn inspect(&self, links: Option<RwLockWriteGuard<ObjectLinks>>) -> String {
		let links = links.unwrap_or_else(|| self.links.write().unwrap());
		let state = self.state.read().unwrap();
		format!(
			"├─────\n│ Object: {:?}\n│ Active: {:?}\n│ Observe contracts: {:?}\n│ Expose contracts: {:?}\n│ Tags: {:?}\n│ Assigned: {:?}\n│ Synchronized: {:?}\n| Diffs stored: {}  diff states stored: {}  Arc refs: {:?}\n",
			self.descriptor,
			links.active,
			links.observe_contracts.keys(),
			links.expose_contracts.keys(),
			self.tags,
			links.assigned_expose_contract,
			state.synchronized,
			self.diffs.lock().unwrap().len(),
			self.diff_states.lock().unwrap().len(),
			self.state.read().unwrap().data.as_ref().map(|raw_data| Arc::strong_count(&raw_data.raw))
		) + &match &state.data {
			Some(data) => {
				format!("│ Data: {:?} {:?} {:?}B\n", data.version, data.data_type, data.raw.len())
			}
			None => "│ Data: None\n".to_string(),
		}
	}

	pub(super) fn link_object_observe_contract(self: &Arc<Self>, contract: Arc<dyn ObserveContract>) {
		let mut links = self.links.write().unwrap();
		links.observe_contracts.entry(contract.node_core().clone()).or_insert_with(HashSet::new).insert(contract.clone());
		links.recalculate_active(self);
		if links.observe_contracts.len() == 1 {
			self.changes.change();
		}
	}

	pub(super) fn link_object_expose_contract(self: &Arc<Self>, contract: Arc<dyn ExposeContract>) {
		let mut links = self.links.write().unwrap();
		links.expose_contracts.entry(contract.node_core().clone()).or_insert_with(HashSet::new).insert(contract.clone());
		links.potentially_reassign_to(self, contract.clone() as Arc<dyn ExposeContract>);
		links.recalculate_active(self);
		if links.expose_contracts.len() == 1 {
			self.changes.change(); //TODO: is this
		};
	}

	pub(super) fn unlink_object_expose_contract(self: &Arc<Self>, contract: &Arc<dyn ExposeContract>) {
		let mut links = self.links.write().unwrap();
		let node_expose_contracts = links.expose_contracts.get_mut(contract.node_core()).unwrap();
		node_expose_contracts.remove(contract);
		if node_expose_contracts.is_empty() {
			links.expose_contracts.remove(contract.node_core());
		};
		if let Some(ref expose_contract) = links.assigned_expose_contract {
			if expose_contract.id() == contract.id() {
				links.expose_contract_unassign(self);
			}
		};
		links.recalculate_active(self);
		//TODO: notify?
	}

	pub(super) fn unlink_object_observe_contract(self: &Arc<Self>, contract: &Arc<dyn ObserveContract>) {
		let mut links = self.links.write().unwrap();
		let node_observe_contracts = links.observe_contracts.get_mut(contract.node_core()).unwrap();
		node_observe_contracts.remove(contract);
		if node_observe_contracts.is_empty() {
			links.observe_contracts.remove(contract.node_core());
		};
		links.recalculate_active(self);
		//TODO: notify?
	}

	pub(crate) fn expose_contract_unassign(self: &Arc<Self>, contract: &Arc<dyn ExposeContract>) {
		let mut links = self.links.write().unwrap();
		if let Some(ref expose_contract) = links.assigned_expose_contract {
			if expose_contract.id() == contract.id() {
				links.expose_contract_unassign(self);
			}
		}
	}

	pub(crate) fn state_get(&self) -> (bool, Option<ObjectVersion>, Option<ObjectType>, Option<ObjectRawData>) {
		let locked_state = &self.state.read().unwrap();
		if let Some(ref data) = locked_state.data {
			(locked_state.synchronized, Some(data.version.clone()), Some(data.data_type.clone()), Some(data.raw.clone()))
		} else {
			(locked_state.synchronized, None, None, None)
		}
	}

	pub(crate) fn version_get(&self) -> Option<ObjectVersion> {
		let locked_state = &self.state.read().unwrap();
		locked_state.data.as_ref().map(|data| data.version.clone())
	}

	#[allow(clippy::ptr_arg)]
	pub(crate) fn state_set(&self, node: &Arc<NodeCore>, version: &ObjectVersion, data_type: ObjectType, raw: ObjectRawData) -> bool {
		let links = self.links.read().unwrap();
		let mut locked_state = self.state.write().unwrap();
		let fresh = if let Some(current_version) = &locked_state.data { current_version.version < *version } else { true }; //TODO: prettify when let can be used with other conditions
		if fresh {
			locked_state.data = Some(RawData { raw, data_type, version: version.clone() });
			self.synchronize_if_assigned(&links, &mut locked_state, node);
			self.changes.change();
		}
		fresh
	}

	pub(crate) fn version_set(self: &Arc<Self>, node: &Arc<NodeCore>, mut version: Option<ObjectVersion>) -> bool {
		let links = self.links.read().unwrap();
		let mut locked_state = self.state.write().unwrap();
		//TODO: prettify when let can be used with other conditions
		let fresh = if let Some(new_version) = &version {
			if let Some(current_version) = &locked_state.data {
				current_version.version < *new_version
			} else {
				true
			}
		} else {
			let previous_version = locked_state.data.as_ref().map(|data| data.version.clone());
			version = Some(Self::version_next(previous_version));
			true
		};
		if fresh {
			if let Some(raw_data) = &mut locked_state.data {
				raw_data.version = version.unwrap();
			} else {
				panic!(); //TODO: maybe not? but what if not? error? return false?
			}
			self.synchronize_if_assigned(&links, &mut locked_state, node);
			self.changes.change();
		}
		fresh
	}

	pub(crate) fn data_set(self: &Arc<Self>, node: &Arc<NodeCore>, data_type: ObjectType, raw: ObjectRawData) {
		let links = self.links.read().unwrap();
		let mut locked_state = self.state.write().unwrap();
		let previous_version = locked_state.data.as_ref().map(|data| data.version.clone());
		let version = Self::version_next(previous_version);
		locked_state.data = Some(RawData { raw, data_type, version });
		self.synchronize_if_assigned(&links, &mut locked_state, node);
		self.changes.change();
	}

	//TODO: maybe create a PROPER Version type, and put this in it?
	fn version_next(previous: Option<ObjectVersion>) -> ObjectVersion {
		match previous {
			Some(mut elements) => {
				if let Some(last_number) = elements.pop() {
					elements.push(last_number + 1)
				} else {
					panic!();
				};
				elements
			}
			None => {
				let duration_since_epoch = SystemTime::now().duration_since(SystemTime::UNIX_EPOCH).expect("SystemTime before UNIX EPOCH!");
				vec![1, duration_since_epoch.as_secs() as i64, duration_since_epoch.subsec_nanos() as i64, 0]
			}
		}
	}

	pub(super) fn synchronize_if_assigned(&self, links: &ObjectLinks, state: &mut ObjectState, node: &Arc<NodeCore>) -> bool {
		let previous_synchronized = state.synchronized;
		if links.is_assigned_to_node(node) {
			state.synchronized = true;
		}
		state.synchronized != previous_synchronized
	}

	// only used in object/remote.rs
	pub(super) fn synchronize_if_assigned_and_version_matches(self: &Arc<Self>, node: &Arc<NodeCore>, version: &[i64]) {
		let links = self.links.read().unwrap();
		let mut locked_state = self.state.write().unwrap();
		if let Some(current_version) = &locked_state.data {
			if current_version.version == *version && self.synchronize_if_assigned(&links, &mut locked_state, node) {
				self.changes.change();
			}
		}
	}

	pub(crate) fn synchronized(&self) -> bool {
		self.state.read().unwrap().synchronized
	}

	pub(crate) fn tag_changed(self: &Arc<Self>) {
		self.links.write().unwrap().recalculate_active(self);
	}

	pub(crate) fn changes(&self) -> EventSource<ObjectDescriptor> {
		self.changes.clone()
	}

	pub(super) fn has_observer_other_than(self: &Arc<Self>, node: &Arc<NodeCore>, check_tags: bool) -> bool {
		let object_links = self.links.read().unwrap();
		object_links.observe_contracts.len() > 1
			|| (!object_links.observe_contracts.is_empty() && !object_links.observe_contracts.contains_key(node))
			|| (check_tags && self.tags.values().any(|tag| tag.has_observer_other_than(node)))
	}

	pub(super) fn has_exposer_other_than(self: &Arc<Self>, node: &Arc<NodeCore>, check_tags: bool) -> bool {
		let object_links = self.links.read().unwrap();
		object_links.expose_contracts.len() > 1
			|| (!object_links.expose_contracts.is_empty() && !object_links.expose_contracts.contains_key(node))
			|| (check_tags && self.tags.values().any(|tag| tag.has_exposer_other_than(node)))
	}

	pub(super) fn is_assigned_to_node(self: &Arc<Self>, node: &Arc<NodeCore>) -> bool {
		let object_links = self.links.read().unwrap();
		if let Some(ref expose_contract) = object_links.assigned_expose_contract {
			expose_contract.node_core() == node
		} else {
			false
		}
	}

	pub(super) fn is_assigned_to_node_other_than(self: &Arc<Self>, node: &Arc<NodeCore>) -> bool {
		let object_links = self.links.read().unwrap();
		if let Some(ref expose_contract) = object_links.assigned_expose_contract {
			expose_contract.node_core() != node
		} else {
			false
		}
	}

	pub(super) fn is_exposed_by_node(self: &Arc<Self>, node: &Arc<NodeCore>) -> bool {
		let object_links = self.links.read().unwrap();
		object_links.expose_contracts.contains_key(node) || self.tags.values().any(|tag| tag.is_exposed_by_node(node))
	}

	pub(super) fn is_observed_by_node(self: &Arc<Self>, node: &Arc<NodeCore>) -> bool {
		let object_links = self.links.read().unwrap();
		object_links.observe_contracts.contains_key(node) || self.tags.values().any(|tag| tag.is_observed_by_node(node))
	}

	pub(super) fn minimum_cost_excluding_node(self: &Arc<Self>, node: &Arc<NodeCore>) -> Option<u32> {
		let object_links = self.links.read().unwrap();

		let mut lowest_cost = u32::MAX;

		for expose_contracts in object_links.expose_contracts.values() {
			for expose_contract in expose_contracts {
				if expose_contract.node_core() != node {
					let cost = expose_contract.composition_cost();
					if cost < lowest_cost {
						lowest_cost = cost;
					}
				}
			}
		}

		for tag in self.tags.values() {
			if let Some(cost) = tag.minimum_cost_excluding_node(node) {
				if cost < lowest_cost {
					lowest_cost = cost;
				}
			}
		}

		if lowest_cost == u32::MAX {
			None
		} else {
			Some(lowest_cost)
		}
	}
}

impl Drop for ObjectCore {
	fn drop(&mut self) {
		self.local_node.object_release(&self.descriptor);
	}
}
