use std::{collections::{HashMap, VecDeque}, sync::{atomic::Ordering, Arc, Mutex}};

use log::trace;

use super::{core::NodeCore, local::LocalNodeShared};
use crate::{diff, events::{Action, Event, EventEmitter, EventMap, EventMerge, EventMergeInput}, message::{Change, DataChange, Message, ObjectChange, OptionsChange, TagChange}, object::{core::{ObjectRawData, ObjectVersion}, remote::RemoteObject}, tag::remote::RemoteTag, Descriptor, LocalNode, ObjectDescriptor, ObjectType, TagDescriptor};

/*
This file is a bit heavy with data structures, because there is a lot to keep track of.
There is stuff that remote side has sent to us, the stuff we sent to the remote side, and there are contracts and interfaces for the local node.
On top of that there is quite a lot code at the bottom, which is entirety of the networking logic (everything above the transport layer).
*/


//TODO: document how local_node and remote_node work
//TODO: change most unwraps and panics to error returns

//TODO: consider merging this with RemoteNode
pub(super) struct State {
	options: OptionsState,
	objects: HashMap<ObjectDescriptor, ObjectState>,
	tags: HashMap<TagDescriptor, TagState>,
	object_changes: Option<EventMerge<ObjectDescriptor>>,
	tag_changes: Option<EventMerge<TagDescriptor>>,
}

#[derive(Clone)]
pub(super) struct OptionsState {
	pub(super) diff_request: Option<bool>,
}

pub(super) struct ObjectState {
	pub(super) received: ReceivedRemoteObjectState,
	pub(super) sent: SentRemoteObjectState,
	pub(super) local: LocalObjectState,
}

pub(super) struct TagState {
	pub(super) received: RemoteTagState,
	pub(super) sent: RemoteTagState,
	pub(super) local: LocalTagState,
}

#[derive(Debug)]
struct ArchivedSentDataVersion {
	can_be_dropped_at_ack: u64, // only used for dropping archived versions from "sent" archive
	data_version: ObjectVersion,
	data_transmitted: bool, // used only for "sent"
	data_type: ObjectType,
	data: ObjectRawData,
}

#[derive(Debug)]
struct ArchivedReceivedDataVersion {
	data_version: ObjectVersion,
	data_type: Option<ObjectType>, // None if data_version was only declared by sender, without sending data
	data: Option<ObjectRawData>,   // None if remote side didn't send data and we don't know the data version submitted
}


pub(super) struct SentRemoteObjectState {
	pub(super) observed: bool,
	pub(super) exposed: bool,
	data: VecDeque<ArchivedSentDataVersion>, // newest versions on the front
	//TODO: this doesn't work
	pub(super) cost: Option<Option<u32>>,
}

pub(super) struct ReceivedRemoteObjectState {
	pub(super) observed: bool,
	pub(super) exposed: bool,
	data: VecDeque<ArchivedReceivedDataVersion>, // newest versions on the front
	//TODO: this doesn't work
	pub(super) cost: Option<Option<u32>>,
}

pub(super) struct RemoteTagState {
	pub(super) observed: bool,
	pub(super) exposed: bool,
	pub(super) cost: Option<Option<u32>>,
}

pub(super) struct LocalObjectState {
	pub(super) object_interface: RemoteObject,
	pub(super) changes_token: Option<EventMergeInput<ObjectDescriptor>>,
}

pub(super) struct LocalTagState {
	pub(super) tag_interface: RemoteTag,
	pub(super) changes_token: Option<EventMergeInput<TagDescriptor>>,
	pub(super) objects_changes_token: Option<EventMergeInput<ObjectDescriptor>>,
}



impl State {
	pub(super) fn object_maybe_borrow_mut(&mut self, descriptor: &ObjectDescriptor) -> Option<&mut ObjectState> {
		if self.objects.contains_key(descriptor) {
			Some(self.objects.get_mut(descriptor).unwrap())
		} else {
			None
		}
	}

	pub(super) fn object_borrow_mut(&mut self, descriptor: &ObjectDescriptor, local_node: &Arc<LocalNodeShared>, node: &Arc<NodeCore>) -> &mut ObjectState {
		if !self.objects.contains_key(descriptor) {
			let object_state = ObjectState {
				received: ReceivedRemoteObjectState { observed: false, exposed: false, data: VecDeque::new(), cost: None },
				sent: SentRemoteObjectState { observed: false, exposed: false, data: VecDeque::new(), cost: None },
				local: LocalObjectState { object_interface: RemoteObject::new(local_node.clone(), node.clone(), descriptor), changes_token: None },
			};
			self.objects.insert(descriptor.clone(), object_state);
		};
		self.objects.get_mut(descriptor).unwrap()
	}

	pub(super) fn object_return(&mut self, descriptor: &ObjectDescriptor, remote_is_upstream: bool) {
		// On DOWNSTREAM side:
		// If upstream sent us "observed" or "exposed" then we need to keep the object to retain corresponding observed/exposed contracts.
		// We also keep all objects which have downsteam-side observers or exposers, because we may have sent "highest known version" to upstream at contract announcement.
		//
		// On UPSTREAM side:
		// We keep all objects which are "observed" or "exposed" by the downstream side, to retain corresponding observed/exposed contracts
		// We also keep all objects that are relevant to the remote side, aka. "cross-active" objects - those with observer on one side and assigned exposer on the other side of the connection.
		//
		// On BOTH sides:
		// We keep all objects which were declared to be observed or exposed to the remote side.
		// There may be objects that are no longer interesting (== should otherwise be dropped), but remote side has not yet been informed about it.
		// We need to keep those for event handler to know it needs to send observed=false and exposed=false.

		let object_state = self.objects.get(descriptor).unwrap();
		//dbg!(descriptor,object_state.sent.observed,object_state.sent.exposed,object_state.received.observed,object_state.received.exposed,remote_is_upstream,self.remote_is_tag_observer_of_object(descriptor), self.local_is_exposer_of_object(object_state),self.remote_is_tag_exposer_of_object(descriptor),self.local_is_observer_of_object(object_state));

		if object_state.sent.observed
			|| object_state.sent.exposed
			|| object_state.received.observed
			|| object_state.received.exposed
			|| (remote_is_upstream
				&& (object_state.local.object_interface.is_observed_by_other_node(false)
					|| object_state.local.object_interface.is_exposed_by_other_node(false)))
			|| (!remote_is_upstream
				&& ((self.remote_is_tag_observer_of_object(descriptor) && self.local_is_exposer_of_object(object_state))
					|| (self.remote_is_tag_exposer_of_object(descriptor) && self.local_is_observer_of_object(object_state))))
		{
			//keeping the object
		} else {
			let mut object_state = self.objects.remove(descriptor).unwrap();
			drop(object_state.local.changes_token.take());
			drop(object_state);
		}
		if let Some(object_state) = self.objects.get_mut(descriptor) {
			if let Some(object_changes) = &mut self.object_changes {
				if object_state.local.changes_token.is_none() {
					object_state.local.changes_token = Some(object_changes.include(object_state.local.object_interface.changes()));
				}
			}
		}
	}

	pub(super) fn tag_maybe_borrow(&mut self, descriptor: &TagDescriptor) -> Option<&mut TagState> {
		if self.tags.contains_key(descriptor) {
			Some(self.tags.get_mut(descriptor).unwrap())
		} else {
			None
		}
	}

	pub(super) fn tag_borrow_mut(&mut self, descriptor: &TagDescriptor, local_node: &Arc<LocalNodeShared>, node: &Arc<NodeCore>) -> &mut TagState {
		if !self.tags.contains_key(descriptor) {
			let tag_state = TagState {
				received: RemoteTagState { observed: false, exposed: false, cost: None },
				sent: RemoteTagState { observed: false, exposed: false, cost: None },
				local: LocalTagState {
					tag_interface: RemoteTag::new(local_node.clone(), node.clone(), descriptor),
					changes_token: None,
					objects_changes_token: None,
				},
			};
			self.tags.insert(descriptor.clone(), tag_state);
		};
		self.tags.get_mut(descriptor).unwrap()
	}

	pub(super) fn tag_return(&mut self, descriptor: &TagDescriptor, remote_is_upstream: bool) {
		// On DOWNSTREAM side:
		// We would keep tags observed or exposed by remote side, to retain corresponding contracts, but upstream is not supposed to send us tag changes at all.
		// We also keep tags which have downstream-side observers or composers.
		//
		// On UPSTREAM side:
		// We keep tags observed or exposed by remote side, to retain corresponding local contracts.
		//
		// On BOTH sides:
		// We keep tags which were declared to be observed or exposed to the remote side.

		let tag_state = self.tags.get(descriptor).unwrap();

		if tag_state.sent.observed
			|| tag_state.sent.exposed
			|| tag_state.received.observed
			|| tag_state.received.exposed
			|| (remote_is_upstream && (self.local_is_observer_of_tag(tag_state) || self.local_is_exposer_of_tag(tag_state)))
		{
			// keeping the tag
		} else {
			let mut tag_state = self.tags.remove(descriptor).unwrap();
			drop(tag_state.local.changes_token.take());
			drop(tag_state.local.objects_changes_token.take());
		}

		if let Some(tag_state) = self.tags.get_mut(descriptor) {
			if let Some(object_changes) = &mut self.object_changes {
				if tag_state.local.changes_token.is_none() {
					tag_state.local.objects_changes_token = Some(object_changes.include(tag_state.local.tag_interface.objects_changes()));
				}
			}
			if let Some(tag_changes) = &mut self.tag_changes {
				tag_state.local.changes_token = Some(tag_changes.include(tag_state.local.tag_interface.changes()));
			}
		}
	}

	fn remote_is_tag_observer_of_object(&self, object_descriptor: &ObjectDescriptor) -> bool {
		object_descriptor
			.tags
			.iter()
			.any(|tag_descriptor| self.tags.get(tag_descriptor).map(|tag| tag.local.tag_interface.is_observed_by_this_node()).unwrap_or(false))
	}

	fn remote_is_tag_exposer_of_object(&self, object_descriptor: &ObjectDescriptor) -> bool {
		object_descriptor
			.tags
			.iter()
			.any(|tag_descriptor| self.tags.get(tag_descriptor).map(|tag| tag.local.tag_interface.is_exposed_by_this_node()).unwrap_or(false))
	}

	fn local_is_exposer_of_object(&self, object_state: &ObjectState) -> bool {
		object_state.local.object_interface.is_exposed_by_other_node(false)
	}

	fn local_is_observer_of_object(&self, object_state: &ObjectState) -> bool {
		object_state.local.object_interface.is_observed_by_other_node(false)
	}

	fn local_is_exposer_of_tag(&self, tag_state: &TagState) -> bool {
		tag_state.local.tag_interface.is_exposed_by_other_node()
	}

	fn local_is_observer_of_tag(&self, tag_state: &TagState) -> bool {
		tag_state.local.tag_interface.is_observed_by_other_node()
	}
}


impl SentRemoteObjectState {
	pub(super) fn apply_change(&mut self, change: &ObjectChange) {
		self.observed = change.observer.unwrap_or(self.observed);
		self.exposed = change.exposer.unwrap_or(self.exposed);
		self.cost = change.cost.or(self.cost);
	}
}

impl RemoteTagState {
	pub(super) fn apply_change(&mut self, change: &TagChange) {
		self.observed = change.observer.unwrap_or(self.observed);
		self.exposed = change.exposer.unwrap_or(self.exposed);
		//TODO: cost? self.cost = change.cost.or(self.cost);
	}
}


pub struct RemoteNodeMutable {
	state: State,
	// last sent message number
	last_sent_message_sequence_number: u64,
	// last received message number
	last_received_message_sequence_number: u64,
	// last sent acknowledgment
	last_sent_message_acknowledgement_number: u64,
	diff_produce: Option<bool>,
	diff_request: Option<bool>,
}

pub struct RemoteNode {
	local_node: Arc<LocalNodeShared>,
	//TODO: de-pub, and implement accessor
	pub local_node_events: EventMerge<Descriptor>,
	//TODO: remote should die when nodecore dies
	node_core: Arc<NodeCore>,
	mutable: Mutex<RemoteNodeMutable>,
}


impl Drop for RemoteNode {
	fn drop(&mut self) {
		trace!("Dropping RemoteNode {:?}", self.node_core);
		//FORMER-BUG, objects and tags clear was missing, leading to wrong ref counts
		//REGRESSION-TEST-NEEDED
		let mut mutable = self.mutable.lock().unwrap();
		mutable.state.objects.clear();
		mutable.state.tags.clear();
		assert_eq!(Arc::strong_count(&self.node_core), 1);
	}
}



impl RemoteNode {
	pub fn new(local_node: &LocalNode, upstream: bool, diff_produce: Option<bool>, diff_request: Option<bool>) -> RemoteNode {
		let local_node_events = EventMerge::new();
		let object_events_map = EventMap::new(EventMerge::new(), Descriptor::Object);
		let tag_events_map = EventMap::new(EventMerge::new(), Descriptor::Tag);
		let mut object_changes = None;
		let mut tag_changes = None;

		if upstream {
			object_events_map.input().include(local_node.shared.objects_changes()).forever();
			tag_events_map.input().include(local_node.shared.tags_changes()).forever();
		} else {
			object_changes = Some(object_events_map.input().clone());
			tag_changes = Some(tag_events_map.input().clone());
		};
		local_node_events.include(object_events_map).forever();
		local_node_events.include(tag_events_map).forever();

		let state = State { objects: HashMap::new(), tags: HashMap::new(), object_changes, tag_changes, options: OptionsState { diff_request: None } };
		RemoteNode {
			node_core: NodeCore::new(upstream, Some(0), Some(0), None),
			local_node: local_node.shared.clone(),
			local_node_events,
			mutable: Mutex::new(RemoteNodeMutable {
				state,
				last_sent_message_sequence_number: 0,
				last_received_message_sequence_number: 0,
				last_sent_message_acknowledgement_number: 0,
				diff_produce,
				diff_request,
			}),
		}
	}

	//TODO: some kind of id() maybe too
	pub fn name(&self) -> String {
		self.node_core.name.read().unwrap().clone().unwrap_or_else(|| "-".to_string())
	}

	pub fn connected(&self) -> Result<Vec<Message>, String> {
		let mut mutable = self.mutable.lock().unwrap();
		let mut changes_to_send = vec![Change::Options(OptionsChange {
			name: self.local_node.node_core.as_ref().unwrap().name.read().unwrap().clone(),
			custom: None,
			expose_capacity: Some(self.local_node.node_core.as_ref().unwrap().capacity.load(Ordering::Relaxed)),
			expose_load_limit: Some(self.local_node.node_core.as_ref().unwrap().limit.load(Ordering::Relaxed)),
			diff_request: Some(mutable.diff_request),
		})];
		for key in self.local_node_events.keys() {
			changes_to_send.extend(match key {
				Descriptor::Object(descriptor) => self.process_object_event(&mut mutable, descriptor, Action::Change)?,
				Descriptor::Tag(descriptor) => self.process_tag_event(&mut mutable, descriptor, Action::Change)?,
			});
		}
		Ok(self.build_messages(&mut mutable, changes_to_send))
	}

	pub fn disconnected(&self) {
		let mut mutable = self.mutable.lock().unwrap();
		mutable.state.objects.clear();
		mutable.state.tags.clear();
		mutable.state.options.diff_request = None;
		mutable.last_sent_message_sequence_number = 0;
		mutable.last_received_message_sequence_number = 0;
		mutable.last_sent_message_acknowledgement_number = 0;
	}

	pub fn ack(&self) -> Option<Message> {
		let mut mutable = self.mutable.lock().unwrap();
		if mutable.last_sent_message_acknowledgement_number == mutable.last_received_message_sequence_number {
			None
		} else {
			mutable.last_sent_message_acknowledgement_number = mutable.last_received_message_sequence_number;
			Some(Message {
				sequence_number: mutable.last_sent_message_sequence_number,
				acknowledgment_number: mutable.last_received_message_sequence_number,
				changes: vec![],
			})
		}
	}

	//TODO replace String error with proper error enum
	pub fn received_message(&self, message: Message) -> Result<Vec<Message>, String> {
		trace!("{:?} message received\n{}", self.node_core, message.inspect());

		let changes_to_send = vec![];
		let mut mutable = self.mutable.lock().unwrap();
		mutable.last_received_message_sequence_number = message.sequence_number;

		for change in message.changes {
			match change {
				Change::Options(options_change) => {
					if let Some(diff_request) = options_change.diff_request {
						mutable.state.options.diff_request = diff_request;
					};
					if let Some(name) = options_change.name {
						self.node_core.name_set(&name);
					};
					if let Some(capacity) = options_change.expose_capacity {
						self.node_core.capacity.store(capacity, Ordering::SeqCst);
						//TODO: rebalance
					}
					if let Some(limit) = options_change.expose_load_limit {
						self.node_core.limit.store(limit, Ordering::SeqCst);
						//TODO: rebalance
					}
					if let Some(diff_request) = options_change.diff_request {
						mutable.state.options.diff_request = diff_request;
					}
					//TODO: custom data
				}

				Change::Object(object_change) => {
					let object_state = mutable.state.object_borrow_mut(&object_change.descriptor, &self.local_node, &self.node_core);
					if let Some(observer) = object_change.observer {
						if observer == object_state.received.observed {
							return Err("Protocol error. Unneeded data received (object observer field).".to_string());
						};
						object_state.local.object_interface.observe(observer);
						object_state.received.observed = observer;
					}
					if let Some(exposer) = object_change.exposer {
						if exposer == object_state.received.exposed {
							return Err("Protocol error. Unneeded data received (object exposer field).".to_string());
						};
						let cost = if exposer { object_change.cost.flatten().or_else(|| object_state.received.cost.flatten()).expect("No cost?") } else { 0 };
						object_state.local.object_interface.expose(exposer, cost);
						object_state.received.exposed = exposer;
					}
					if let Some(data_version) = &object_change.data_version {
						//TODO: fail if same or lower data_version received?
						//TODO: fail if data_type received without data
						if let Some(data) = object_change.data {
							match data {
								DataChange::Data { data } => {
									let previous_archived_data = object_state.received.data.pop_back();
									object_state.received.data.push_front(ArchivedReceivedDataVersion {
										data_type: Some(object_change.data_type.or_else(|| previous_archived_data.unwrap().data_type).unwrap()),
										data_version: data_version.clone(),
										data: Some(data),
									});
								}
								DataChange::Diff { diff, old_data_version } => {
									let previous_archived_data = object_state.received.data.pop_back();
									let empty = Arc::new(vec![]);
									let (old_data, old_data_type) = if let Some(old_data_version) = old_data_version {
										if let Some(archived_data) =
											object_state.sent.data.iter().find(|archived_data| *archived_data.data_version == old_data_version)
										{
											(&archived_data.data, Some(&archived_data.data_type))
										} else if previous_archived_data.is_some()
											&& previous_archived_data.as_ref().unwrap().data.is_some()
											&& *previous_archived_data.as_ref().unwrap().data_version == old_data_version
										{
											(
												previous_archived_data.as_ref().unwrap().data.as_ref().unwrap(),
												Some(previous_archived_data.as_ref().unwrap().data_type.as_ref().unwrap()),
											)
										} else {
											//this can happen if remote side sent us data while we lost interest in the object
											//TEST-NEEDED
											//TODO: think if we should retain data for this case
											continue;
										}
									} else {
										(&empty, None)
									};
									object_state.received.data.push_front(ArchivedReceivedDataVersion {
										data_version: data_version.clone(),
										data: Some(Arc::new(diff::patch(old_data, &diff)?)),
										data_type: Some(object_change.data_type.or_else(|| old_data_type.cloned()).unwrap()),
									});
								}
							};
							if object_state.local.object_interface.is_exposed_by_this_node() {
								//TODO: or should we check if node is the **assigned** exposer?
								object_state.local.object_interface.state_set(
									data_version,
									object_state.received.data.front().as_ref().unwrap().data.as_ref().unwrap().clone(),
									object_state.received.data.front().as_ref().unwrap().data_type.as_ref().unwrap().clone(),
								);
							} else {
								return Err("Remote state error. Data submitted for object remote is not a exposer of.".to_string());
							}
						} else {
							//TODO: if there is no data sent, should that only be permitted for initial message? only for observers?
							object_state.local.object_interface.maybe_synchronize(data_version);
							let _previous_archived_data = object_state.received.data.pop_back();
							object_state.received.data.push_front(ArchivedReceivedDataVersion {
								data_version: data_version.clone(),
								data_type: None,
								data: None, //TODO: lookup locally
							});
						}
					} else if object_change.data.is_some() || object_change.data_type.is_some() {
						return Err("Remote state error. Data or data type submitted without data version.".to_string());
					}
					mutable.state.object_return(&object_change.descriptor, self.node_core.upstream);
				}

				Change::Tag(tag_change) => {
					let tag_state = mutable.state.tag_borrow_mut(&tag_change.descriptor, &self.local_node, &self.node_core);
					if let Some(observer) = tag_change.observer {
						if observer == tag_state.received.observed {
							return Err("Protocol error. Unneeded data received (tag observer field).".to_string());
						};
						tag_state.local.tag_interface.observe(observer);
						tag_state.received.observed = observer;
					};
					if let Some(exposer) = tag_change.exposer {
						if exposer == tag_state.received.exposed {
							return Err("Protocol error. Unneeded data received (tag exposer field).".to_string());
						};
						let cost = if exposer { tag_change.cost.flatten().or_else(|| tag_state.received.cost.flatten()).expect("No cost?") } else { 0 };
						tag_state.local.tag_interface.expose(exposer, cost);
						tag_state.received.exposed = exposer;
					};
					mutable.state.tag_return(&tag_change.descriptor, self.node_core.upstream);
				}
			}
		}

		for object_state in mutable.state.objects.values_mut() {
			while object_state.sent.data.len() > 1 && object_state.sent.data.front().unwrap().can_be_dropped_at_ack <= message.acknowledgment_number {
				object_state.sent.data.pop_back();
			}
		}

		Ok(self.build_messages(&mut mutable, changes_to_send))
	}

	pub fn received_local_node_event(&self, event: Event<Descriptor>) -> Result<Vec<Message>, String> {
		let mut mutable = self.mutable.lock().unwrap();
		let changes = match event.key {
			Descriptor::Object(descriptor) => self.process_object_event(&mut mutable, descriptor, event.action)?,
			Descriptor::Tag(descriptor) => self.process_tag_event(&mut mutable, descriptor, event.action)?,
		};
		Ok(self.build_messages(&mut mutable, changes))
	}

	fn build_messages(&self, mutable: &mut RemoteNodeMutable, changes: Vec<Change>) -> Vec<Message> {
		if changes.is_empty() {
			vec![]
		} else {
			mutable.last_sent_message_sequence_number += 1;
			mutable.last_sent_message_acknowledgement_number = mutable.last_received_message_sequence_number;
			let ret = vec![Message {
				sequence_number: mutable.last_sent_message_sequence_number,
				acknowledgment_number: mutable.last_received_message_sequence_number,
				changes,
			}];
			trace!("{:?}, message sending:\n{}", self.node_core, ret[0].inspect());
			ret
		}
	}

	fn process_object_event(&self, mutable: &mut RemoteNodeMutable, descriptor: ObjectDescriptor, action: Action) -> Result<Vec<Change>, String> {
		trace!("{:?}, object event received {:?} {:?}", self.node_core, &action, &descriptor);
		let mut changes = vec![];
		match action {
			Action::Insert | Action::Change => {
				{
					let options = mutable.state.options.clone();
					let object_state = mutable.state.object_borrow_mut(&descriptor, &self.local_node, &self.node_core);

					let last_received_version = object_state.received.data.front();
					let last_sent_version = object_state.sent.data.iter().find(|archived_version| archived_version.data_transmitted);
					let highest_version_known_by_remote = vec![
						last_received_version.as_ref().map(|archived| &archived.data_version),
						last_sent_version.as_ref().map(|archived| &archived.data_version),
					]
					.into_iter()
					.flatten()
					.max();

					let (_synchronized, data_version, data_type, data) = object_state.local.object_interface.state();
					let local_node_knows_newer_version_than_remote = data_version
						.as_ref()
						.and_then(|newest_version| highest_version_known_by_remote.as_ref().map(|known_version| newest_version > known_version).or(Some(true)))
						.unwrap_or(false);

					let (declare_local_as_observer, declare_local_as_exposer) = if self.node_core.upstream {
						(
							object_state.local.object_interface.is_observed_by_other_node(false),
							object_state.local.object_interface.is_exposed_by_other_node(false),
						)
					} else {
						(
							object_state.local.object_interface.is_assigned_to_this_node()
								&& object_state.local.object_interface.is_observed_by_other_node(true),
							object_state.local.object_interface.is_assigned_to_other_node() && object_state.local.object_interface.is_observed_by_this_node(),
						)
					};

					let declare_cost = if declare_local_as_exposer { object_state.local.object_interface.minimum_cost_in_other_nodes() } else { None };

					let (declare_data, data_type_of_declared_data) = if object_state.local.object_interface.is_assigned_to_other_node()
						&& object_state.local.object_interface.is_observed_by_this_node()
						&& local_node_knows_newer_version_than_remote
					{
						if mutable.diff_produce.unwrap_or_else(|| options.diff_request.unwrap_or(true)) {
							let (newest_shared_data, newest_shared_data_type, newest_shared_data_version) = match (last_received_version, last_sent_version) {
								(None, None) => (None, None, None),
								(Some(received_version), None) =>
									if received_version.data.is_some() {
										(
											Some(received_version.data.as_ref().unwrap()),
											Some(received_version.data_type.as_ref().unwrap()),
											Some(&received_version.data_version),
										)
									} else {
										(None, None, None)
									},
								(None, Some(sent_version)) => (Some(&sent_version.data), Some(&sent_version.data_type), Some(&sent_version.data_version)),
								(Some(received_version), Some(sent_version)) =>
									if received_version.data_version > sent_version.data_version && received_version.data.is_some() {
										(
											Some(received_version.data.as_ref().unwrap()),
											Some(received_version.data_type.as_ref().unwrap()),
											Some(&received_version.data_version),
										)
									} else {
										(Some(&sent_version.data), Some(&sent_version.data_type), Some(&sent_version.data_version))
									},
							};
							let diff = object_state.local.object_interface.get_diff(
								newest_shared_data_version,
								newest_shared_data,
								data_version.as_ref().unwrap(),
								data.as_ref().unwrap(),
							);
							if let Some(diff) = diff {
								(
									Some(DataChange::Diff { diff, old_data_version: newest_shared_data_version.cloned() }),
									newest_shared_data_type.or_else(|| Some(data_type.as_ref().unwrap())),
								)
							} else {
								(Some(DataChange::Data { data: data.clone().unwrap() }), Some(data_type.as_ref().unwrap()))
							}
						} else {
							(Some(DataChange::Data { data: data.clone().unwrap() }), Some(data_type.as_ref().unwrap()))
						}
					} else {
						(None, None)
					};

					let declare_data_type = data_type_of_declared_data
						.filter(|new_data_type| last_sent_version.map(|last_sent_version| last_sent_version.data_type != **new_data_type).unwrap_or(true));

					let change = ObjectChange {
						descriptor: descriptor.clone(),
						observer: if declare_local_as_observer != object_state.sent.observed { Some(declare_local_as_observer) } else { None },
						exposer: if declare_local_as_exposer != object_state.sent.exposed { Some(declare_local_as_exposer) } else { None },
						data_version: if declare_data.is_some()
							|| ((declare_local_as_observer && !object_state.sent.observed) || (declare_local_as_exposer && !object_state.sent.exposed))
						{
							data_version.clone()
						} else {
							None
						},
						data_type: declare_data_type.cloned(),
						data: declare_data,
						cost: if Some(declare_cost) != object_state.sent.cost { Some(declare_cost) } else { None },
					};
					object_state.sent.apply_change(&change);

					if !change.is_empty() {
						if change.data_version.is_some() {
							object_state.sent.data.push_front(ArchivedSentDataVersion {
								can_be_dropped_at_ack: mutable.last_sent_message_sequence_number,
								data_version: data_version.unwrap(),
								data_transmitted: change.data.is_some(),
								data_type: data_type.unwrap(),
								data: data.unwrap(),
							});
						}
						changes.push(Change::Object(change));
					};
				}
				mutable.state.object_return(&descriptor, self.node_core.upstream);
			}
			Action::Remove => {
				if let Some(object_state) = mutable.state.object_maybe_borrow_mut(&descriptor) {
					let change = ObjectChange {
						descriptor: descriptor.clone(),
						observer: if object_state.sent.observed { Some(false) } else { None },
						exposer: if object_state.sent.exposed { Some(false) } else { None },
						//TODO: should we clear all following fields?
						data: None,
						data_type: None,
						data_version: None,
						cost: None,
					};

					object_state.sent.apply_change(&change);
					mutable.state.object_return(&descriptor, self.node_core.upstream);

					if !change.is_empty() {
						changes.push(Change::Object(change));
					};
				}
			}
		}
		Ok(changes)
	}

	fn process_tag_event(&self, mutable: &mut RemoteNodeMutable, descriptor: TagDescriptor, action: Action) -> Result<Vec<Change>, String> {
		let mut changes = vec![];
		if self.node_core.upstream {
			match action {
				Action::Insert | Action::Change => {
					let tag_state = mutable.state.tag_borrow_mut(&descriptor, &self.local_node, &self.node_core);

					let declare_local_as_observer = tag_state.local.tag_interface.is_observed_by_other_node();
					let declare_local_as_exposer = tag_state.local.tag_interface.is_exposed_by_other_node();
					let declare_cost = if declare_local_as_exposer { tag_state.local.tag_interface.minimum_cost_in_other_nodes() } else { None };

					let change = TagChange {
						descriptor: descriptor.clone(),
						observer: if declare_local_as_observer != tag_state.sent.observed { Some(declare_local_as_observer) } else { None },
						exposer: if declare_local_as_exposer != tag_state.sent.exposed { Some(declare_local_as_exposer) } else { None },
						cost: if Some(declare_cost) != tag_state.sent.cost { Some(declare_cost) } else { None },
					};

					tag_state.sent.apply_change(&change);
					mutable.state.tag_return(&descriptor, self.node_core.upstream);

					if !change.is_empty() {
						changes.push(Change::Tag(change))
					};
				}
				Action::Remove =>
					if let Some(tag_state) = mutable.state.tag_maybe_borrow(&descriptor) {
						let change = TagChange {
							descriptor: descriptor.clone(),
							observer: if tag_state.sent.observed { Some(false) } else { None },
							exposer: if tag_state.sent.exposed { Some(false) } else { None },
							//TODO: should we clear this explicitly?
							cost: None,
						};

						tag_state.sent.apply_change(&change);
						mutable.state.tag_return(&descriptor, self.node_core.upstream);

						if !change.is_empty() {
							changes.push(Change::Tag(change));
						};
					},
			}
		}
		Ok(changes)
	}

	#[doc(hidden)]
	pub fn inspect(&self) -> String {
		let mutable = self.mutable.lock().unwrap();
		let mut ret = format!("│ RemoteNode: {:?}\n", self.node_core,);
		for (descriptor, state) in &mutable.state.objects {
			ret.push_str(
				format!(
					"│ {:?} Received({}) Sent({}) ReceivedVersionsStored:{}  SentVersionsStored:{}\n",
					&descriptor,
					vec![if state.received.observed { "observed" } else { "" }, if state.received.exposed { "exposed" } else { "" }].join(","),
					vec![if state.sent.observed { "observed" } else { "" }, if state.sent.exposed { "exposed" } else { "" }].join(","),
					state.received.data.len(),
					state.sent.data.len()
				)
				.as_str(),
			);
		}
		for (descriptor, state) in &mutable.state.tags {
			ret.push_str(
				format!(
					"│ {:?} Received({}) Sent({})\n",
					&descriptor,
					vec![if state.received.observed { "observed" } else { "" }, if state.received.exposed { "exposed" } else { "" }].join(","),
					vec![if state.sent.observed { "observed" } else { "" }, if state.sent.exposed { "exposed" } else { "" }].join(","),
				)
				.as_str(),
			);
		}
		ret
	}
}
