//use rand::Rng;
use std::{collections::{hash_map::Entry, HashMap}, fmt, sync::{Arc, RwLock, Weak}};

use super::builder::LocalNodeBuilder;
use crate::{descriptor::ObjectDescriptor, events::{EventMerge, EventMergeInput}, node::core::NodeCore, object::core::ObjectCore, tag::core::TagCore, TagDescriptor};

/// Main access point to the Hakuban network
#[derive(Clone)]
pub struct LocalNode {
	pub(crate) shared: Arc<LocalNodeShared>,
}

pub(crate) struct LocalNodeShared {
	pub(crate) node_core: Option<Arc<NodeCore>>,
	objects: RwLock<HashMap<ObjectDescriptor, ObjectAllocation>>,
	tags: RwLock<HashMap<TagDescriptor, TagAllocation>>,
	objects_changes: EventMerge<ObjectDescriptor>,
	tags_changes: EventMerge<TagDescriptor>,
}

struct ObjectAllocation {
	object_core: Weak<ObjectCore>,
	//we keep this only to be dropped when allocation is dropped
	_registered_merge_input: EventMergeInput<ObjectDescriptor>,
}

struct TagAllocation {
	tag_core: Weak<TagCore>,
	//we keep this only to be dropped when allocation is dropped
	_registered_merge_input: EventMergeInput<TagDescriptor>,
}


impl LocalNode {
	pub(super) fn new(name: Option<String>, capacity: Option<u32>, limit: Option<u32>) -> LocalNode {
		LocalNode {
			shared: Arc::new(LocalNodeShared {
				node_core: Some(NodeCore::new(false, capacity, limit, name)),
				objects: RwLock::new(HashMap::new()),
				tags: RwLock::new(HashMap::new()),
				objects_changes: EventMerge::new(),
				tags_changes: EventMerge::new(),
			}),
		}
	}

	pub fn builder() -> LocalNodeBuilder {
		LocalNodeBuilder::new()
	}

	pub fn object(&self, descriptor: impl Into<ObjectDescriptor>) -> crate::object::builder::ObjectContractBuilder {
		self.shared.object(descriptor.into())
	}

	pub fn tag(&self, descriptor: impl Into<TagDescriptor>) -> crate::tag::builder::TagContractBuilder {
		self.shared.tag(descriptor.into())
	}

	#[doc(hidden)]
	pub fn inspect(&self) -> String {
		self.shared.inspect()
	}
}


impl LocalNodeShared {
	pub fn local_node(self: &Arc<Self>) -> LocalNode {
		LocalNode { shared: self.clone() }
	}

	fn object(self: &Arc<Self>, descriptor: ObjectDescriptor) -> crate::object::builder::ObjectContractBuilder {
		crate::object::builder::ObjectContractBuilder::new(self.local_node(), descriptor)
	}

	fn tag(self: &Arc<Self>, descriptor: TagDescriptor) -> crate::tag::builder::TagContractBuilder {
		crate::tag::builder::TagContractBuilder::new(self.local_node(), descriptor)
	}

	fn object_allocation_create(self: &Arc<Self>, descriptor: ObjectDescriptor) -> (ObjectAllocation, Arc<ObjectCore>) {
		let tags: HashMap<TagDescriptor, Arc<TagCore>> =
			descriptor.tags.iter().map(|tag_descriptor| (tag_descriptor.clone(), self.tag_acquire(tag_descriptor))).collect();
		let object_core = Arc::new(ObjectCore::new(self.clone(), descriptor, tags.clone()));
		for tag in tags.values() {
			tag.object_insert(&object_core);
		}
		(
			ObjectAllocation { object_core: Arc::downgrade(&object_core), _registered_merge_input: self.objects_changes.include(object_core.changes()) },
			object_core,
		)
	}

	fn object_allocation_destroy(self: &Arc<Self>, descriptor: &ObjectDescriptor, object_allocation: ObjectAllocation) {
		for tag_descriptor in descriptor.tags.iter() {
			self.tag_acquire(tag_descriptor).object_remove(descriptor);
		}
		assert_eq!(Weak::strong_count(&object_allocation.object_core), 0);
		if Weak::weak_count(&object_allocation.object_core) > 1 {
			panic!();
		};
		drop(object_allocation);
	}

	pub(crate) fn object_acquire(self: &Arc<Self>, descriptor: &ObjectDescriptor) -> Arc<ObjectCore> {
		let mut objects = self.objects.write().unwrap();
		if let Entry::Occupied(entry) = objects.entry(descriptor.clone()) {
			if let Some(arc) = entry.get().object_core.upgrade() {
				return arc;
			} else {
				self.object_allocation_destroy(descriptor, entry.remove());
			}
		}
		let (object_allocation, object_core) = self.object_allocation_create(descriptor.clone());
		objects.insert(descriptor.clone(), object_allocation);
		object_core
	}

	pub(crate) fn object_release(self: &Arc<Self>, descriptor: &ObjectDescriptor) {
		if let Entry::Occupied(entry) = self.objects.write().unwrap().entry(descriptor.clone()) {
			if Weak::strong_count(&entry.get().object_core) == 0 {
				self.object_allocation_destroy(descriptor, entry.remove());
			}
		}
	}

	fn tag_allocation_create(self: &Arc<Self>, descriptor: TagDescriptor) -> (TagAllocation, Arc<TagCore>) {
		let tag_core = Arc::new(TagCore::new(self.clone(), descriptor));
		(TagAllocation { tag_core: Arc::downgrade(&tag_core), _registered_merge_input: self.tags_changes.include(tag_core.changes()) }, tag_core)
	}

	fn tag_core_destroy(self: &Arc<Self>, tag_allocation: TagAllocation) {
		assert_eq!(Weak::strong_count(&tag_allocation.tag_core), 0);
		if Weak::weak_count(&tag_allocation.tag_core) > 1 {
			panic!();
		};
		drop(tag_allocation);
	}

	pub(crate) fn tag_acquire(self: &Arc<Self>, descriptor: &TagDescriptor) -> Arc<TagCore> {
		let mut tags = self.tags.write().unwrap();
		if let Entry::Occupied(entry) = tags.entry(descriptor.clone()) {
			if let Some(arc) = entry.get().tag_core.upgrade() {
				return arc;
			} else {
				self.tag_core_destroy(entry.remove());
			}
		}
		let (tag_allocation, tag_core) = self.tag_allocation_create(descriptor.clone());
		tags.insert(descriptor.clone(), tag_allocation);
		tag_core
	}

	pub(crate) fn tag_release(self: &Arc<Self>, descriptor: &TagDescriptor) {
		if let Entry::Occupied(entry) = self.tags.write().unwrap().entry(descriptor.clone()) {
			if Weak::strong_count(&entry.get().tag_core) == 0 {
				self.tag_core_destroy(entry.remove());
			}
		}
	}

	pub(crate) fn objects_changes(&self) -> EventMerge<ObjectDescriptor> {
		self.objects_changes.clone()
	}

	pub(crate) fn tags_changes(&self) -> EventMerge<TagDescriptor> {
		self.tags_changes.clone()
	}

	#[doc(hidden)]
	pub fn inspect(self: &Arc<Self>) -> String {
		let mut ret = format!(
			"┌───────────────────────────────────────────────────────────────────── LocalNode: {}\n",
			*self.node_core.as_ref().unwrap().name.read().unwrap().as_ref().unwrap()
		);
		for (_descriptor, object) in self.objects.read().unwrap().iter() {
			if let Some(object) = object.object_core.upgrade() {
				ret.push_str(&object.inspect(None));
			}
		}
		for (_descriptor, tag) in self.tags.read().unwrap().iter() {
			if let Some(tag) = tag.tag_core.upgrade() {
				ret.push_str(&tag.inspect());
			}
		}
		ret.push_str("└╼");
		ret
	}
}


impl fmt::Debug for LocalNode {
	fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
		f.write_fmt(format_args!("LocalNode({})", self.shared.node_core.as_ref().unwrap().name.read().unwrap().as_ref().unwrap()))
	}
}
