#![allow(clippy::not_unsafe_ptr_arg_deref)]
use std::{convert::TryFrom, ffi::c_void, mem::forget};

use super::{raw_serialize, FFIObjectDescriptor, FFIObjectDescriptors, FFIObjectExposeState, FFIObjectExposeStateResult, FFITagDescriptor};
use crate::{events::{Callback, CallbackRegistry, Event}, ffi::{make_object_descriptor_callback, FFIError, ObjectEventCallbackFn}, object::core::ObjectRawData, LocalNode, ObjectDescriptor, TagDescriptor, TagExpose};

#[repr(C)]
pub struct FFITagExposeResult {
	pub error: FFIError,
	pub tag_expose_pointer: *mut TagExpose<(Vec<String>, Vec<u8>), ()>,
}


#[no_mangle]
pub extern "C" fn hakuban_tag_expose_new(local_node: *mut LocalNode, descriptor: FFITagDescriptor) -> FFITagExposeResult {
	let descriptor: TagDescriptor = match TagDescriptor::try_from(&descriptor) {
		Ok(descriptor) => descriptor,
		Err(error) => return FFITagExposeResult { error, tag_expose_pointer: std::ptr::null_mut() },
	};
	let local_node = unsafe { Box::from_raw(local_node) };
	let tag_expose_pointer = Box::into_raw(Box::new(local_node.tag(descriptor).with_serializer(raw_serialize).expose()));
	forget(local_node);
	FFITagExposeResult { error: FFIError::None, tag_expose_pointer }
}

#[no_mangle]
pub extern "C" fn hakuban_tag_expose_drop(tag_expose_ptr: *mut TagExpose<ObjectRawData, ()>) {
	drop(unsafe { Box::from_raw(tag_expose_ptr) });
}



#[no_mangle]
pub extern "C" fn hakuban_tag_expose_object_state(
	tag_expose_ptr: *mut TagExpose<(Vec<String>, Vec<u8>), ()>, descriptor: FFIObjectDescriptor, state: FFIObjectExposeState,
) -> FFIObjectExposeStateResult {
	let descriptor: ObjectDescriptor = match ObjectDescriptor::try_from(&descriptor) {
		Ok(descriptor) => descriptor,
		Err(error) => return FFIObjectExposeStateResult { error, changed: 0 },
	};
	let (data, data_type, data_version) = match state.to_rust() {
		Ok(state) => state,
		Err(error) => return FFIObjectExposeStateResult { error, changed: 0 },
	};
	let cooked = (data_type, data);
	let tag_expose = unsafe { Box::from_raw(tag_expose_ptr) };
	let changed = tag_expose.set_object_state(&descriptor, &data_version, &cooked).unwrap();
	forget(tag_expose);
	forget(cooked.1);
	forget(data_version);
	FFIObjectExposeStateResult { error: FFIError::None, changed: if changed { 1 } else { 0 } }
}

#[no_mangle]
pub extern "C" fn hakuban_tag_expose_object_descriptors_borrow(tag_expose_ptr: *mut TagExpose<(Vec<String>, Vec<u8>), ()>) -> FFIObjectDescriptors {
	let tag_expose = unsafe { Box::from_raw(tag_expose_ptr) };
	let descriptors: Vec<FFIObjectDescriptor> = tag_expose.object_descriptors().iter().map(|descriptor| descriptor.into()).collect();
	forget(tag_expose);
	FFIObjectDescriptors::new(descriptors)
}

#[no_mangle]
pub extern "C" fn hakuban_tag_expose_objects_changes_callback_register(
	tag_expose_ptr: *mut TagExpose<ObjectRawData, ()>, callback: ObjectEventCallbackFn, userdata: *const c_void,
) -> *mut Box<dyn Callback<Event<ObjectDescriptor>>> {
	let tag_expose = unsafe { Box::from_raw(tag_expose_ptr) };
	let ret = Box::new(tag_expose.changes().register(make_object_descriptor_callback(callback, userdata)));
	forget(tag_expose);
	Box::into_raw(ret)
}
