#![allow(clippy::not_unsafe_ptr_arg_deref)]

use std::{convert::TryFrom, ffi::{c_void, CString}, intrinsics::transmute, mem::forget, sync::Arc};

use super::FFIObjectDescriptor;
use crate::{events::{Callback, CallbackRegistry, Event}, ffi::FFIError, object::{core::{ObjectRawData, ObjectType}, observe::ObjectObserveState}, LocalNode, ObjectDescriptor, ObjectObserve};

#[allow(clippy::ptr_arg)] //it's less powerful but more readable
pub(super) fn raw_deserialize(_data_type: &ObjectType, data: &ObjectRawData) -> Result<ObjectRawData, ()> {
	Ok(data.clone())
}


//TODO: implement Try for this and other results
//TODO: better error api
#[repr(C)]
pub struct FFIObjectObserveResult {
	pub error: FFIError,
	pub object_observe_pointer: *mut ObjectObserve<ObjectRawData, ()>,
}

#[no_mangle]
pub extern "C" fn hakuban_object_observe_new(local_node: *mut LocalNode, descriptor: FFIObjectDescriptor) -> FFIObjectObserveResult {
	let descriptor: ObjectDescriptor = match ObjectDescriptor::try_from(&descriptor) {
		Ok(descriptor) => descriptor,
		Err(error) => return FFIObjectObserveResult { error, object_observe_pointer: std::ptr::null_mut() },
	};
	let local_node = unsafe { Box::from_raw(local_node) };
	let object_observe_pointer = Box::into_raw(Box::new(local_node.object(descriptor).with_deserializer(raw_deserialize).observe()));
	forget(local_node);
	FFIObjectObserveResult { error: FFIError::None, object_observe_pointer }
}


#[no_mangle]
pub extern "C" fn hakuban_object_observe_drop(object_ptr: *mut ObjectObserve<ObjectRawData, ()>) {
	drop(unsafe { Box::from_raw(object_ptr) });
}


//TODO: think about keeping this on rust side, returning pointer over ffi and expose accessors over ffi
#[repr(C)]
pub struct FFIObjectObserveState {
	pub synchronized: u8,
	pub version_length: usize,
	pub version: *mut i64,
	pub data_type_length: usize,
	pub data_type: *mut *mut i8,
	pub raw_length: usize,
	pub raw: *const u8,
	raw_arc: *const Vec<u8>,
}

impl FFIObjectObserveState {
	pub(super) fn empty() -> FFIObjectObserveState {
		FFIObjectObserveState {
			synchronized: 0,
			version_length: 0,
			version: std::ptr::null_mut(),
			data_type_length: 0,
			data_type: std::ptr::null_mut(),
			raw_length: 0,
			raw: std::ptr::null_mut(),
			raw_arc: std::ptr::null_mut(),
		}
	}

	pub(super) fn from_state(mut state: ObjectObserveState<ObjectRawData>) -> FFIObjectObserveState {
		FFIObjectObserveState {
			synchronized: state.synchronized as u8,
			version_length: state.data_version.as_ref().map(|version| version.len()).unwrap_or(0),
			version: state
				.data_version
				.take()
				.and_then(|mut version| {
					version.shrink_to_fit();
					let ret = Some(version.as_mut_ptr()); //TODO: into_raw_parts ?
					forget(version);
					ret
				})
				.unwrap_or(std::ptr::null_mut()),
			data_type_length: state.data_type.as_ref().map(|data_type| data_type.len()).unwrap_or(0),
			data_type: state
				.data_type
				.take()
				.and_then(|data_type| {
					let mut data_type_array: Vec<*mut i8> = Vec::with_capacity(data_type.len());
					for element in data_type {
						let cstring = CString::new(element).unwrap();
						data_type_array.push(cstring.into_raw());
					}
					//TODO: into_raw_parts
					let ret = Some(data_type_array.as_mut_ptr());
					forget(data_type_array);
					ret
				})
				.unwrap_or(std::ptr::null_mut()),
			raw_length: state.data.as_ref().map(|data| data.len()).unwrap_or(0),
			raw: state.data.as_ref().map(|data| data.as_ptr()).unwrap_or(std::ptr::null_mut()),
			raw_arc: state.data.take().map(Arc::into_raw).unwrap_or(std::ptr::null_mut()),
		}
	}

	pub(super) fn free(self) {
		if !self.version.is_null() {
			drop(unsafe { Vec::from_raw_parts(self.version, self.version_length, self.version_length) });
		}
		if !self.data_type.is_null() {
			let vec = unsafe { Vec::from_raw_parts(self.data_type, self.data_type_length, self.data_type_length) };
			for element in vec {
				drop(unsafe { CString::from_raw(element) });
			}
		}
		if !self.raw_arc.is_null() {
			drop(unsafe { Arc::from_raw(self.raw_arc) });
		}
	}
}

#[no_mangle]
pub extern "C" fn hakuban_object_observe_state_borrow(object_observe_ptr: *mut ObjectObserve<ObjectRawData, ()>) -> FFIObjectObserveState {
	let object_observe = unsafe { Box::from_raw(object_observe_ptr) };
	let state = object_observe.object_state().unwrap();
	forget(object_observe);
	FFIObjectObserveState::from_state(state)
}

#[no_mangle]
pub extern "C" fn hakuban_object_observe_state_return(state: FFIObjectObserveState) {
	state.free();
}


pub type ObjectEventCallbackFn = extern "C" fn(*const c_void, object_descriptor: FFIObjectDescriptor, action: u8);

pub(super) fn make_object_descriptor_callback(callback: ObjectEventCallbackFn, userdata: *const c_void) -> Box<impl Fn(Event<ObjectDescriptor>)> {
	let pointers_are_not_send: usize = userdata as usize;
	Box::new(move |event: Event<ObjectDescriptor>| {
		let ffi_descriptor: FFIObjectDescriptor = (&event.key).into();
		(callback)(unsafe { transmute(pointers_are_not_send) }, ffi_descriptor.clone(), event.action as u8);
		ffi_descriptor.free();
	})
}

#[no_mangle]
pub extern "C" fn hakuban_object_observe_changes_callback_register(
	object_observe_ptr: *mut ObjectObserve<ObjectRawData, ()>, callback: ObjectEventCallbackFn, userdata: *const c_void,
) -> *mut Box<dyn Callback<Event<ObjectDescriptor>>> {
	let object_observe = unsafe { Box::from_raw(object_observe_ptr) };
	let ret = Box::new(object_observe.changes().register(make_object_descriptor_callback(callback, userdata)));
	forget(object_observe);
	Box::into_raw(ret)
}

#[no_mangle]
pub extern "C" fn hakuban_object_callback_unregister(callback: *mut Box<dyn Callback<Event<ObjectDescriptor>>>) {
	drop(unsafe { Box::from_raw(callback) });
}
