#![allow(clippy::not_unsafe_ptr_arg_deref)]
use std::{convert::TryFrom, ffi::c_void, mem::forget, sync::Arc};

use log::error;

use super::FFIObjectDescriptor;
use crate::{events::{Callback, CallbackRegistry, Event}, ffi::{make_object_descriptor_callback, FFIError, ObjectEventCallbackFn}, object::core::{ObjectRawData, ObjectType, ObjectVersion}, LocalNode, ObjectDescriptor, ObjectExpose};


pub(super) fn raw_serialize(cooked: &(ObjectType, Vec<u8>)) -> Result<(ObjectType, ObjectRawData), ()> {
	Ok((cooked.0.clone(), Arc::new(cooked.1.clone())))
}


#[repr(C)]
pub struct FFIObjectExposeResult {
	pub error: FFIError,
	pub object_expose_pointer: *mut ObjectExpose<(Vec<String>, Vec<u8>), ()>,
}

#[no_mangle]
pub extern "C" fn hakuban_object_expose_new(local_node: *mut LocalNode, descriptor: FFIObjectDescriptor) -> FFIObjectExposeResult {
	let descriptor: ObjectDescriptor = match ObjectDescriptor::try_from(&descriptor) {
		Ok(descriptor) => descriptor,
		Err(error) => return FFIObjectExposeResult { error, object_expose_pointer: std::ptr::null_mut() },
	};
	let local_node = unsafe { Box::from_raw(local_node) };
	let object_expose_pointer = Box::into_raw(Box::new(local_node.object(descriptor).with_serializer(raw_serialize).expose()));
	forget(local_node);
	FFIObjectExposeResult { error: FFIError::None, object_expose_pointer }
}


#[no_mangle]
pub extern "C" fn hakuban_object_expose_drop(object_expose_ptr: *mut ObjectExpose<ObjectRawData, ()>) {
	drop(unsafe { Box::from_raw(object_expose_ptr) });
}

#[repr(C)]
pub struct FFIObjectExposeState {
	pub version_length: usize,
	pub version: *mut i64,
	pub data_type_length: usize,
	pub data_type: *const *const i8,
	pub raw_length: usize,
	pub raw: *mut u8,
}

impl FFIObjectExposeState {
	pub(super) fn to_rust(self: &FFIObjectExposeState) -> Result<(Vec<u8>, ObjectType, ObjectVersion), FFIError> {
		//TODO: Arcify that vec?
		let version = unsafe { Vec::from_raw_parts(self.version, self.version_length, self.version_length) };
		let data = unsafe { Vec::from_raw_parts(self.raw, self.raw_length, self.raw_length) };
		let data_type_slice: &[*const i8] = unsafe { std::slice::from_raw_parts(self.data_type, self.data_type_length) };
		//TODO: don't clone these strings so many times
		let data_type: Result<Vec<String>, std::str::Utf8Error> =
			data_type_slice.iter().map(|c_str| unsafe { Ok(std::ffi::CStr::from_ptr(*c_str).to_str()?.to_owned()) }).collect();
		let data_type = match data_type {
			Ok(strings) => strings,
			Err(error) => {
				forget(version);
				forget(data);
				error!("Couldn't process data_type field: {:?}", error);
				return Err(FFIError::InvalidString);
			}
		};
		Ok((data, data_type, version))
	}
}

#[repr(C)]
pub struct FFIObjectExposeStateResult {
	pub error: FFIError,
	pub changed: u8,
}

#[no_mangle]
pub extern "C" fn hakuban_object_expose_state(
	object_expose_ptr: *mut ObjectExpose<(Vec<String>, Vec<u8>), ()>, state: FFIObjectExposeState,
) -> FFIObjectExposeStateResult {
	let (data, data_type, data_version) = match state.to_rust() {
		Ok(state) => state,
		Err(error) => return FFIObjectExposeStateResult { error, changed: 0 },
	};
	let cooked = (data_type, data);
	let object_expose = unsafe { Box::from_raw(object_expose_ptr) };
	let changed = object_expose.set_object_state(&data_version, &cooked).unwrap();
	forget(object_expose);
	forget(cooked.1);
	forget(data_version);
	FFIObjectExposeStateResult { error: FFIError::None, changed: if changed { 1 } else { 0 } }
}


#[no_mangle]
pub extern "C" fn hakuban_object_expose_assigned(object_expose_ptr: *mut ObjectExpose<(Vec<String>, Vec<u8>), ()>) -> u8 {
	let object_expose = unsafe { Box::from_raw(object_expose_ptr) };
	let ret = if object_expose.assigned() { 1 } else { 0 };
	forget(object_expose);
	ret
}

#[no_mangle]
pub extern "C" fn hakuban_object_expose_changes_callback_register(
	object_expose_ptr: *mut ObjectExpose<ObjectRawData, ()>, callback: ObjectEventCallbackFn, userdata: *const c_void,
) -> *mut Box<dyn Callback<Event<ObjectDescriptor>>> {
	let object_expose = unsafe { Box::from_raw(object_expose_ptr) };
	let ret = Box::new(object_expose.changes().register(make_object_descriptor_callback(callback, userdata)));
	forget(object_expose);
	Box::into_raw(ret)
}
