//if we don't mark pub function as unsafe when it has unsafe sections, clippy will complain
//if we put an unsafe block inside an unsafe function, clippy will complain too
//we'd rather have every unsafe block explicit, so disabling warning that function is not unsafe
#![allow(clippy::not_unsafe_ptr_arg_deref)]

use log::{error, trace};

use crate::LocalNode;


#[derive(Copy, Clone, Debug)]
#[repr(u8)]
pub enum FFIError {
	None = 0,
	InvalidString = 1,
	InvalidJSON = 2,
	InvalidURL = 3,
	ObjectNotFound = 4,
}


#[repr(C)]
pub struct FFILocalNodeNewResult {
	pub error: FFIError,
	pub local_node_pointer: *mut LocalNode,
}

//TODO: capacity, limit

#[no_mangle]
pub extern "C" fn hakuban_local_node_new(name_c: *const i8) -> FFILocalNodeNewResult {
	let name_str: &str = match unsafe { std::ffi::CStr::from_ptr(name_c).to_str() } {
		Ok(string) => string,
		Err(error) => {
			error!("Couldn't convert node name parameter to string: {:?}", error);
			return FFILocalNodeNewResult { error: FFIError::InvalidString, local_node_pointer: std::ptr::null_mut() };
		}
	};
	trace!("Constructing local node {:?}", name_str);
	FFILocalNodeNewResult { error: FFIError::None, local_node_pointer: Box::into_raw(Box::new(LocalNode::builder().with_name(name_str).build())) }
}

#[no_mangle]
pub extern "C" fn hakuban_local_node_drop(local_node: *mut LocalNode) {
	let local_node = unsafe { Box::from_raw(local_node) };
	trace!("Dropping local node {:?}", local_node);
	drop(local_node);
}
