use std::{error::Error, time::Duration};

use hakuban::{tokio::WebsocketConnector, LocalNode};
use serde::{Deserialize, Serialize};
use serde_json::json;
use sysinfo::{ProcessorExt, SystemExt};
use tokio::time::sleep;

#[derive(Serialize, Deserialize)]
struct CpuUtilization {
	percentage: Vec<u8>,
}


#[tokio::main]
async fn main() -> Result<(), Box<dyn Error>> {
	let unique_id = format!("{}@{}", std::process::id(), hostname::get().unwrap().into_string().unwrap());

	let hakuban = LocalNode::builder().with_name(&unique_id).build();
	let _upstream = WebsocketConnector::new(hakuban.clone(), std::env::args().nth(1).unwrap_or_else(|| "ws://127.0.0.1:3001".to_string()).as_str())?;
	let all_utilizations = hakuban.tag("utilizations").observe::<CpuUtilization>();
	let my_utilization = hakuban.object((["utilizations"], json!({ "id": unique_id }))).expose();

	let mut system = sysinfo::System::default();

	loop {
		system.refresh_cpu();
		let my_current_utilization = CpuUtilization { percentage: system.get_processors().iter().map(|processor| processor.get_cpu_usage() as u8).collect() };
		my_utilization.set_object_data(&my_current_utilization)?;

		println!();
		for other_utilization_descriptor in all_utilizations.object_descriptors() {
			if let Some(state) = all_utilizations.object_state(&other_utilization_descriptor)? {
				if let Some(utilization) = state.data {
					println!("{} -> {:?}", other_utilization_descriptor.json.get("id").unwrap().as_str().unwrap(), utilization.percentage);
				}
			}
		}

		sleep(Duration::from_millis(1000)).await;
	}
}
