import hakuban_wasm from "../pkg/hakuban_bg.wasm"
import * as hakuban_js from "../pkg/hakuban.js"


logger_initialized = false

export initialize = ()->
		await hakuban_js.default(await hakuban_wasm())

export logger_initialize = (log_level, skip_if_already_initialized=false) ->
		if logger_initialized and skip_if_already_initialized
			return true

		if 0 == (error = await hakuban_js.hakuban_logger_initialize(log_level))
			logger_initialized = true
		else
			console.error "Failed to initialize logger, error " + error  
			false


raise_if_error = (error_code)->
	if error_code != 0
		throw "Hakuban error: "+error_code


export JSON_serialize = (data_type, cooked)->
	{
		data_type: ["JSON"].concat(data_type)
		data: new TextEncoder().encode(JSON.stringify(cooked))
	}

export JSON_deserialize = (data_type, raw)-> 
	throw "Expected JSON data_type, got: "+data_type  if data_type.length == 0 or data_type[0] != "JSON"
	{
		data_type: data_type.slice(1)
		data: JSON.parse(new TextDecoder().decode(raw))
	}


export class ObjectDescriptor
	constructor: (@tags, @json) -> null
	as_json: ()->JSON.stringify(tags: @tags, json: @json)

export class TagDescriptor
	constructor: (@json) -> null
	as_json: ()->JSON.stringify(@json)


LocalNode_finalization_registry = new FinalizationRegistry (pointer)=>
	hakuban_js.hakuban_local_node_drop(pointer)

export class LocalNode
	constructor: (@name="wasm") ->
		logger_initialize("warn", true)
		result = hakuban_js.hakuban_local_node_new(@name)
		raise_if_error(result.error)
		@pointer = result.local_node_pointer
		LocalNode_finalization_registry.register(@, @pointer)
		@with_default_serializer(JSON_serialize)
		@with_default_deserializer(JSON_deserialize)

	destroy: () ->
		hakuban_js.hakuban_local_node_drop(@pointer)
		LocalNode_finalization_registry.unregister(@)

	object: (tags, json) ->
		new ObjectBuilder(@, @default_serializer, @default_deserializer, tags, json)

	tag: (json) ->
		new TagBuilder(@, @default_serializer, @default_deserializer, json)

	with_default_serializer: (@default_serializer) -> @
	with_default_deserializer: (@default_deserializer) -> @


class ObjectBuilder
	constructor: (@local_node, @serializer, @deserializer, @tags, @json) -> null
	observe: () -> new ObjectObserve(@local_node, new ObjectDescriptor(@tags, @json), @deserializer)
	expose: () -> new ObjectExpose(@local_node, new ObjectDescriptor(@tags, @json), @serializer)
	with_serializer: (@serializer) -> @
	with_deserializer: (@deserializer) -> @

class TagBuilder
	constructor: (@local_node, @serializer, @deserializer, @json) -> null
	observe: () -> new TagObserve(@local_node, new TagDescriptor(@json), @deserializer)
	expose: () -> new TagExpose(@local_node, new TagDescriptor(@json), @serializer)
	with_serializer: (@serializer) -> @
	with_deserializer: (@deserializer) -> @


RustPromise_finalization_registry = new FinalizationRegistry (cancel_waiting_resolve)=>
	cancel_waiting_resolve.resolve(true)


ObjectObserve_finalization_registry = new FinalizationRegistry (pointer)=>
	hakuban_js.hakuban_object_observe_drop(pointer)

class ObjectObserve
	constructor: (@local_node, @descriptor, @deserializer) ->
		result = hakuban_js.hakuban_object_observe_new(@local_node.pointer, @descriptor.as_json())
		raise_if_error(result.error)
		@pointer = result.object_observe_pointer
		ObjectObserve_finalization_registry.register(@, @pointer)
	
	destroy: () ->
		hakuban_js.hakuban_object_observe_drop(@pointer)
		ObjectObserve_finalization_registry.unregister(@)

	object_state: () ->
		state_pointer = hakuban_js.hakuban_object_observe_state_borrow(@pointer)
		return { initialized: false }  if not hakuban_js.hakuban_object_observe_state_initialized(state_pointer)
		version = JSON.parse(hakuban_js.hakuban_object_observe_state_data_version(state_pointer))
		data_type = JSON.parse(hakuban_js.hakuban_object_observe_state_data_type(state_pointer))
		raw = hakuban_js.hakuban_object_observe_state_data(state_pointer)
		hakuban_js.hakuban_object_observe_state_return(state_pointer)
		deserialized = @deserializer(data_type, raw)
		{
			initialized: true
			version: version
			data: deserialized.data
			data_type: deserialized.data_type
		}

	next_event: ()-> 
		cancel_waiting_resolve = null
		cancel_waiting_promise = new Promise (resolve) -> cancel_waiting_resolve = resolve
		next_event_promise = hakuban_js.hakuban_object_observe_next_event(@pointer, cancel_waiting_promise)
		next_event_promise.cancel = (value)->
			cancel_waiting_resolve.resolve(value)
			await next_event_promise
		RustPromise_finalization_registry.register(@, cancel_waiting_resolve)
		next_event_promise


ObjectExpose_finalization_registry = new FinalizationRegistry (pointer)=>
	hakuban_js.hakuban_object_expose_drop(pointer)

class ObjectExpose
	constructor: (@local_node, @descriptor, @serializer)->
		result = hakuban_js.hakuban_object_expose_new(@local_node.pointer, @descriptor.as_json())
		raise_if_error(result.error)
		@pointer = result.object_expose_pointer
		ObjectExpose_finalization_registry.register(@, @pointer)

	destroy: ()->
		hakuban_js.hakuban_object_expose_drop(@pointer)
		ObjectExpose_finalization_registry.unregister(@)

	set_object_state: (version, cooked_data, data_type=[])->
		serialized = @serializer(data_type, cooked_data)
		result = hakuban_js.hakuban_object_expose_state(@pointer, JSON.stringify(version), JSON.stringify(serialized.data_type), serialized.data)
		raise_if_error(result.error)
		result.changed

	assigned: ()->
		hakuban_js.hakuban_object_expose_assigned(@pointer)

	next_event: ()-> 
		cancel_waiting_resolve = null
		cancel_waiting_promise = new Promise (resolve) -> cancel_waiting_resolve = resolve
		next_event_promise = hakuban_js.hakuban_object_expose_next_event(@pointer, cancel_waiting_promise)
		next_event_promise.cancel = (value)->
			cancel_waiting_resolve.resolve(value)
			await next_event_promise
		RustPromise_finalization_registry.register(@, cancel_waiting_resolve)
		next_event_promise


TagObserve_finalization_registry = new FinalizationRegistry (pointer)=>
	hakuban_js.hakuban_tag_observe_drop(pointer)

class TagObserve
	constructor: (@local_node, @descriptor, @deserializer) ->
		result = hakuban_js.hakuban_tag_observe_new(@local_node.pointer, @descriptor.as_json())
		raise_if_error(result.error)
		@pointer = result.tag_observe_pointer
		TagObserve_finalization_registry.register(@, @pointer)
	
	destroy: () ->
		hakuban_js.hakuban_tag_observe_drop(@pointer)
		TagObserve_finalization_registry.unregister(@)

	object_descriptors: ()->
		result = hakuban_js.hakuban_tag_observe_object_descriptors(@pointer)
		for descriptor_as_json in JSON.parse(result)
			new ObjectDescriptor(descriptor_as_json.tags, descriptor_as_json.json)

	object_state: (object_descriptor) ->
		result = hakuban_js.hakuban_tag_observe_object_state_borrow(@pointer, object_descriptor.as_json())
		return null  if result.error == 4
		if hakuban_js.hakuban_object_observe_state_initialized(result.state_pointer)
			version = JSON.parse(hakuban_js.hakuban_object_observe_state_data_version(result.state_pointer))
			data_type = JSON.parse(hakuban_js.hakuban_object_observe_state_data_type(result.state_pointer))
			raw = hakuban_js.hakuban_object_observe_state_data(result.state_pointer)
			hakuban_js.hakuban_object_observe_state_return(result.state_pointer)
			deserialized = @deserializer(data_type, raw)
			{
				initialized: true
				version: version
				data: deserialized.data
				data_type: deserialized.data_type
			}
		else
			hakuban_js.hakuban_object_observe_state_return(result.state_pointer)
			{ initialized: false }

	next_event: ()-> 
		cancel_waiting_resolve = null
		cancel_waiting_promise = new Promise (resolve) -> cancel_waiting_resolve = resolve
		next_event_promise = hakuban_js.hakuban_tag_observe_next_object_event(@pointer, cancel_waiting_promise)
		next_event_promise.cancel = (value)->
			cancel_waiting_resolve.resolve(value)
			await next_event_promise
		RustPromise_finalization_registry.register(@, cancel_waiting_resolve)
		next_event_promise



TagExpose_finalization_registry = new FinalizationRegistry (pointer)=>
	hakuban_js.hakuban_tag_expose_drop(pointer)

class TagExpose
	constructor: (@local_node, @descriptor, @serializer) ->
		result = hakuban_js.hakuban_tag_expose_new(@local_node.pointer, @descriptor.as_json())
		raise_if_error(result.error)
		@pointer = result.tag_expose_pointer
		TagExpose_finalization_registry.register(@, @pointer)
	
	destroy: () ->
		hakuban_js.hakuban_tag_expose_drop(@pointer)
		TagExpose_finalization_registry.unregister(@)

	object_descriptors: ()->
		result = hakuban_js.hakuban_tag_expose_object_descriptors(@pointer)
		for descriptor_as_json in JSON.parse(result)
			new ObjectDescriptor(descriptor_as_json.tags, descriptor_as_json.json)

	set_object_state: (object_descriptor, version, cooked_data, data_type=[]) ->
		serialized = @serializer(data_type, cooked_data)
		result = hakuban_js.hakuban_tag_expose_object_state(@pointer, object_descriptor.as_json(), JSON.stringify(version), JSON.stringify(serialized.data_type), serialized.data)
		raise_if_error(result.error)
		result.changed

	next_event: ()-> 
		cancel_waiting_resolve = null
		cancel_waiting_promise = new Promise (resolve) -> cancel_waiting_resolve = resolve
		next_event_promise = hakuban_js.hakuban_tag_expose_next_object_event(@pointer, cancel_waiting_promise)
		next_event_promise.cancel = (value)->
			cancel_waiting_resolve.resolve(value)
			await next_event_promise
		RustPromise_finalization_registry.register(@, cancel_waiting_resolve)
		next_event_promise


WebsocketConnector_finalization_registry = new FinalizationRegistry (pointer)=>
	hakuban_js.hakuban_remote_node_drop(pointer)

#TODO: hide private methods
#TODO: handle errors somehow
export class WebsocketConnector

	constructor: (@local_node, @uri, @options={})->
		@connected = false
		@socket = undefined
		@error = undefined
		@reconnection_timer = undefined
		@reconnect_on_close = true
		@ack_timer = undefined
		result = hakuban_js.hakuban_remote_node_new(@local_node.pointer, true, true, true)
		raise_if_error(result.error)		
		@remote_node_pointer = result.remote_node_pointer
		@event_loop_stop = new Promise (resolve) => @event_loop_stop_resolve = resolve
		@event_loop_promise = @event_loop()
		WebsocketConnector_finalization_registry.register(@, @remote_node_pointer)
		@connect()

	destroy: () ->
		@event_loop_stop_resolve(["stop"])
		await @event_loop_promise
		@reconnect_on_close = false
		if @keep_alive_interval
			clearInterval(@keep_alive_interval)  
		if @reconnection_timer?
			clearTimeout(@reconnection_timer)
			@reconnection_timer = null
		if @socket?
			@socket.onclose = null
			@socket.onerror = null
			@socket.onmessage = null
			@socket.close()
			@socket_onclose()
		if @ack_timer?
			clearTimeout(@ack_timer)
			@ack_timer = null
		hakuban_js.hakuban_remote_node_drop(@remote_node_pointer)
		WebsocketConnector_finalization_registry.unregister(@)

		
	connect: ()=>
		@reconnection_timer = undefined

		try
			@socket = new WebSocket(@uri)
		catch error
			@error =  { while: "connecting", error: error }
			@onstatuschange_proc(@)  if @onstatuschange_proc?
			return false

		@socket.binaryType = "arraybuffer"

		@socket.onopen = ()=>
			console.debug('Connected to hakuban remote node')  if @options.debug
			@connected = true
			@error = undefined
			@onstatuschange_proc(@)  if @onstatuschange_proc?
			bytes_to_send = hakuban_js.hakuban_remote_node_connected(@remote_node_pointer)
			@send(bytes_to_send)

		@socket.onclose = @socket_onclose = ()=>
			console.debug('Disconnected from hakuban remote node')  if @options.debug
			clearInterval(@keep_alive_interval)  if @keep_alive_interval
			@connected = false
			@socket = undefined
			hakuban_js.hakuban_remote_node_disconnected(@remote_node_pointer)
			@onstatuschange_proc(@)  if @onstatuschange_proc?
			clearTimeout(@reconnection_timer)  if @reconnection_timer?
			@reconnection_timer = null
			@reconnection_timer = setTimeout(@connect, 1000)  if @reconnect_on_close

		@socket.onerror = (error)=>
			console.log('Hakuban socket error', error)  if @options.debug
			@error = { while: "connected", error: error }

		@socket.onmessage = (event) =>
			bytes_to_send = hakuban_js.hakuban_remote_node_received_message(@remote_node_pointer, new Uint8Array(event.data))
			@send(bytes_to_send)
			clearTimeout(@ack_timer)  if @ack_timer?
			@ack_timer = setTimeout(@ack, 1000)
			null

	ping: ()=>
		@socket.send(new Uint8Array())  if @socket

	send: (bytes_to_send)=>
		if bytes_to_send.length > 0
			@socket.send(bytes_to_send)
			clearInterval(@keep_alive_interval)  if @keep_alive_interval
			@keep_alive_interval = setInterval(@ping, 10000)

	ack: ()=>
		if @socket?
			bytes_to_send = hakuban_js.hakuban_remote_node_ack(@remote_node_pointer)
			@send(bytes_to_send)  if bytes_to_send.length > 0

	event_loop: ()->
		next_local_node_event = false
		cancel_waiting_resolve = null
		cancel_waiting_promise = new Promise (resolve) -> cancel_waiting_resolve = resolve
		while true
			next_local_node_event = hakuban_js.hakuban_remote_node_next_local_node_event(@remote_node_pointer, cancel_waiting_promise).then((event)->["local_node_event", event])  if not next_local_node_event
			[promise, event] = await Promise.any([@event_loop_stop, next_local_node_event])
			if promise == "local_node_event"
				next_local_node_event = false
				setTimeout((()=>
					if @connected  #there's a race here, but lets just ignore that!
						bytes_to_send = hakuban_js.hakuban_remote_node_received_local_node_event(@remote_node_pointer, event)
						@send(bytes_to_send)
				), 0)
			else
				break
		if next_local_node_event
			cancel_waiting_resolve(true)
			await next_local_node_event

	onstatuschange: (proc)->
		@onstatuschange_proc = proc
		@onstatuschange_proc()  if @onstatuschange_proc?
		@
