use std::{sync::Arc, time::Duration};

extern crate hakuban;

use async_tungstenite::tungstenite;
use futures::{SinkExt, StreamExt};
use hakuban::message::Message;
use tokio::{sync::mpsc, task::JoinHandle};

use crate::scene::*;

pub struct MockClient;

impl MockClient {
	pub fn spawn(scene: Arc<Scene>, name: String) -> (mpsc::Sender<Order>, JoinHandle<()>) {
		let (tx, mut rx) = mpsc::channel::<Order>(1000);
		let join_handle = tokio::spawn(async move {
			let mut ws_stream = None;
			let mut target_address = None;
			while let Some(order) = rx.recv().await {
				match order {
					Order::Event(_frame, event, done) => {
						match event {
							TestAction::Connect { connect_to } => {
								target_address = Some(format!("ws://127.0.0.1:{}", scene.ports.read().unwrap().get(&connect_to.clone()).unwrap()));
								match async_tungstenite::tokio::connect_async(target_address.as_ref().unwrap().clone()).await {
									Ok((stream, _response)) => {
										ws_stream = Some(stream);
									}
									Err(error) => {
										panic!("Failed to connect: {:?}", error);
									}
								};
							}
							TestAction::Disconnect => {
								ws_stream.take().unwrap().close(None).await.unwrap();
							}
							TestAction::NetPartition => {
								ws_stream.take().unwrap().close(None).await.unwrap();
							}
							TestAction::NetJoin => {
								ws_stream = Some(async_tungstenite::tokio::connect_async(target_address.as_ref().unwrap().clone()).await.unwrap().0);
							}
							TestAction::Send(message) => {
								//println!("sending message {:?}", message);
								match ws_stream {
									Some(ref mut stream) => stream.send(tungstenite::Message::Binary(rmp_serde::to_vec(&message).unwrap())).await.unwrap(),
									None => panic!(),
								};
							}
							TestAction::Receive(mut expected_messages) => {
								//println!("waiting for message {:?}", expected_message);
								match ws_stream {
									Some(ref mut stream) => {
										while !expected_messages.is_empty() {
											tokio::select! {
												next_message = stream.next() => match next_message {
													Some(Ok(tungstenite::Message::Binary(message_msgpack))) => {
														let target_changes = rmp_serde::from_read_ref::<Vec<u8>, Message>(&message_msgpack).unwrap().changes;
														let found_at = expected_messages
															.iter()
															.enumerate()
															.find(|(_, expected_message)| {
																for (expected_change, target_change) in expected_message.changes.iter().zip(target_changes.iter()) {
																	if expected_change != target_change {
																		return false;
																	};
																}
																//to_send.push();
																true
															})
															.unwrap_or_else(|| panic!("❌ERROR: {} received message different than expected.\nExpected:\n{:?}\nReceived:\n{:?}\n", name, expected_messages, target_changes))
															.0;
														expected_messages.remove(found_at);
													},
													Some(Err(error)) => panic!("Connection error: {:?}", error),
													other => panic!("Unexpected happened: {:?}", other)
												},
												_ = tokio::time::sleep(Duration::from_secs_f32(1.0)) => {
													panic!("\n❌{} - TIMEOUT waiting for messages {:?}", name, expected_messages);
												}
											};
										}
									}
									None => panic!(),
								};
							}
							TestAction::GetDisconnected => loop {
								match ws_stream {
									Some(ref mut stream) => tokio::select! {
										next_message = stream.next() => match next_message {
											Some(Ok(tungstenite::Message::Ping(_))) => { },
											Some(Ok(tungstenite::Message::Close(_))) => { break; },
											Some(Ok(tungstenite::Message::Binary(message_msgpack))) => {
												let received_message = rmp_serde::from_read_ref::<Vec<u8>,Message>(&message_msgpack).unwrap();
												panic!("\n❌ERROR: {} received message while expecting disconnect. message: {:?}", name, received_message);
											},
											other => panic!("\n❌ERROR: {} received unexpected event while expecting disconnect. event: {:?}", name, other),
										},
										_ = tokio::time::sleep(Duration::from_secs_f32(1.0)) => {
											panic!("\n❌){} - TIMEOUT waiting for disconnection", name);
										}
									},
									None => panic!(),
								};
							},
							TestAction::ReceiveNothing(duration) => {
								match ws_stream {
									Some(ref mut stream) => tokio::select! {
										_ = tokio::time::sleep(Duration::from_secs_f32(duration)) => {},
										message = stream.next() => match message {
											Some(Ok(tungstenite::Message::Binary(message_msgpack))) => {
												let received_message = rmp_serde::from_read_ref::<Vec<u8>,Message>(&message_msgpack).unwrap();
												panic!("\n❌ERROR: {} received message, while it should not. Message: {:?}",&name, &received_message);
											},
											_ => panic!()
										}
									},
									None => panic!(),
								};
							}
							TestAction::Sleep(duration) => {
								tokio::time::sleep(Duration::from_secs_f32(duration)).await;
							}
							// commands for testing client implementation, ignoring
							TestAction::ObjectObserveNew { .. } => {}
							TestAction::ObjectExposeNew { .. } => {}
							TestAction::TagObserveNew { .. } => {}
							TestAction::TagExposeNew { .. } => {}
							TestAction::ObjectObserveEvent { .. } => {}
							TestAction::ObjectVersionExposed { .. } => {}
							TestAction::ObjectExposeNewVersion { .. } => {}
							TestAction::ObjectExposeEvent { .. } => {}
							TestAction::TagObjectObserveEvent { .. } => {}
							TestAction::SendNothing { .. } => {}
							TestAction::TagObjectExposeNewVersion { .. } => {}
							TestAction::ObjectObserveDrop { .. } => {}
							TestAction::ObjectExposeDrop { .. } => {}
							TestAction::TagObserveDrop { .. } => {}
							TestAction::TagExposeDrop { .. } => {}
							TestAction::SendSomeMessages { .. } => {}
							TestAction::ReceiveSomeMessages { .. } => {}
							_ => panic!("MockClient can't do: {:?}", event),
						};
						done.send(Ok(())).unwrap();
					}
					Order::Exit => {
						return;
					}
				}
			}
		});
		(tx, join_handle)
	}
}
