#![allow(clippy::mutex_atomic)] //because I want the value AND the locking. actually, rethink this (TODO)

use std::{collections::HashMap, hash::{Hash, Hasher}, sync::{atomic::{AtomicU32, Ordering}, Arc, Mutex, RwLock}};

use super::core::TagCore;
use crate::{contract::Contract, events::{EventMerge, EventMergeInput, EventSource}, expose_contract::ExposeContract, node::{core::NodeCore, local::LocalNodeShared}, object::core::ObjectCore, observe_contract::ObserveContract, ObjectDescriptor, TagDescriptor};

pub(crate) struct RemoteTag {
	tag_interface: Arc<RemoteTagInterface>,
}


impl RemoteTag {
	pub(crate) fn new(local_node: Arc<LocalNodeShared>, node_core: Arc<NodeCore>, descriptor: &TagDescriptor) -> RemoteTag {
		let tag_core = local_node.tag_acquire(descriptor);
		RemoteTag { tag_interface: RemoteTagInterface::new(tag_core, node_core, 0) }
	}

	pub(crate) fn expose(&self, value: bool, cost: u32) {
		self.tag_interface.cost.store(cost, Ordering::SeqCst); //TODO weaken
		let mut expose = self.tag_interface.expose.lock().unwrap();
		if *expose != value {
			let dyn_self: Arc<dyn ExposeContract> = self.tag_interface.clone();
			if value {
				self.tag_interface.tag_core.link_tag_expose_contract(dyn_self);
			} else {
				self.tag_interface.tag_core.unlink_tag_expose_contract(&dyn_self);
			}
			*expose = value;
		}
	}

	pub(crate) fn observe(&self, value: bool) {
		let mut observe = self.tag_interface.observe.lock().unwrap();
		if *observe != value {
			let dyn_self: Arc<dyn ObserveContract> = self.tag_interface.clone();
			if value {
				self.tag_interface.tag_core.link_tag_observe_contract(dyn_self);
			} else {
				self.tag_interface.tag_core.unlink_tag_observe_contract(&dyn_self);
			}
			*observe = value;
		}
	}

	pub(crate) fn changes(&self) -> EventSource<TagDescriptor> {
		self.tag_interface.tag_core.changes()
	}

	pub(crate) fn objects_changes(&self) -> EventMerge<ObjectDescriptor> {
		self.tag_interface.tag_core.objects_changes()
	}

	pub(crate) fn minimum_cost_in_other_nodes(&self) -> Option<u32> {
		self.tag_interface.tag_core.minimum_cost_excluding_node(&self.tag_interface.node_core)
	}

	pub(crate) fn is_observed_by_this_node(&self) -> bool {
		*self.tag_interface.observe.lock().unwrap()
	}

	pub(crate) fn is_exposed_by_this_node(&self) -> bool {
		*self.tag_interface.expose.lock().unwrap()
	}

	pub(crate) fn is_observed_by_other_node(&self) -> bool {
		self.tag_interface.tag_core.has_observer_other_than(&self.tag_interface.node_core)
	}

	pub(crate) fn is_exposed_by_other_node(&self) -> bool {
		self.tag_interface.tag_core.has_exposer_other_than(&self.tag_interface.node_core)
	}
}

impl Drop for RemoteTag {
	fn drop(&mut self) {
		self.tag_interface.destroy();
	}
}


struct RemoteTagInterface {
	node_core: Arc<NodeCore>,
	tag_core: Arc<TagCore>,
	expose: Mutex<bool>,
	observe: Mutex<bool>,
	cost: AtomicU32,
	assigned_objects: RwLock<HashMap<ObjectDescriptor, Arc<ObjectCore>>>,
	assigned_objects_changes: EventMerge<ObjectDescriptor>,
	assigned_objects_changes_tokens: Mutex<HashMap<ObjectDescriptor, EventMergeInput<ObjectDescriptor>>>,
}

impl RemoteTagInterface {
	fn id(&self) -> usize {
		(self as *const Self) as usize
	}

	pub(super) fn new(tag_core: Arc<TagCore>, node_core: Arc<NodeCore>, cost: u32) -> Arc<RemoteTagInterface> {
		Arc::new(RemoteTagInterface {
			node_core,
			tag_core,
			expose: Mutex::new(false),
			observe: Mutex::new(false),
			cost: AtomicU32::new(cost),
			assigned_objects: RwLock::new(HashMap::new()),
			assigned_objects_changes: EventMerge::new(),
			assigned_objects_changes_tokens: Mutex::new(HashMap::new()),
		})
	}

	pub(super) fn destroy(self: &Arc<Self>) {
		if *self.observe.lock().unwrap() {
			let dyn_self: Arc<dyn ObserveContract> = self.clone();
			self.tag_core.unlink_tag_observe_contract(&dyn_self);
		};
		if *self.expose.lock().unwrap() {
			let dyn_self: Arc<dyn ExposeContract> = self.clone();
			self.tag_core.unlink_tag_expose_contract(&dyn_self);
		};
		assert_eq!(Arc::strong_count(self), 1);
	}
}


impl Hash for RemoteTagInterface {
	fn hash<H: Hasher>(&self, state: &mut H) {
		self.id().hash(state);
	}
}

impl PartialEq for RemoteTagInterface {
	fn eq(&self, other: &Self) -> bool {
		self.id() == other.id()
	}
}

impl Eq for RemoteTagInterface {}


impl std::fmt::Debug for RemoteTagInterface {
	fn fmt(&self, formatter: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
		formatter.write_str(&format!("RemoteTagInterface:{}", self.tag_core.descriptor))
	}
}


impl Contract for RemoteTagInterface {
	fn id(&self) -> usize {
		(self as *const Self) as usize
	}

	fn node_core(&self) -> &Arc<NodeCore> {
		&self.node_core
	}
}

impl ObserveContract for RemoteTagInterface {}

impl ExposeContract for RemoteTagInterface {
	fn assign(&self, object_core: Arc<ObjectCore>) -> bool {
		self.assigned_objects_changes_tokens
			.lock()
			.unwrap()
			.insert(object_core.descriptor.clone(), self.assigned_objects_changes.include(object_core.changes())); //TODO: check if we still exist
		self.assigned_objects.write().unwrap().insert(object_core.descriptor.clone(), object_core);
		true
	}

	fn unassign(&self, object_descriptor: &ObjectDescriptor) -> bool {
		self.assigned_objects_changes_tokens.lock().unwrap().remove(object_descriptor).unwrap().remove();
		self.assigned_objects.write().unwrap().remove(object_descriptor).unwrap(); //TODO: check if we still exist
		true
	}

	fn assigned(&self, object_descriptor: &ObjectDescriptor) -> bool {
		self.assigned_objects.read().unwrap().contains_key(object_descriptor)
	}

	fn composition_cost(&self) -> u32 {
		self.cost.load(Ordering::SeqCst)
	}
}
