use std::{collections::{HashMap, HashSet}, sync::{Arc, Mutex, RwLock}};

use super::core::TagCore;
use crate::{contract::Contract, events::{EventMerge, EventMergeInput}, expose_contract::ExposeContract, node::core::NodeCore, object::{core::{ObjectCore, ObjectVersion}, expose::ObjectSerializer}, DefaultSerializerError, LocalNode, ObjectDescriptor, TagDescriptor};


/// Represents a wish, a __contract__ to expose any object with specific tag
pub struct TagExpose<T: Send + Sync + 'static, E = DefaultSerializerError> {
	contract: Arc<TagExposeContract>,
	serializer: ObjectSerializer<T, E>,
}

impl<T: Send + Sync + 'static, E> TagExpose<T, E> {
	pub(super) fn new(contract: Arc<TagExposeContract>, serializer: ObjectSerializer<T, E>) -> TagExpose<T, E> {
		TagExpose { contract, serializer }
	}

	pub fn descriptor(&self) -> &TagDescriptor {
		&self.contract.tag_core.descriptor
	}

	pub fn object_descriptors(&self) -> HashSet<ObjectDescriptor> {
		//MISSING-REGRESSION-TEST: this reported active objects instead of assigned objects and no test caught it
		self.contract.assigned_objects.read().unwrap().keys().cloned().collect()
	}

	pub fn changes(&self) -> EventMerge<ObjectDescriptor> {
		self.contract.assigned_objects_changes()
	}

	pub fn set_object_state<'a>(
		&self, descriptor: impl Into<&'a ObjectDescriptor>, version: impl Into<&'a ObjectVersion>, cooked: impl Into<&'a T>,
	) -> Result<bool, E> {
		//TEST-MISSING: following 2 lines were one, keeping the lock while calling state_set, leading to deadlocks with unlink_object_observe_contract->recalculate_active->expose_contract_unassign->unassign
		let object_core = self.contract.assigned_objects.read().unwrap().get(descriptor.into()).cloned();
		if let Some(object_core) = object_core {
			let (data_type, raw) = (self.serializer)(cooked.into())?;
			let ret = object_core.state_set(self.contract.node_core(), version.into(), data_type, raw);
			Ok(ret)
		} else {
			Ok(false)
		}
	}

	pub fn set_object_data<'a>(&self, descriptor: impl Into<&'a ObjectDescriptor>, cooked: impl Into<&'a T>) -> Result<(), E> {
		let object_core = self.contract.assigned_objects.read().unwrap().get(descriptor.into()).cloned();
		if let Some(object_core) = object_core {
			//this is racy by nature
			let (data_type, raw) = (self.serializer)(cooked.into())?;
			object_core.data_set(self.contract.node_core(), data_type, raw);
		};
		Ok(())
	}

	pub fn set_object_version<'a>(&self, descriptor: impl Into<&'a ObjectDescriptor>, new_version: Option<impl Into<ObjectVersion>>) -> bool {
		let object_core = self.contract.assigned_objects.read().unwrap().get(descriptor.into()).cloned();
		if let Some(object_core) = object_core {
			object_core.version_set(self.contract.node_core(), new_version.map(|value| value.into()))
		} else {
			false
		}
	}
}


impl<T: Sync + Send, E> Drop for TagExpose<T, E> {
	fn drop(&mut self) {
		self.contract.destroy();
	}
}


pub(super) struct TagExposeContract {
	local_node: Option<LocalNode>,
	tag_core: Arc<TagCore>,
	composition_cost: u32,
	assigned_objects: RwLock<HashMap<ObjectDescriptor, Arc<ObjectCore>>>,
	assigned_objects_changes: EventMerge<ObjectDescriptor>, //TODO: only for tag dir
	assigned_objects_changes_tokens: Mutex<HashMap<ObjectDescriptor, EventMergeInput<ObjectDescriptor>>>, //TODO only for this dir
}

impl TagExposeContract {
	pub(super) fn new(local_node: LocalNode, tag_core: Arc<TagCore>, composition_cost: u32) -> Arc<Self> {
		let contract = Arc::new(TagExposeContract {
			local_node: Some(local_node),
			tag_core,
			composition_cost,
			assigned_objects: RwLock::new(HashMap::new()),
			assigned_objects_changes: EventMerge::new(),
			assigned_objects_changes_tokens: Mutex::new(HashMap::new()),
		});
		contract.tag_core.link_tag_expose_contract(contract.clone());
		contract
	}

	pub(super) fn destroy(self: &Arc<Self>) {
		let dyn_self: Arc<dyn ExposeContract> = self.clone();
		self.tag_core.unlink_tag_expose_contract(&dyn_self);
		drop(dyn_self);
		assert_eq!(Arc::strong_count(self), 1);
	}

	pub(super) fn assigned_objects_changes(&self) -> EventMerge<ObjectDescriptor> {
		self.assigned_objects_changes.clone()
	}
}

impl Contract for TagExposeContract {
	fn id(&self) -> usize {
		(self as *const Self) as usize
	}

	fn node_core(&self) -> &Arc<NodeCore> {
		self.local_node.as_ref().unwrap().shared.node_core.as_ref().unwrap()
	}
}

impl ExposeContract for TagExposeContract {
	fn assign(&self, object_core: Arc<ObjectCore>) -> bool {
		self.assigned_objects_changes_tokens
			.lock()
			.unwrap()
			.insert(object_core.descriptor.clone(), self.assigned_objects_changes.include(object_core.changes())); //TODO: check if we still exist
		self.assigned_objects.write().unwrap().insert(object_core.descriptor.clone(), object_core);
		true
	}

	fn unassign(&self, object_descriptor: &ObjectDescriptor) -> bool {
		self.assigned_objects_changes_tokens.lock().unwrap().remove(object_descriptor).unwrap().remove();
		self.assigned_objects.write().unwrap().remove(object_descriptor).unwrap(); //TODO: check if we still exist
		true
	}

	fn assigned(&self, object_descriptor: &ObjectDescriptor) -> bool {
		self.assigned_objects.read().unwrap().contains_key(object_descriptor)
	}

	fn composition_cost(&self) -> u32 {
		self.composition_cost
	}
}

impl std::fmt::Debug for TagExposeContract {
	fn fmt(&self, formatter: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
		formatter.write_str(&format!("TagExposeContract:{}", self.tag_core.descriptor))
	}
}
