use std::{error::Error, marker::PhantomData, sync::Arc};

use super::{core::ObjectRawData, expose::{ObjectExposeContract, ObjectSerializer}, observe::{ObjectDeserializer, ObjectObserveContract}};
use crate::{descriptor::ObjectDescriptor, object::{expose::ObjectExpose, observe::ObjectObserve}, LocalNode, ObjectType};

/// Error returned by object accessors if (de)serialization fails and default (de)serializer is used
#[derive(Debug)]
pub enum DefaultSerializerError {
	WrongDataType { data_type: ObjectType },
	MessagePackSerializerError(rmp_serde::encode::Error),
	MessagePackDeserializerError(rmp_serde::decode::Error),
}

impl std::fmt::Display for DefaultSerializerError {
	fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
		std::fmt::Debug::fmt(self, f)
	}
}

impl Error for DefaultSerializerError {}

#[must_use = "Call .expose() or .observe() at the end of the build chain."]
pub struct ObjectContractBuilder {
	local_node: LocalNode,
	descriptor: ObjectDescriptor,
}

//TODO: composition cost
//TODO: do we really need this structure parametrized?
#[must_use = "Call .expose() or .observe() at the end of the build chain."]
pub struct ObjectContractBuilderTyped<T: Sync + Send + 'static> {
	object_contract_builder: ObjectContractBuilder,
	_data_type: std::marker::PhantomData<T>,
}

#[must_use = "Call .expose() or .observe() at the end of the build chain."]
pub struct ObjectContractBuilderWithDeserializer<T: Sync + Send + 'static, E> {
	typed_object_contract_builder: ObjectContractBuilderTyped<T>,
	deserializer: ObjectDeserializer<T, E>,
}

#[must_use = "Call .expose() or .observe() at the end of the build chain."]
pub struct ObjectContractBuilderWithSerializer<T: Sync + Send + 'static, E> {
	typed_object_contract_builder: ObjectContractBuilderTyped<T>,
	serializer: ObjectSerializer<T, E>,
}



impl ObjectContractBuilder {
	pub(crate) fn new(local_node: LocalNode, descriptor: ObjectDescriptor) -> ObjectContractBuilder {
		ObjectContractBuilder { local_node, descriptor }
	}

	pub fn of_type<T: Sync + Send + 'static>(self) -> ObjectContractBuilderTyped<T> {
		ObjectContractBuilderTyped { object_contract_builder: self, _data_type: PhantomData }
	}

	pub fn with_deserializer<T: Sync + Send + 'static, E, F: 'static + Send + Sync + Fn(&ObjectType, &ObjectRawData) -> Result<T, E>>(
		self, deserializer: F,
	) -> ObjectContractBuilderWithDeserializer<T, E> {
		ObjectContractBuilderWithDeserializer { typed_object_contract_builder: self.of_type(), deserializer: Box::new(deserializer) }
	}

	pub fn with_serializer<T: Sync + Send + 'static, E, F: 'static + Send + Sync + Fn(&T) -> Result<(ObjectType, ObjectRawData), E>>(
		self, serializer: F,
	) -> ObjectContractBuilderWithSerializer<T, E> {
		ObjectContractBuilderWithSerializer { typed_object_contract_builder: self.of_type(), serializer: Box::new(serializer) }
	}

	pub fn observe<T: Sync + Send + 'static + serde::de::DeserializeOwned>(self) -> ObjectObserve<T, DefaultSerializerError> {
		self.of_type().observe()
	}

	pub fn expose<T: Sync + Send + serde::Serialize>(self) -> ObjectExpose<T, DefaultSerializerError> {
		self.of_type().expose()
	}
}


impl<T: Sync + Send + 'static> ObjectContractBuilderTyped<T> {
	pub fn with_deserializer<E, F: 'static + Send + Sync + Fn(&ObjectType, &ObjectRawData) -> Result<T, E>>(
		self, deserializer: F,
	) -> ObjectContractBuilderWithDeserializer<T, E> {
		ObjectContractBuilderWithDeserializer { typed_object_contract_builder: self, deserializer: Box::new(deserializer) }
	}

	pub fn with_serializer<E, F: 'static + Send + Sync + Fn(&T) -> Result<(ObjectType, ObjectRawData), E>>(
		self, serializer: F,
	) -> ObjectContractBuilderWithSerializer<T, E> {
		ObjectContractBuilderWithSerializer { typed_object_contract_builder: self, serializer: Box::new(serializer) }
	}
}

//TODO: think about usage of data_type
impl<T: Sync + Send + 'static + serde::de::DeserializeOwned> ObjectContractBuilderTyped<T> {
	pub fn observe(self) -> ObjectObserve<T, DefaultSerializerError> {
		self.with_deserializer(|data_type: &ObjectType, raw_data: &ObjectRawData| -> Result<T, DefaultSerializerError> {
			if data_type.len() != 1 || data_type[0] != "MessagePack" {
				return Err(DefaultSerializerError::WrongDataType { data_type: data_type.clone() });
			}
			rmp_serde::from_read_ref::<Vec<u8>, T>(raw_data).map_err(DefaultSerializerError::MessagePackDeserializerError)
		})
		.observe()
	}
}

//TODO: think about usage of data_type
impl<T: Sync + Send + serde::Serialize> ObjectContractBuilderTyped<T> {
	pub fn expose(self) -> ObjectExpose<T, DefaultSerializerError> {
		self.with_serializer(|data: &T| -> Result<(ObjectType, ObjectRawData), DefaultSerializerError> {
			let serialized = rmp_serde::to_vec(data).map_err(DefaultSerializerError::MessagePackSerializerError)?;
			Ok((vec!["MessagePack".to_string()], Arc::new(serialized)))
		})
		.expose()
	}
}



impl<T: Sync + Send + 'static, E> ObjectContractBuilderWithDeserializer<T, E> {
	pub fn observe(self) -> ObjectObserve<T, E> {
		let object = self
			.typed_object_contract_builder
			.object_contract_builder
			.local_node
			.shared
			.object_acquire(&self.typed_object_contract_builder.object_contract_builder.descriptor);
		let contract = ObjectObserveContract::new(self.typed_object_contract_builder.object_contract_builder.local_node.shared.local_node(), object);
		ObjectObserve::new(contract, self.deserializer)
	}
}


impl<T: Sync + Send + 'static, E> ObjectContractBuilderWithSerializer<T, E> {
	pub fn expose(self) -> ObjectExpose<T, E> {
		let object = self
			.typed_object_contract_builder
			.object_contract_builder
			.local_node
			.shared
			.object_acquire(&self.typed_object_contract_builder.object_contract_builder.descriptor);
		let contract = ObjectExposeContract::new(self.typed_object_contract_builder.object_contract_builder.local_node.shared.local_node(), object, 1000); //TODO: function
		ObjectExpose::new(contract, self.serializer)
	}
}
