#![allow(clippy::not_unsafe_ptr_arg_deref)]

use std::{convert::TryFrom, ffi::c_void, mem::forget};

use super::{FFIObjectDescriptor, FFITagDescriptor};
use crate::{events::{Callback, CallbackRegistry, Event}, ffi::{make_object_descriptor_callback, raw_deserialize, FFIError, FFIObjectObserveState, ObjectEventCallbackFn}, object::core::ObjectRawData, LocalNode, ObjectDescriptor, TagDescriptor, TagObserve};

#[repr(C)]
pub struct FFITagObserveResult {
	pub error: FFIError,
	pub tag_observe_pointer: *mut TagObserve<ObjectRawData, ()>,
}


#[no_mangle]
pub extern "C" fn hakuban_tag_observe_new(local_node: *mut LocalNode, descriptor: FFITagDescriptor) -> FFITagObserveResult {
	let descriptor: TagDescriptor = match TagDescriptor::try_from(&descriptor) {
		Ok(descriptor) => descriptor,
		Err(error) => return FFITagObserveResult { error, tag_observe_pointer: std::ptr::null_mut() },
	};
	let local_node = unsafe { Box::from_raw(local_node) };
	let tag_observe_pointer = Box::into_raw(Box::new(local_node.tag(descriptor).with_deserializer(raw_deserialize).observe()));
	forget(local_node);
	FFITagObserveResult { error: FFIError::None, tag_observe_pointer }
}

#[no_mangle]
pub extern "C" fn hakuban_tag_observe_drop(tag_ptr: *mut TagObserve<ObjectRawData, ()>) {
	drop(unsafe { Box::from_raw(tag_ptr) });
}

#[repr(C)]
pub struct FFITagObserveObjectStateBorrowResult {
	pub error: FFIError,
	pub state: FFIObjectObserveState,
}

#[no_mangle]
pub extern "C" fn hakuban_tag_observe_object_state_borrow(
	tag_observe_ptr: *mut TagObserve<ObjectRawData, ()>, descriptor: FFIObjectDescriptor,
) -> FFITagObserveObjectStateBorrowResult {
	let descriptor: ObjectDescriptor = match ObjectDescriptor::try_from(&descriptor) {
		Ok(descriptor) => descriptor,
		Err(error) => return FFITagObserveObjectStateBorrowResult { error, state: FFIObjectObserveState::empty() },
	};
	let tag_observe = unsafe { Box::from_raw(tag_observe_ptr) };
	let ret = match tag_observe.object_state(&descriptor).unwrap() {
		Some(state) => FFITagObserveObjectStateBorrowResult { error: FFIError::None, state: FFIObjectObserveState::from_state(state) },
		None => FFITagObserveObjectStateBorrowResult { error: FFIError::ObjectNotFound, state: FFIObjectObserveState::empty() },
	};
	forget(tag_observe);
	ret
}

#[repr(C)]
pub struct FFIObjectDescriptors {
	pub count: usize,
	pub descriptors: *mut FFIObjectDescriptor,
}

impl FFIObjectDescriptors {
	pub(super) fn new(mut descriptors: Vec<FFIObjectDescriptor>) -> FFIObjectDescriptors {
		descriptors.shrink_to_fit();
		let ret = FFIObjectDescriptors { count: descriptors.len(), descriptors: descriptors.as_mut_ptr() };
		forget(descriptors);
		ret
	}

	pub(super) fn free(self) {
		let mut descriptors: Vec<FFIObjectDescriptor> = unsafe { Vec::from_raw_parts(self.descriptors, self.count, self.count) };
		for descriptor in descriptors.drain(..) {
			descriptor.free();
		}
	}
}

#[no_mangle]
pub extern "C" fn hakuban_tag_observe_object_descriptors_borrow(tag_observe_ptr: *mut TagObserve<ObjectRawData, ()>) -> FFIObjectDescriptors {
	let tag_observe = unsafe { Box::from_raw(tag_observe_ptr) };
	let descriptors: Vec<FFIObjectDescriptor> = tag_observe.object_descriptors().iter().map(|descriptor| descriptor.into()).collect();
	forget(tag_observe);
	FFIObjectDescriptors::new(descriptors)
}

#[no_mangle]
pub extern "C" fn hakuban_object_descriptors_return(descriptors: FFIObjectDescriptors) {
	descriptors.free();
}

#[no_mangle]
pub extern "C" fn hakuban_tag_observe_objects_changes_callback_register(
	tag_observe_ptr: *mut TagObserve<ObjectRawData, ()>, callback: ObjectEventCallbackFn, userdata: *const c_void,
) -> *mut Box<dyn Callback<Event<ObjectDescriptor>>> {
	let tag_observe = unsafe { Box::from_raw(tag_observe_ptr) };
	let ret = Box::new(tag_observe.changes().register(make_object_descriptor_callback(callback, userdata)));
	forget(tag_observe);
	Box::into_raw(ret)
}
